﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/xcd/xcd_AttachmentDevice.h>
#include <nn/xcd/xcd_BleDevice.h>
#include <nn/xcd/xcd_DeviceState.h>
#include <nn/xcd/xcd_DataFormat.h>
#include "xcd_IEventTask.h"
#include "xcd_MultiWaitEvent.h"
#include "xcd_BleCommandHandler.h"
#include "xcd_BlePadInput.h"
#include "xcd_IBleCommandListener.h"
#include "xcd_TransactionMonitor.h"
#include "detail/xcd_BleHidAccessor.h"

namespace nn { namespace xcd {

enum BleDeviceSequence
{
    BleDeviceSequence_Nolink,               //!< リンクアップしていない状態
    BleDeviceSequence_Linkup,               //!< リンクアップしている状態 (通信は開始していない)
    BleDeviceSequence_GetGattServerInfo,    //!< Gatt サーバの情報の取得中
    BleDeviceSequence_UpdateMtuSize,        //!< MTU サイズの変更中
    BleDeviceSequence_ReadDeviceInfo,       //!< デバイスの情報の取得中
    BleDeviceSequence_ReadBatteryStatus,    //!< 電池情報の取得中
    BleDeviceSequence_SuspendFeature,       //!< 機能の一時停止操作中

    BleDeviceSequence_Active,               //!< 初期化処理が完了し、接続済みの状態
    BleDeviceSequence_Detach,               //!< 切断処理中
};

struct BleDeviceHandlerFlag
{
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<0> NotifyOnConnectionComplete;   //!< 接続完了時に通知を行うためのフラグ
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<1> IsActivated;                  //!< アクティベートされているかどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<2> IsErrorOccurred;              //!< 何かしらのエラーが起こった状態
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<6> ReadInputReportFormatIdComplete;     //!< NHOG FormatId の取得完了のフラグ
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<7> ReadOutputCommandFormatIdComplete;   //!< NHOG FormatId の取得完了のフラグ
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<8> ReadCommandResponseFormatIdComplete; //!< NHOG FormatId の取得完了のフラグ
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<9> IsSamplingStarted;            //!< HID レポートのサンプリングが始まっているかどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<10> UpdateMtuSizeComplete;       //!< MTU サイズ変更完了のフラグ
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<11> IsProductTypeReading;        //!< ProductType の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<12> IsModelInformationReading;   //!< ModelInformation の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<13> IsSensorCalReading;          //!< SensorCal の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<14> IsAnalogStickCalReading;     //!< AnalogStickCal の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<15> IsBatteryStatusReadComplete;   //!< BatteryInfo の読み出し中かどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag>::Flag<16> IsSuspendFeatureComplete;      //!< Suspend 実行中かどうか
};

struct BleDeviceCalibrationFlag
{
    typedef ::nn::util::BitFlagSet<32, BleDeviceCalibrationFlag>::Flag<0> IsAnalogStickCalibrated;   //!< アナログスティックのキャリブレーション済みかどうか
    typedef ::nn::util::BitFlagSet<32, BleDeviceCalibrationFlag>::Flag<1> IsSensorCalibrated;        //!< センサーのキャリブレーション済みかどうか
};

/**
 * @brief       DeviceHandler で扱うフラグの集合です
 */
typedef ::nn::util::BitFlagSet<32, BleDeviceHandlerFlag> BleDeviceHandlerFlagSet;
typedef ::nn::util::BitFlagSet<32, BleDeviceCalibrationFlag> BleDeviceCalibrationFlagSet;

//!< Nx Ble デバイスを管理/制御するためのクラスです。
class BleDeviceHandler final : public IEventTask, public IBleCommandListener
{

private:
    //!< BleDeviceHandler で使用するフラグの集合
    BleDeviceHandlerFlagSet m_Flags;

    //!< デバイスハンドル
    BleConnectionHandle m_Handle;

    //!< Ble 制御を行うためのアクセサ
    detail::BleHidAccessor* m_pAccessor;

    //!< BleCommand 制御を行うためのハンドラ
    BleCommandHandler m_Command;

    //!< Pad (ボタン, スティック, センサー) 制御
    BlePadInput m_PadInput;

    //!< パケット受信履歴管理
    TransactionMonitor m_TransactionMonitor;

    //!< 送受信に使用するバッファのサイズ
    static const size_t HidBufferSize = 512;

    //!< 接続状態に更新を通知するイベントオブジェクト
    nn::os::SystemEventType* m_pStatusUpdateEvent;

    //!< 新しいサンプルを受信した際に通知するイベントオブジェクト
    nn::os::SystemEventType* m_pSamplingEvent;

    //!< デバイスの状態
    BleDeviceSequence m_Sequence;

    //!< デバイスの情報
    BleDeviceInfo m_Info;

    //!< デバイスの状態
    DeviceStatus m_Status;

    //!< Firmware のバージョン

    //!< Ble Report のフォーマットバージョン
    BleReportFormatVersion m_ReportFormatVersion;

    //!< プロダクト情報
    BleDeviceProductType m_ProductType;

    //!< モデル情報
    BleDeviceModelInformation m_ModelInfo;

    //!< Sensor Cal 情報
    SensorCalibrationValue m_SensorCalibration;

    //!< AnalogStick Cal 情報
    AnalogStickValidRange m_AnalogStickCalibration;

    //!< キャリブレーション済みかどうか
    BleDeviceCalibrationFlagSet m_CalibrationFlags;

public:
    BleDeviceHandler() NN_NOEXCEPT;
    virtual ~BleDeviceHandler() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< BleDeviceHandlerを有効にする
    void Activate(BleConnectionHandle handle, detail::BleHidAccessor* pAccessor) NN_NOEXCEPT;

    //!< DeviceHandlerを無効にする
    void Deactivate() NN_NOEXCEPT;

    //!< アクティベートされているかどうか
    bool IsActivated() NN_NOEXCEPT
    {
        return m_Flags.Test<BleDeviceHandlerFlag::IsActivated>();
    };

    //!< サンプリングが始まっているかどうか
    bool IsSamplingStarted() NN_NOEXCEPT
    {
        return m_Flags.Test<BleDeviceHandlerFlag::IsSamplingStarted>();
    };

    //!< DeviceHandleを取得する
    BleConnectionHandle GetDeviceHandle() NN_NOEXCEPT
    {
        return m_Handle;
    };

    //!< デバイスの情報を取得する
    void GetDeviceInfo(BleDeviceInfo* pInfo) NN_NOEXCEPT
    {
        *pInfo = m_Info;
    };

    //!< デバイスの状態を取得する
    void GetDeviceStatus(DeviceStatus* pStatus) NN_NOEXCEPT
    {
        *pStatus = m_Status;
    };

    //!< Firmware のバージョンを取得する

    //!< 切断する
    void Detach() NN_NOEXCEPT;

    //!< 接続状態に更新があった場合に通知するイベントオブジェクトを登録する
    void SetStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< 接続状態に更新があった場合に通知するシステムイベントを解除します
    void ClearStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< サンプリングイベントをセットする
    Result SetSamplingEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< 各下位モジュールへのインタフェース
    BlePadInput* GetBlePadInput() NN_NOEXCEPT
    {
        return &m_PadInput;
    }

    BleCommandHandler* GetBleCommand() NN_NOEXCEPT
    {
        return &m_Command;
    }

    virtual void NotifyConfigureMtuCompleted(uint16_t mtu) NN_NOEXCEPT NN_OVERRIDE;

    virtual void CharacteristicReadComplete(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT NN_OVERRIDE;

    virtual void DescriptorReadComplete(const nn::bluetooth::GattAttributeUuid& characteristicUuid, const nn::bluetooth::GattAttributeUuid& descriptorUuid) NN_NOEXCEPT NN_OVERRIDE;

    virtual void DescriptorWriteComplete(const nn::bluetooth::GattAttributeUuid& characteristicUuid, const nn::bluetooth::GattAttributeUuid& descriptorUuid) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyGattOperationError(uint32_t status) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyCommandCompleted(BleDeviceOperationType operationType, BleDeviceOperationResultType resultType) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyReadProductType(const BleDeviceProductType& type) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyReadModelInformation(const BleDeviceModelInformation& info) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyReadSensorCalibration(const SensorCalibrationValue& value) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyReadAnalogStickCalibration(const AnalogStickValidRange& value) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyRespSuspendFeatureSet(BleDeviceOperationResultType resultType, uint32_t featureFlagSet) NN_NOEXCEPT NN_OVERRIDE;

    //!< パケットの受信履歴を取得します
    nn::hid::detail::RxPacketHistory GetRxPacketHistory() NN_NOEXCEPT;

    //!< センサーのキャリブレーション値を取得する
    void GetSensorCalibrationValue(SensorCalibrationValue* pOutValue) NN_NOEXCEPT;

    //!< センサーのモデル値を取得する
    SensorState GetSensorHorizontalOffset() NN_NOEXCEPT;

    //!< アナログスティックの有効範囲を取得する
    void GetAnalogStickValidRange(AnalogStickValidRange* pOutValue) NN_NOEXCEPT;

    //!< アナログスティックのモデル値を取得する
    AnalogStickDeviceParameter GetAnalogStickDeviceParameter() NN_NOEXCEPT;

private:
    //!< InputReport を受信した際に呼ばれるハンドラ
    static void InputReportHandler(void* pParser, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< 各ペリフェラルデバイスの初期化処理
    void InitializeDevice() NN_NOEXCEPT;

    //!< InputReport を パースする
    void ParseInputReport(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< シーケンスの更新処理を行う
    void UpdateSequence() NN_NOEXCEPT;

    //!< 次のシーケンスの処理をおこなう
    void HandleNextSequence() NN_NOEXCEPT;

    //!< Gatt サーバの情報の読み出しが完了したかどうか
    bool IsReadGattServerInfoCompleted() NN_NOEXCEPT;

    //!< BLE デバイス種別の判定が完了したかどうか
    bool IsDeviceTypeDetectionCompleted() NN_NOEXCEPT;

    //!< Gatt サーバから情報を取得する
    void GetGattServerInfo() NN_NOEXCEPT;

    //!< BLE デバイス情報を取得する
    void ReadDeviceInfo() NN_NOEXCEPT;

    //!< 電池の状態を読み出す
    void GetBatteryStatus() NN_NOEXCEPT;

    //!< 規定のサイズに MTU を設定する
    void UpdateMtuSize() NN_NOEXCEPT;

    const char* GetName(BleDeviceSequence sequence) NN_NOEXCEPT;

    void PrintDeviceInformation() NN_NOEXCEPT;
};

}} // namespace nn::xcd
