﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_BleDeviceHandler.h"
#include "detail/xcd_BleGattUuids.h"
#include "detail/xcd_LinkMonitor.h"

namespace nn { namespace xcd {

namespace {

struct Range
{
    int16_t min;
    int16_t max;
};

const int16_t StickMax = 4096;         //!< アナログスティックの最大値
const uint16_t TargetMtu = 512;        //!< 設定する MTU サイズ

const AnalogStickValidRange AnalogStickDefaultValues = {
    { 0, 0 },      // スティックの原点値
    { 730, 730 },  // スティックの外周最小値
    { 730, 730 },  // スティックの外周最大値
    321,  // 原点の遊び領域の大きさ
};

const SensorCalibrationValue DefaultSensorCalibration = {
    { 0, 0, 0 },  // 加速度センサーの原点値
    { 16384, 16384, 16384 },  // 加速度センサーの感度
    { 0, 0, 0 },  // ジャイロセンサーの原点値
    { 13371, 13371, 13371 },  // ジャイロセンサーの感度
};

// モデル値カウントを FSR = 8G から FSR = 2G の単位系に変換
const SensorState SensorHorizontalOffsetTypical = { 0, 0, 4096 * 4 };

const Range AccelerometerOriginRange         = {-3276,  3276};
const Range AccelerometerSensitivityRange    = {14745, 18022};
const Range GyroscopeOriginRange             = { -819,   819};
const Range GyroscopeSensitivityRange        = {12034, 16868};

//!< ProductType が Palma かどうか判定します
bool ISPalmaProductType(const BleDeviceProductType& type)
{
    if (type.main == 0x01)
    {
        return true;
    }
    return false;
}

//!< 範囲内かどうか検証します
bool ValidateRange(int16_t value, int16_t min, int16_t max)
{
    return (value >= min && value <= max);
};

bool ValidateRange(int32_t value, int16_t min, int16_t max)
{
    return (value >= min && value <= max);
};

bool ValidateRange(int32_t value, Range range)
{
    return ValidateRange(value, range.min, range.max);
};

//!< アナログスティックの補正値を検証します
bool ValidateAnalogStickValidRange(const AnalogStickValidRange& value, const BleDeviceModelInformation& modelInfo)
{
    return (ValidateRange(value.origin.x,     modelInfo.analogStickOriginRangeMin.x, modelInfo.analogStickOriginRangeMax.x) &&
            ValidateRange(value.origin.y,     modelInfo.analogStickOriginRangeMin.y, modelInfo.analogStickOriginRangeMax.y) &&
            ValidateRange(value.circuitMin.x, modelInfo.analogStickMinimumStrokeNegative.x, StickMax - value.origin.x) &&
            ValidateRange(value.circuitMin.y, modelInfo.analogStickMinimumStrokeNegative.y, StickMax - value.origin.y) &&
            ValidateRange(value.circuitMax.x, modelInfo.analogStickMinimumStrokePositive.x, StickMax - value.origin.x) &&
            ValidateRange(value.circuitMax.y, modelInfo.analogStickMinimumStrokePositive.y, StickMax - value.origin.y));
}

//!< センサーの補正値を検証します
bool ValidateCalibrationValue(const SensorCalibrationValue& value)
{
    bool validate =
        (ValidateRange(value.accelerometerOrigin.x, AccelerometerOriginRange) &&
            ValidateRange(value.accelerometerOrigin.y, AccelerometerOriginRange) &&
            ValidateRange(value.accelerometerOrigin.z, AccelerometerOriginRange) &&
            ValidateRange(value.accelerometerSensitivity.x, AccelerometerSensitivityRange) &&
            ValidateRange(value.accelerometerSensitivity.y, AccelerometerSensitivityRange) &&
            ValidateRange(value.accelerometerSensitivity.z, AccelerometerSensitivityRange) &&
            ValidateRange(value.gyroscopeOrigin.x, GyroscopeOriginRange) &&
            ValidateRange(value.gyroscopeOrigin.y, GyroscopeOriginRange) &&
            ValidateRange(value.gyroscopeOrigin.z, GyroscopeOriginRange) &&
            ValidateRange(value.gyroscopeSensitivity.x, GyroscopeSensitivityRange) &&
            ValidateRange(value.gyroscopeSensitivity.y, GyroscopeSensitivityRange) &&
            ValidateRange(value.gyroscopeSensitivity.z, GyroscopeSensitivityRange)
            );

    return validate;
};

//!< 外周値に不感帯を付加します
void AddPlayOnStickCircuit(AnalogStickValidRange* pValue, bool isScalingRequired, int16_t circuitValidRatio)
{
    int xScale = (isScalingRequired) ? 90 : 100;
    pValue->circuitMin.x = static_cast<int16_t>(
                                (static_cast<int64_t>(pValue->circuitMin.x) * circuitValidRatio * xScale) / StickMax / 100
                            );
    pValue->circuitMin.y = static_cast<int16_t>(
                                (static_cast<int64_t>(pValue->circuitMin.y) * circuitValidRatio) / StickMax
                            );
    pValue->circuitMax.x = static_cast<int16_t>(
                                (static_cast<int64_t>(pValue->circuitMax.x) * circuitValidRatio * xScale) / StickMax / 100
                            );
    pValue->circuitMax.y = static_cast<int16_t>(
                                (static_cast<int64_t>(pValue->circuitMax.y) * circuitValidRatio) / StickMax
                            );
}

void DumpStickValue(const AnalogStickState& state)
{
    NN_UNUSED(state);
    NN_SDK_LOG("x:%5d y:%5d\n", state.x, state.y);
}

void DumpSensorValue(const SensorState& state)
{
    NN_UNUSED(state);
    NN_SDK_LOG("x:%5d y:%5d z:%5d\n", state.x, state.y, state.z);
}

}


BleDeviceHandler::BleDeviceHandler() NN_NOEXCEPT
    : m_Handle(BleInvalidConnectionHandle)
    , m_Sequence(BleDeviceSequence_Nolink)
{
    m_Flags.Reset();
    m_CalibrationFlags.Reset();
}

BleDeviceHandler::~BleDeviceHandler() NN_NOEXCEPT
{
    // 何もしない
}

void BleDeviceHandler::Activate(BleConnectionHandle handle, detail::BleHidAccessor* pAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Flags.Test<BleDeviceHandlerFlag::IsActivated>(), false);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    m_Handle = handle;
    m_pAccessor = pAccessor;

    m_Sequence = BleDeviceSequence_Linkup;

    m_Flags.Reset();
    m_Flags.Set<BleDeviceHandlerFlag::IsActivated>();
    m_Flags.Set<BleDeviceHandlerFlag::NotifyOnConnectionComplete>();

    m_Info.deviceType = BleDeviceType_Unknown;
    m_pAccessor->GetDeviceAddress(&m_Info.address);
    m_pAccessor->SetListener(this);

    m_Status.interfaceType = InterfaceType_Bluetooth;
    m_Status.batteryLevel = BatteryLevel_None;
    m_Status.powered = false;
    m_Status.charged = false;
    m_Status.cablePlugged = false;
    m_Status.connector = Connector_Detached;
    m_Status.isAttachmentReady = false;

    m_CalibrationFlags.Reset();

    GetTaskManager().RegisterPeriodicTask(this);
}

void BleDeviceHandler::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Flags.Test<BleDeviceHandlerFlag::IsActivated>(), true);

    GetTaskManager().UnregisterPeriodicTask(this);

    m_Sequence = BleDeviceSequence_Nolink;

    if (m_Command.IsActivated())
    {
        m_Command.Deactivate(m_pAccessor);
    }
    if (m_PadInput.IsActivated())
    {
        m_PadInput.Deactivate();
    }

    // ハンドルを初期化
    m_Handle = BleInvalidConnectionHandle;

    m_pSamplingEvent = nullptr;

    m_Flags.Reset();

    if (m_pStatusUpdateEvent != nullptr)
    {
        nn::os::SignalSystemEvent(m_pStatusUpdateEvent);
    }
}

void BleDeviceHandler::SetStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pStatusUpdateEvent = pEvent;
}

void BleDeviceHandler::ClearStatusUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    if (m_pStatusUpdateEvent == pEvent)
    {
        m_pStatusUpdateEvent = nullptr;
    }
}

Result BleDeviceHandler::SetSamplingEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pSamplingEvent = pEvent;

    NN_RESULT_SUCCESS;
}

void BleDeviceHandler::Detach() NN_NOEXCEPT
{
    m_pAccessor->DetachDevice();
}

void BleDeviceHandler::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void BleDeviceHandler::PeriodicEventFunction() NN_NOEXCEPT
{
    m_Command.PeriodicOperation();
    GetBatteryStatus();
    UpdateSequence();
}

void BleDeviceHandler::ParseInputReport(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    m_PadInput.ParseInputReport(buffer, size);
    if (m_TransactionMonitor.IsActivated())
    {
        int sampleSinceLast;
        ::nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(5);
        m_TransactionMonitor.UpdateRxMonitoringResult(&sampleSinceLast, buffer[BleInputReportFormat_Palma::SampleNumber], interval);
    }
    // InputReport を受信し始めたら IsSamplingStarted フラグを立てる
    if (m_Flags.Test<BleDeviceHandlerFlag::IsSamplingStarted>() == false)
    {
        m_Flags.Set<BleDeviceHandlerFlag::IsSamplingStarted>();
    }
}

void BleDeviceHandler::NotifyConfigureMtuCompleted(uint16_t mtu) NN_NOEXCEPT
{
    // xcd がサポートしないデバイスは無視
    if (m_pAccessor->GetBleNhogClient() == nullptr ||
        m_pAccessor->GetBleNbatClient() == nullptr)
    {
        return;
    }

    if (mtu >= TargetMtu)
    {
        m_Flags.Set<BleDeviceHandlerFlag::UpdateMtuSizeComplete>();
    }
    else
    {
        // xcd がサポートするデバイスは 512 より小さな MTU は想定しない
        m_Flags.Set<BleDeviceHandlerFlag::IsErrorOccurred>();
    }

    UpdateSequence();
}

void BleDeviceHandler::CharacteristicReadComplete(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    NN_UNUSED(uuid);
    UpdateSequence();
}

void BleDeviceHandler::DescriptorReadComplete(const nn::bluetooth::GattAttributeUuid& characteristicUuid, const nn::bluetooth::GattAttributeUuid& descriptorUuid) NN_NOEXCEPT
{
    if (characteristicUuid == detail::InputReportCharacteristic.Uuid)
    {
        if (descriptorUuid == detail::ReportReferenceDescriptor.Uuid)
        {
            m_Flags.Set<BleDeviceHandlerFlag::ReadInputReportFormatIdComplete>();
        }
    }
    else if (characteristicUuid == detail::OutputCommandCharacteristic.Uuid)
    {
        if (descriptorUuid == detail::ReportReferenceDescriptor.Uuid)
        {
            m_Flags.Set<BleDeviceHandlerFlag::ReadOutputCommandFormatIdComplete>();
        }
    }
    else if (characteristicUuid == detail::CommandResponseCharacteristic.Uuid)
    {
        if (descriptorUuid == detail::ReportReferenceDescriptor.Uuid)
        {
            m_Flags.Set<BleDeviceHandlerFlag::ReadCommandResponseFormatIdComplete>();
        }
    }
    UpdateSequence();
}

void BleDeviceHandler::DescriptorWriteComplete(const nn::bluetooth::GattAttributeUuid& characteristicUuid, const nn::bluetooth::GattAttributeUuid& descriptorUuid) NN_NOEXCEPT
{
    NN_UNUSED(characteristicUuid);
    NN_UNUSED(descriptorUuid);
    // 何もしない
}

void BleDeviceHandler::NotifyCommandCompleted(BleDeviceOperationType operationType, BleDeviceOperationResultType resultType) NN_NOEXCEPT
{
    NN_UNUSED(operationType);
    NN_UNUSED(resultType);
    // 何もしない
}

void BleDeviceHandler::NotifyGattOperationError(uint32_t status) NN_NOEXCEPT
{
    NN_UNUSED(status);
    // エラーが発生したフラグをセット
    m_Flags.Set<BleDeviceHandlerFlag::IsErrorOccurred>();
    UpdateSequence();
}

void BleDeviceHandler::NotifyReadProductType(const BleDeviceProductType& type) NN_NOEXCEPT
{
    m_ProductType = type;

    // モデル情報を読み出す
    m_Flags.Set<BleDeviceHandlerFlag::IsModelInformationReading>();
    m_Command.GetModelInformation();
    m_Flags.Reset<BleDeviceHandlerFlag::IsProductTypeReading>();
}

void BleDeviceHandler::NotifyReadModelInformation(const BleDeviceModelInformation& info) NN_NOEXCEPT
{
    m_ModelInfo = info;

    // モデル値カウントを FSR = 8G から FSR = 2G の単位系に変換
    m_ModelInfo.sensorHorizontalOffset.x *= 4;
    m_ModelInfo.sensorHorizontalOffset.y *= 4;
    m_ModelInfo.sensorHorizontalOffset.z *= 4;

    // Sensor Cal 情報を読み出す
    m_Flags.Set<BleDeviceHandlerFlag::IsSensorCalReading>();
    m_Command.GetSensorCalibrationValue();
    m_Flags.Reset<BleDeviceHandlerFlag::IsModelInformationReading>();
}

void BleDeviceHandler::NotifyReadSensorCalibration(const SensorCalibrationValue& value) NN_NOEXCEPT
{
    if (ValidateCalibrationValue(value))
    {
        m_SensorCalibration = value;
        m_CalibrationFlags.Set<BleDeviceCalibrationFlag::IsSensorCalibrated>();
    }

    // AnalogStick Cal 情報を読み出す
    m_Flags.Set<BleDeviceHandlerFlag::IsAnalogStickCalReading>();
    m_Command.GetAnalogStickCalibrationValue();
    m_Flags.Reset<BleDeviceHandlerFlag::IsSensorCalReading>();
}

void BleDeviceHandler::NotifyReadAnalogStickCalibration(const AnalogStickValidRange& value) NN_NOEXCEPT
{
    m_AnalogStickCalibration = value;
    if (ValidateAnalogStickValidRange(value, m_ModelInfo))
    {
        m_AnalogStickCalibration = value;
        m_CalibrationFlags.Set<BleDeviceCalibrationFlag::IsAnalogStickCalibrated>();
    }

    m_Flags.Reset<BleDeviceHandlerFlag::IsAnalogStickCalReading>();
    UpdateSequence();
}

void BleDeviceHandler::NotifyRespSuspendFeatureSet(BleDeviceOperationResultType resultType, uint32_t featureFlagSet) NN_NOEXCEPT
{
    NN_UNUSED(featureFlagSet);
    if (m_Sequence == BleDeviceSequence_SuspendFeature && resultType == BleDeviceOperationResultType_Success)
    {
        m_Flags.Set<BleDeviceHandlerFlag::IsSuspendFeatureComplete>();
    }
    else
    {
        m_Flags.Set<BleDeviceHandlerFlag::IsErrorOccurred>();
    }
    UpdateSequence();
}

nn::hid::detail::RxPacketHistory BleDeviceHandler::GetRxPacketHistory() NN_NOEXCEPT
{
    return m_TransactionMonitor.GetRxPacketHistory();
}

void BleDeviceHandler::GetSensorCalibrationValue(SensorCalibrationValue* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (m_CalibrationFlags.Test<BleDeviceCalibrationFlag::IsSensorCalibrated>())
    {
        *pOutValue = m_SensorCalibration;
    }
    else
    {
        *pOutValue = DefaultSensorCalibration;
        NN_DETAIL_XCD_WARN("SixAxisSensor CAL invalid\n");
    }
}

SensorState BleDeviceHandler::GetSensorHorizontalOffset() NN_NOEXCEPT
{
    if (m_CalibrationFlags.Test<BleDeviceCalibrationFlag::IsSensorCalibrated>())
    {
        return m_ModelInfo.sensorHorizontalOffset;
    }
    else
    {
        return SensorHorizontalOffsetTypical;
    }
}

void BleDeviceHandler::GetAnalogStickValidRange(AnalogStickValidRange* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (m_CalibrationFlags.Test<BleDeviceCalibrationFlag::IsAnalogStickCalibrated>())
    {
        pOutValue->origin = m_AnalogStickCalibration.origin;
        pOutValue->circuitMin = m_AnalogStickCalibration.circuitMin;
        pOutValue->circuitMax = m_AnalogStickCalibration.circuitMax;
        pOutValue->originPlay = m_ModelInfo.analogStickOriginPlay;
        AddPlayOnStickCircuit(pOutValue, m_ModelInfo.isScalingRequired, m_ModelInfo.analogStickCircuitValidRatio);
    }
    else
    {
        *pOutValue = AnalogStickDefaultValues;
        NN_DETAIL_XCD_WARN("AnalogStick CAL invalid\n");
    }
}

AnalogStickDeviceParameter BleDeviceHandler::GetAnalogStickDeviceParameter() NN_NOEXCEPT
{
    AnalogStickDeviceParameter value;
    value.minimumStrokePositive = m_ModelInfo.analogStickMinimumStrokePositive;
    value.minimumStrokeNegative = m_ModelInfo.analogStickMinimumStrokeNegative;
    value.originRangeMin = m_ModelInfo.analogStickOriginRangeMin;
    value.originRangeMax = m_ModelInfo.analogStickOriginRangeMax;
    return value;
}

void BleDeviceHandler::InputReportHandler(void* pParser, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    // 受信した Report の Parse
    reinterpret_cast<BleDeviceHandler*>(pParser)->ParseInputReport(buffer, size);
}

void BleDeviceHandler::InitializeDevice() NN_NOEXCEPT
{
    // CommandHandler を起動する
    m_Command.Activate(m_pAccessor, m_ReportFormatVersion.outputCommandFormatVer, m_ReportFormatVersion.commandResponseFormatVer);
    m_Command.SetListener(this);
    m_PadInput.Activate(m_ReportFormatVersion);
    m_TransactionMonitor.ActivateBleDevice();
}

void BleDeviceHandler::UpdateSequence() NN_NOEXCEPT
{
    auto nextSequence = m_Sequence;

    if (m_Flags.Test<BleDeviceHandlerFlag::IsErrorOccurred>())
    {
        // 何か想定されないエラーが起きた場合は切断シーケンスに入る
        m_Sequence = BleDeviceSequence_Detach;
        Detach();
        m_Flags.Reset<BleDeviceHandlerFlag::IsErrorOccurred>();
        return;
    }

    switch (m_Sequence)
    {
    case BleDeviceSequence_Linkup:
        // Gatt サーバの情報を読み出す
        nextSequence = BleDeviceSequence_GetGattServerInfo;
        break;
    case BleDeviceSequence_GetGattServerInfo:
        if (IsReadGattServerInfoCompleted())
        {
            // MTU サイズを更新する
            nextSequence = BleDeviceSequence_UpdateMtuSize;
        }
        break;
    case BleDeviceSequence_UpdateMtuSize:
        if (m_Flags.Test<BleDeviceHandlerFlag::UpdateMtuSizeComplete>())
        {
            // デバイスの情報を読み出す
            nextSequence = BleDeviceSequence_ReadDeviceInfo;
        }
        break;
    case BleDeviceSequence_ReadDeviceInfo:
        // DeviceType の判定完了を待つ
        if (IsDeviceTypeDetectionCompleted())
        {
            nextSequence = BleDeviceSequence_ReadBatteryStatus;
        }
        break;
    case BleDeviceSequence_ReadBatteryStatus:
        // 電池状態の取得完了を待つ
        if (m_Flags.Test<BleDeviceHandlerFlag::IsBatteryStatusReadComplete>())
        {
            nextSequence = BleDeviceSequence_SuspendFeature;
        }
        break;
    case BleDeviceSequence_SuspendFeature:
        // 機能の一時停止完了を待つ
        if (m_Flags.Test<BleDeviceHandlerFlag::IsSuspendFeatureComplete>())
        {
            nextSequence = BleDeviceSequence_Active;
        }
        break;
    case BleDeviceSequence_Active:
    case BleDeviceSequence_Detach:
    case BleDeviceSequence_Nolink:
        // 何もしない
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (nextSequence != m_Sequence)
    {
        m_Sequence = nextSequence;
        HandleNextSequence();
    }
}

void BleDeviceHandler::HandleNextSequence() NN_NOEXCEPT
{
    switch (m_Sequence)
    {
    case BleDeviceSequence_Linkup:
    case BleDeviceSequence_Nolink:
        // 何もしない
        break;
    case BleDeviceSequence_GetGattServerInfo:
        GetGattServerInfo();
        break;
    case BleDeviceSequence_UpdateMtuSize:
        UpdateMtuSize();
        break;
    case BleDeviceSequence_ReadDeviceInfo:
        InitializeDevice();
        // プロダクト情報を読み出す
        m_Flags.Set<BleDeviceHandlerFlag::IsProductTypeReading>();
        m_Command.GetProductType();
        break;

    case BleDeviceSequence_ReadBatteryStatus:
    {
        // NbatClient 内で最初の Read をトリガーすると Busy で死ぬので一旦 DeviceHandler から最初の Read をトリガーする
        auto pNbat = m_pAccessor->GetBleNbatClient();
        if (pNbat != nullptr)
        {
            pNbat->UpdateBatteryStatus();
        }
        break;
    }
    case BleDeviceSequence_SuspendFeature:
    {
        m_Command.SuspendFeatureSystem(0xF);
        break;
    }
    case BleDeviceSequence_Active:
        if (m_Flags.Test<BleDeviceHandlerFlag::NotifyOnConnectionComplete>())
        {
            if (m_pStatusUpdateEvent != nullptr)
            {
                nn::os::SignalSystemEvent(m_pStatusUpdateEvent);
            }
            m_Flags.Reset<BleDeviceHandlerFlag::NotifyOnConnectionComplete>();
        }
        m_pAccessor->StartInputReport(InputReportHandler, this);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool BleDeviceHandler::IsReadGattServerInfoCompleted() NN_NOEXCEPT
{
    if (m_Flags.Test<BleDeviceHandlerFlag::ReadInputReportFormatIdComplete>()
        && m_Flags.Test<BleDeviceHandlerFlag::ReadOutputCommandFormatIdComplete>()
        && m_Flags.Test<BleDeviceHandlerFlag::ReadCommandResponseFormatIdComplete>())
    {
        auto nhog = m_pAccessor->GetBleNhogClient();
        if (nhog != nullptr)
        {
            m_ReportFormatVersion.inputReportFormatVer = nhog->GetInputReportFormatVersion();
            m_ReportFormatVersion.outputCommandFormatVer = nhog->GetOutputCommandFormatVersion();
            m_ReportFormatVersion.commandResponseFormatVer = nhog->GetCommandResponseFormatVersion();

            return true;
        }
    }
    return false;
}

bool BleDeviceHandler::IsDeviceTypeDetectionCompleted() NN_NOEXCEPT
{
    // 読み出しが終わっていない場合は何もしない
    if (m_Flags.Test<BleDeviceHandlerFlag::IsProductTypeReading>()
        || m_Flags.Test<BleDeviceHandlerFlag::IsModelInformationReading>()
        || m_Flags.Test<BleDeviceHandlerFlag::IsSensorCalReading>()
        || m_Flags.Test<BleDeviceHandlerFlag::IsAnalogStickCalReading>())
    {
        return false;
    }

    auto nhog = m_pAccessor->GetBleNhogClient();
    if (nhog != nullptr)
    {
        NN_DETAIL_XCD_INFO("BleDeviceHandler: InputReport FormatId=0x%02X\n", nhog->GetInputReportFormatVersion());
        NN_DETAIL_XCD_INFO("BleDeviceHandler: OutputCommand FormatId=0x%02X\n", nhog->GetOutputCommandFormatVersion());
        NN_DETAIL_XCD_INFO("BleDeviceHandler: CommandResponse FormatId=0x%02X\n", nhog->GetCommandResponseFormatVersion());

        if (ISPalmaProductType(m_ProductType))
        {
            m_Info.deviceType = BleDeviceType_Palma;

            PrintDeviceInformation(); // TODO: あとで消す
        }
        else
        {
            NN_DETAIL_XCD_WARN("Unsupported BLE device.\n");
            m_Flags.Set<BleDeviceHandlerFlag::IsErrorOccurred>();
            return false;
        }

        return true;
    }
    return false;
}

void BleDeviceHandler::GetGattServerInfo() NN_NOEXCEPT
{
    // Nintendo Hid over Gatt サービスから各レポートの FormatId を取得する
    auto nhog = m_pAccessor->GetBleNhogClient();
    if (nhog != nullptr)
    {
        nhog->ReadInputReportReportReferenceAsync(this);
        nhog->ReadOutputCommandReportReferenceAsync(this);
        nhog->ReadCommandResponseReportReferenceAsync(this);
    }
}

void BleDeviceHandler::GetBatteryStatus() NN_NOEXCEPT
{
    auto pNbat = m_pAccessor->GetBleNbatClient();
    if (pNbat != nullptr)
    {
        if (pNbat->GetBatteryStatus(&m_Status.batteryLevel, &m_Status.powered, &m_Status.charged) == true)
        {
            m_Flags.Set<BleDeviceHandlerFlag::IsBatteryStatusReadComplete>();
            m_Status.cablePlugged = m_Status.powered;
        }
    }
}

void BleDeviceHandler::UpdateMtuSize() NN_NOEXCEPT
{
    m_pAccessor->ConfigureBleMtu(TargetMtu);
}

const char* BleDeviceHandler::GetName(BleDeviceSequence sequence) NN_NOEXCEPT
{
    switch (sequence)
    {
    case BleDeviceSequence_Linkup:
        return "Linkup";
    case BleDeviceSequence_GetGattServerInfo:
        return "GetGattServerInfo";
    case BleDeviceSequence_Nolink:
        return "Nolink";
    case BleDeviceSequence_ReadDeviceInfo:
        return "ReadDeviceInfo";
    case BleDeviceSequence_UpdateMtuSize:
        return "UpdateMtuSize";
    case BleDeviceSequence_ReadBatteryStatus:
        return "ReadBatteryStatus";
    case BleDeviceSequence_SuspendFeature:
        return "SuspendFeature";
    case BleDeviceSequence_Active:
        return "Active";
    default:
        return "Unknown";
    }
}

void BleDeviceHandler::PrintDeviceInformation() NN_NOEXCEPT
{
    NN_DETAIL_XCD_INFO("Ble device information\n");
    NN_DETAIL_XCD_INFO(" -> ProductType: Main:0x%02X, Sub:0x%02X\n", m_ProductType.main, m_ProductType.sub);
    NN_DETAIL_XCD_INFO(" -> ModelInformation:\n");
    NN_DETAIL_XCD_INFO("    SensorHorizontalOffset: ");
    DumpSensorValue(m_ModelInfo.sensorHorizontalOffset);
    NN_DETAIL_XCD_INFO("    IsScalingRequired: %s\n", m_ModelInfo.isScalingRequired ? "TRUE" : "FALSE");
    NN_DETAIL_XCD_INFO("    Noise: %d\n", m_ModelInfo.analogStickNoise);
    NN_DETAIL_XCD_INFO("    TypicalStroke: %d\n", m_ModelInfo.analogStickTypicalStroke);
    NN_DETAIL_XCD_INFO("    OriginPlay: %d\n", m_ModelInfo.analogStickOriginPlay);
    NN_DETAIL_XCD_INFO("    CircuitValidRation: %d\n", m_ModelInfo.analogStickCircuitValidRatio);
    NN_DETAIL_XCD_INFO("    MinimumStrokePositive: ");
    DumpStickValue(m_ModelInfo.analogStickMinimumStrokePositive);
    NN_DETAIL_XCD_INFO("    MinimumStrokeNegative: ");
    DumpStickValue(m_ModelInfo.analogStickMinimumStrokeNegative);
    NN_DETAIL_XCD_INFO("    OriginRangeMin: ");
    DumpStickValue(m_ModelInfo.analogStickOriginRangeMin);
    NN_DETAIL_XCD_INFO("    OriginRangeMax: ");
    DumpStickValue(m_ModelInfo.analogStickOriginRangeMax);
    NN_DETAIL_XCD_INFO(" -> SensorCal:\n");
    NN_DETAIL_XCD_INFO("    Accelerometer Origin: ");
    DumpSensorValue(m_SensorCalibration.accelerometerOrigin);
    NN_DETAIL_XCD_INFO("    Accelerometer 1G Scale: ");
    DumpSensorValue(m_SensorCalibration.accelerometerSensitivity);
    NN_DETAIL_XCD_INFO("    Gyroscope Origin: ");
    DumpSensorValue(m_SensorCalibration.gyroscopeOrigin);
    NN_DETAIL_XCD_INFO("    Gyroscope +78rpm Scale: ");
    DumpSensorValue(m_SensorCalibration.gyroscopeSensitivity);
    NN_DETAIL_XCD_INFO(" -> AnalogStickCal:\n");
    NN_DETAIL_XCD_INFO("    CircuitMax: ");
    DumpStickValue(m_AnalogStickCalibration.circuitMax);
    NN_DETAIL_XCD_INFO("    Origin: ");
    DumpStickValue(m_AnalogStickCalibration.origin);
    NN_DETAIL_XCD_INFO("    CircuitMin: ");
    DumpStickValue(m_AnalogStickCalibration.circuitMin);
}

}} // namespace nn::xcd
