﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd_AttachmentDevice.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_Peripheral.h"
#include "xcd_TeraBase.h"
#include "detail/xcd_TeraCommon.h"

namespace nn { namespace xcd {

enum AttachmentStatus
{
    AttachmentStatus_Detached               = 0, //!< 未接続
    AttachmentStatus_Attached               = 1, //!< 物理的に接続された状態 (CONDET6 を検知した)
    AttachmentStatus_Ready                  = 2, //!< 電源供給され、Inquiry を待機している状態
    AttachmentStatus_CommunicationAvailable = 3, //!< Inquiry の送受信が完了し、Command のやり取りが可能な状態
};

// BT MCU にコマンドを投げて Event で TimedWait する時間。
// BT 側の Timeout が 3 秒なので、それよりも長めに。
const nn::TimeSpan AttachmentCommandResponseTimeout = nn::TimeSpan::FromSeconds(5);

// 最大サイズは AnalogStick の 2 byte + 0x24 = 38 byte
const size_t MaxPollingModeReceiveDataSizeForSixAxisSensorDisable = 38;

// 最大サイズは AnalogStick の 2 byte + 0x06 = 8 byte
const size_t MaxPollingModeReceiveDataSizeForSixAxisSensorEnable = 8;

/**
 * @brief   Attachment の実装のためのベースとなるクラス
 */
    class AttachmentBase final : public Peripheral, public ICommandListener
    {
        NN_DISALLOW_COPY(AttachmentBase);
        NN_DISALLOW_MOVE(AttachmentBase);

    public:
        AttachmentBase() NN_NOEXCEPT :
            m_AttachmentStatusMutex(true),
            m_AttachmentStatus(AttachmentStatus_Detached),
            m_ExtDevType(0x00),
            m_ExtDataStatus(0),
            m_ExtDataSize(0),
            m_pAttachmentDataReceiveEvent(nullptr),
            m_AckEventForConfigExtDevIn(nn::os::EventClearMode_AutoClear),
            m_AckEventForEnablePollingMode(nn::os::EventClearMode_AutoClear),
            m_AckEventForDisablePollingMode(nn::os::EventClearMode_AutoClear),
            m_AckEventForExtControl(nn::os::EventClearMode_AutoClear),
            m_AckEventForSensorPayload(nn::os::EventClearMode_AutoClear),
            m_IsEnablePollingMode(false),
            m_AttachmentExBitTemp(false)
        {}

        virtual void Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT NN_OVERRIDE;

        //!< 拡張デバイスが接続されているかどうかを取得します。
        bool IsExtDevAttached() NN_NOEXCEPT;

        //!< 拡張デバイスの状態を更新します。
        void UpdateAttachmentStatus(DeviceStatus status) NN_NOEXCEPT;

        //!< 拡張デバイスの状態を取得します。
        AttachmentStatus GetAttachmentStatus() NN_NOEXCEPT;

        //!< 拡張デバイスの状態を設定します。
        void SetAttachmentStatus(AttachmentStatus status) NN_NOEXCEPT;

        //!< 拡張デバイスの情報を設定します。
        void SetAttachmentType(uint8_t type) NN_NOEXCEPT;

        //!< 拡張デバイスの情報を取得します。
        Result GetAttachmentType(uint8_t* pOutAttachmentType) NN_NOEXCEPT;

        //!< 拡張デバイスと Inquiry Command の送受信を行います。
        Result SendInquiryCommand() NN_NOEXCEPT;

        //!< 拡張デバイスへコマンドを送信します。
        Result SendCommandToAttachmentDevice(const uint8_t* pInData, size_t size) NN_NOEXCEPT;

        //!< 拡張デバイスからの返答を取得します。
        Result GetReceivedCommandFromAttachmentDevice(size_t* pOutSize, uint8_t* pOutExtDevData, uint8_t size) NN_NOEXCEPT;

        //!< 拡張デバイスからのデータ受信を通知するイベントを設定します。
        Result SetAttachmentDataReceiveEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

        //!< 拡張デバイスの Polling Mode を有効化します。
        Result EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, AttachmentPollingMode mode, nn::xcd::DeviceType type) NN_NOEXCEPT;

        //!< 拡張デバイスの Polling Mode を無効化します。
        Result DisablePollingReceiveModeForAttachmentDevice() NN_NOEXCEPT;

        //!< 拡張デバイスの Polling Mode で取得したデータを取り出します。
        Result GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize) NN_NOEXCEPT;

        //!< 拡張デバイスの Polling Mode が有効化どうかを取得します。
        bool IsEnablePollingMode() NN_NOEXCEPT
        {
            return m_IsEnablePollingMode;
        }

        //!< 拡張デバイスの Polling Mode を取得します。
        AttachmentPollingMode GetAttachmentPollingMode() NN_NOEXCEPT
        {
            return m_PollingMode;
        }

        //!< 拡張デバイスの Polling Mode 使用時の有効なセンサーデータの数を取得します。
        //!< (TORIAEZU : 本来使う分は 6byte だけですが、今はサンプリングをまるっと 2 つにしている)
        int GetValidSensorCount() NN_NOEXCEPT
        {
            if (!IsEnablePollingMode())
            {
                return 3;
            }
            else
            {
                if (GetAttachmentPollingMode() == AttachmentPollingMode_SixAxisSendorDisable)
                {
                     return 0;
                }
                else if (GetAttachmentPollingMode() == AttachmentPollingMode_SixAxisSendorEnable)
                {
                    return 2;
                }
                else
                {
                    NN_ABORT("Invalid Polling Mode\n");
                }
            }
        }

        //!< 拡張デバイスの Inquiry 完了を受け取るための通知関数
        virtual void NotifyExtDevInfo(uint8_t extDeviceType) NN_NOEXCEPT NN_OVERRIDE;

        //!< 拡張デバイスのデータ受信完了を受け取るための通知関数
        virtual void NotifyExtDevRead(uint8_t status, const uint8_t* pData, uint8_t size) NN_NOEXCEPT NN_OVERRIDE;

        //!< 拡張デバイス機能の要求バージョンを設定します。
        void SetRequiredVersion(AttachmentFunctionRequiredVersion version) NN_NOEXCEPT
        {
            m_RequiredVersion = version;
        }

        //!< 拡張デバイス機能の BTFW の要求バージョンを満たしているか確認します。
        bool BtVersionCheck(BtFirmwareVersion btVersion)
        {
            if ((btVersion.major < m_RequiredVersion.btVersion.major) ||
                ((btVersion.major == m_RequiredVersion.btVersion.major) && (btVersion.minor < m_RequiredVersion.btVersion.minor)))
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        //!< 拡張デバイス機能の MCU の要求バージョンを満たしているか確認します。
        bool McuVersionCheck(McuVersionData mcuVersion)
        {
            if ((mcuVersion.major < m_RequiredVersion.mcuVersion.major) ||
                ((mcuVersion.major == m_RequiredVersion.mcuVersion.major) && (mcuVersion.minor < m_RequiredVersion.mcuVersion.minor)))
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        //!< 拡張デバイス機能の要求バージョンを取得します。
        AttachmentFunctionRequiredVersion GetRequiredVersion() NN_NOEXCEPT
        {
            return m_RequiredVersion;
        }

        //!< 即値でジョイコンから渡される AttachmentExBit の値を内部で保持し、安定時間が過ぎてなければ古い値を、安定時間が過ぎていれば新しい値を返します。
        bool GetStabledAttachmentExBitValue(bool immediateAttachmentExBitValue, nn::TimeSpan tickValue) NN_NOEXCEPT;

        //!< ExtControl の Ack が返ってくるまで待機します。
        Result WaitExtControlAck() NN_NOEXCEPT;

        //!< 拡張デバイスの受信パケット解析処理
        virtual void ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT NN_OVERRIDE;

        //!< Ack 受信時の処理
        virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;

        //!< Tera の電源制御時のコマンド送信完了を受け取る
        virtual void NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    private:
        //!< 拡張デバイスのステータス管理用 Mutex
        nn::os::Mutex    m_AttachmentStatusMutex;

        //!< xcd の DeviceType
        nn::xcd::DeviceType m_DeviceType;

        //!< 拡張デバイスのステータス
        AttachmentStatus m_AttachmentStatus;

        //!< 拡張デバイスの ID
        uint8_t m_ExtDevType;

        //!< 拡張デバイスからの受信データ
        uint8_t m_ExtDataStatus;
        uint8_t m_ExtData[AttachmentReceiveCommandSizeMax];
        uint8_t m_ExtDataSize;

        //!< 拡張デバイスからの Polling Mode の受信データ
        uint8_t m_ExtPollingData[AttachmentPollingReceiveCommandSizeMax];
        uint8_t m_ExtPollingDataSize;

        //!< 拡張デバイスからのデータの受信を通知するためのイベントへのポインタ
        nn::os::SystemEventType* m_pAttachmentDataReceiveEvent;

        //!< 拡張デバイスの Required Version
        AttachmentFunctionRequiredVersion m_RequiredVersion;

        //!< 最新の Ack の result
        Result m_LatestAckResult;

        //!< 最新の Ack の id
        uint8_t m_LatestAckId;

        //!< ConfigExtDevIn の Ack の通知用のイベント
        nn::os::Event m_AckEventForConfigExtDevIn;

        //!< EnablePollingMode の Ack の通知用のイベント
        nn::os::Event m_AckEventForEnablePollingMode;

        //!< DisablePollingMode の Ack の通知用のイベント
        nn::os::Event m_AckEventForDisablePollingMode;

        //!< ExtControl の Ack の通知用のイベント(Ack として McuRead を受け取る)
        nn::os::Event m_AckEventForExtControl;

        //!< SensorPayload の Ack の通知用のイベント
        nn::os::Event m_AckEventForSensorPayload;

        //!< PollingMode の有効・無効
        bool m_IsEnablePollingMode;

        //!< PollingMode
        AttachmentPollingMode m_PollingMode;

        //!< AttachmentExBit の即値を保持しておく変数
        bool m_AttachmentExBitTemp;

        //!< AttachmentExBit の即値を取得した時間を保持しておく変数
        nn::TimeSpan m_AttachmentExBitTempGetTime;

    };
}} // namespace nn::xcd
