﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Usb.h>
#include <nn/xcd/xcd_KuinaFirmware.h>

#include "xcd_BluetoothHidTask-hardware.nx.h"
#include "xcd_LinkMonitorTypes.h"
#include "xcd_UsbHidDriver-os.horizon.h"
#include "xcd_UsbHidAccessor-os.horizon.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail{

//!< UsbHid で接続できるデバイスの最大数
const int UsbHidDeviceCountMax = 8;

//!< デバイスの接続状態の監視を行うためのプラットフォームごとの実装クラス
class UsbHidManager final : public IEventTask
{
    NN_DISALLOW_MOVE(UsbHidManager);
    NN_DISALLOW_COPY(UsbHidManager);

private:
    //!< デバイスの接続状態に変化があった際に通知するためのイベントオブジェクト
    nn::os::LightEventType* m_pDeviceUpdateEvent;

    UsbHidDriver m_Drivers[UsbHidDeviceCountMax];
    UsbHidAccessor m_Accessors[UsbHidDeviceCountMax];

    //!< Bluetooth接続されているデバイスのリスト
    HidDeviceInfo m_BluetoothDevices[HidDeviceCountMax];

    //!< ファームウェア更新中のデバイス
    HidDeviceInfo m_FirmwareUpdatingDevice;

    //!< 接続されている Bluetooth デバイスの数
    size_t m_BluetoothDeviceCount;

    //!< アクティベートされているかどうか
    bool m_Activated;

    //!< 接続処理の完了通知を受けるイベント
    ::nn::os::LightEventType m_ConnectionEvent;

    //!< USB FullKeyController での操作を有効にするかどうか
    bool m_IsFullKeyUsbEnabled;

    //!< USB FullKeyController の有効状態が変更されたか
    bool m_IsFullKeyUsbEnabledUpdated;

    //!< Suspend 中かどうか
    bool m_IsSuspending;

    //!< Suspend 済みかどうか
    bool m_IsSuspended;

    //!< サスペンド処理が完了したときに上位に通知するイベント
    ::nn::os::LightEventType* m_pSuspendCompletedEvent;

public:
    UsbHidManager() NN_NOEXCEPT;
    virtual ~UsbHidManager() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態の監視を開始する
    void StartMonitoring(nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を停止する
    void StopMonitoring() NN_NOEXCEPT;

    //!< 接続済みデバイスのハンドルを取得する
    size_t GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT;

    //!< デバイスハンドルからHIDAccessorを取得する
    HidAccessor* GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< 接続されているBluetoothのデバイスリストを登録する
    void SetBluetoothDeviceList(HidDeviceInfo* pDevices, size_t deviceCount) NN_NOEXCEPT;

    //!< 指定された Usb デバイスをサポートしているか評価します
    bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

    //!< USB デバイスを追加します
    Result AddDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT;

    //!< USB デバイスを削除します
    Result RemoveDevice(int index) NN_NOEXCEPT;

    //!< Input Report を処理します
    Result SetInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT;

    //!< Output Report を取得します
    size_t GetOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT;

    //!< USB のデバイスリストを更新する
    void UpdateUsbHidDeviceLists() NN_NOEXCEPT;

    //!< USB FullKeyController を有効にする
    void SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT;

    //!< デバイスが USB 接続されているか評価します
    bool IsUsbConnected(::nn::bluetooth::Address address) NN_NOEXCEPT;

    //!< Hid で待機状態のデバイスを再認識する
    void ReInitializePendingDevices() NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイスを指定します
    Result SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイス指定を解除します
    void UnsetFirmwareUpdatingDevice() NN_NOEXCEPT;

    //!< UsbHid の処理を Suspend します。処理は非同期で行われます
    void TriggerSuspend(::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< UsbHid の処理を Resume します。同期的に完了します
    void Resume() NN_NOEXCEPT;

    //!< Kuina の Fw バージョンを取得します
    Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT;

    //!< Kuina の Fw バージョンを要求します
    Result RequestKuinaVersion(int index) NN_NOEXCEPT;

    //!< Kuina の Fw 更新モードに遷移させます
    Result SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT;

};

}}} // namespace nn::xcd::detail
