﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_UsbHidManager-os.horizon.h"
#include "../xcd_DeviceHandleGenerator.h"
#include "../xcd_TaskManager.h"

namespace nn { namespace xcd { namespace detail {

UsbHidManager::UsbHidManager() NN_NOEXCEPT :
     m_FirmwareUpdatingDevice(),
     m_Activated(false),
     m_IsFullKeyUsbEnabled(false),
     m_IsFullKeyUsbEnabledUpdated(false),
     m_IsSuspending(false),
     m_IsSuspended(false)
{
    // 何もしない
}

UsbHidManager::~UsbHidManager() NN_NOEXCEPT
{
    // 何もしない
}

void UsbHidManager::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void UsbHidManager::PeriodicEventFunction() NN_NOEXCEPT
{
    if (::nn::os::TryWaitLightEvent(&m_ConnectionEvent))
    {
        ::nn::os::ClearLightEvent(&m_ConnectionEvent);

        // Suspend 中はデバイス更新を無効にする
        if (!m_IsSuspending && !m_IsSuspended)
        {
            UpdateUsbHidDeviceLists();
        }
        if (m_IsSuspending)
        {
            bool isSuspended = true;
            for (int i = 0; i < UsbHidDeviceCountMax; i++)
            {
                if (m_Drivers[i].IsNxDeviceHidActivated() == true)
                {
                    // TriggerSuspend 後に Activate される場合もあるので、ここでも Deactivate する必要がある
                    // 既に Deactivate シーケンスに入っている場合には内部で Busy 判定となり、重ねて実行されることはない
                    m_Drivers[i].DeactivateUsbHidOnController();
                    isSuspended = false;
                    break;
                }
            }
            if (isSuspended)
            {
                m_IsSuspending = false;
                m_IsSuspended = true;
                ::nn::os::SignalLightEvent(m_pSuspendCompletedEvent);
            }
        }
    }
}

void UsbHidManager::StartMonitoring(nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pDeviceUpdateEvent = pEvent;
    m_BluetoothDeviceCount = 0;

    ::nn::os::InitializeLightEvent(&m_ConnectionEvent, false, ::nn::os::EventClearMode_ManualClear);

    for (auto& driver : m_Drivers)
    {
        driver.StartMonitoring(&m_ConnectionEvent);
    }

    GetTaskManager().RegisterPeriodicTask(this);

    m_Activated = true;
}

void UsbHidManager::StopMonitoring() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);

    for (auto& driver : m_Drivers)
    {
        driver.StopMonitoring();
    }

    ::nn::os::FinalizeLightEvent(&m_ConnectionEvent);

    m_Activated = false;
}

size_t UsbHidManager::GetDevices(HidDeviceInfo* pOutValue, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    size_t returnCount = 0;

    HidDeviceInfo deviceInfo;

    deviceInfo.vid = 0x57E;
    deviceInfo.pid = 0x2006;
    deviceInfo.interfaceType = InterfaceType_Usb;

    for (int i = 0; i < UsbHidDeviceCountMax; i++)
    {
        auto& driver = m_Drivers[i];
        if (driver.IsNxDeviceHidActivated() && driver.GetDeviceInfo(&(deviceInfo.address), &(deviceInfo.deviceType)).IsSuccess())
        {
            deviceInfo.deviceHandle = m_Accessors[i].GetDeviceHandle();
            pOutValue[returnCount] = deviceInfo;
            returnCount++;
        }
    }

    return returnCount;
}

HidAccessor* UsbHidManager::GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    for (auto& accessor : m_Accessors)
    {
        if (accessor.GetDeviceHandle() == deviceHandle && accessor.IsActivated() == true)
        {
            return &accessor;
        }
    }

    return nullptr;
}

void UsbHidManager::SetBluetoothDeviceList(HidDeviceInfo* pDevices, size_t deviceCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDevices);
    for (int i = 0; i < sizeof(m_BluetoothDevices) / sizeof(m_BluetoothDevices[0]); i++)
    {
        if (i < deviceCount)
        {
            m_BluetoothDevices[i] = pDevices[i];
        }
    }
    m_BluetoothDeviceCount = deviceCount;
}

bool UsbHidManager::IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    return UsbHidDriver::IsUsbHidSupported(deviceInfo);
}

Result UsbHidManager::AddDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    NN_RESULT_DO(m_Drivers[index].AddDevice(deviceInfo));
    NN_DETAIL_XCD_INFO("%s index=%d PID=%x VID=%x\n", NN_CURRENT_FUNCTION_NAME, index, deviceInfo.pid, deviceInfo.vid);
    NN_RESULT_SUCCESS;
}

Result UsbHidManager::RemoveDevice(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(m_Drivers[index].RemoveDevice());
    NN_DETAIL_XCD_INFO("%s index=%d\n", NN_CURRENT_FUNCTION_NAME, index);
    NN_RESULT_SUCCESS;
}

Result UsbHidManager::SetInputReport(int index, uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_RESULT_DO(m_Drivers[index].SetInputReport(pBuffer, length));
    NN_RESULT_SUCCESS;
}

size_t UsbHidManager::GetOutputReport(int index, uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT
{
    return m_Drivers[index].GetOutputReport(pOutBuffer, length);
}

void UsbHidManager::UpdateUsbHidDeviceLists() NN_NOEXCEPT
{
    // デバイスリストの更新が完了したら USB 通信変更フラグを解除
    NN_UTIL_SCOPE_EXIT
    {
        m_IsFullKeyUsbEnabledUpdated = false;
    };

    ::nn::bluetooth::Address address;
    DeviceType deviceType;

    for (int i = 0; i < UsbHidDeviceCountMax; i++)
    {
        auto& driver = m_Drivers[i];
        auto& accessor = m_Accessors[i];

        // USB デバイスが接続されているかどうか
        if (driver.GetDeviceInfo(&address, &deviceType).IsSuccess())
        {
            // Nx の Hid プロトコルが有効化されているかどうか
            if (driver.IsNxDeviceHidActivated())
            {
                // Hid プロトコルが有効になった

                if (accessor.IsActivated() == true)
                {
                    // USB FullKeyController の有効状態切り替え直後のみ、USB 通信中のデバイスを切断する
                    if (deviceType == DeviceType_FullKey &&
                        m_IsFullKeyUsbEnabled == false &&
                        m_IsFullKeyUsbEnabledUpdated == true)
                    {
                        accessor.DetachDevice();
                    }
                    else
                    {
                        // 確実に Activate させる
                        nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
                    }
                }
                else
                {
                    // Hid Accessor を初期化する
                    auto handle = DeviceHandleGenerator::Get().GetDeviceHandle();
                    accessor.Activate(handle, &driver);
                    nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
                }
            }
            else
            {
                // USB接続されたデバイスが BT 接続されているかどうかチェック
                bool isBtConnected = false;
                for (int btIndex = 0; btIndex < m_BluetoothDeviceCount; btIndex++)
                {
                    if (address == m_BluetoothDevices[btIndex].address)
                    {
                        isBtConnected = true;
                        break;
                    }
                }

                // SIGLO-63952: ファームウェア更新中のデバイスは、Hid 通信を有効にしないように BT 接続状態と見なす
                isBtConnected |= (address == m_FirmwareUpdatingDevice.address);

                // BT 接続されていない場合のみ対処する
                if (isBtConnected == false ||
                    (deviceType == DeviceType_FullKey && m_IsFullKeyUsbEnabled == true) ||
                    (deviceType == DeviceType_MiyabiLeft) ||
                    (deviceType == DeviceType_MiyabiRight))
                {
                    // 一度も Nx の Hid が有効になっていない場合は有効にする
                    // ペアリング処理を繰り返し行わないため
                    if (driver.GetNxHidProtocolActivationCount() == 0 ||
                        (deviceType == DeviceType_FullKey && m_IsFullKeyUsbEnabled == true) ||
                        (deviceType == DeviceType_MiyabiLeft) ||
                        (deviceType == DeviceType_MiyabiRight))
                    {
                        // 確実に Deactivate させる
                        nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
                        driver.ActivateUsbHidOnController();
                    }
                    // 一度 Nx の Hidプロトコルが有効になった場合は、アクセサを無効にする
                    else
                    {
                        if (accessor.IsActivated() == true)
                        {
                            accessor.Deactivate();
                            nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
                        }
                    }
                }
            }
        }
        // USB デバイスが切断された
        else
        {
            if (accessor.IsActivated() == true)
            {
                accessor.Deactivate();
                nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
            }
        }
    }
}

void UsbHidManager::SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT
{
    m_IsFullKeyUsbEnabled        = enabled;
    m_IsFullKeyUsbEnabledUpdated = true;
}

bool UsbHidManager::IsUsbConnected(::nn::bluetooth::Address address) NN_NOEXCEPT
{
    HidDeviceInfo deviceInfo;

    for (int i = 0; i < UsbHidDeviceCountMax; i++)
    {
        auto& driver = m_Drivers[i];

        // USB デバイスとして接続されているかどうか
        if (driver.GetDeviceInfo(&deviceInfo.address, &deviceInfo.deviceType).IsSuccess())
        {
            if (address == deviceInfo.address)
            {
                return true;
            }
        }
    }

    return false;
}

void UsbHidManager::ReInitializePendingDevices() NN_NOEXCEPT
{
    ::nn::bluetooth::Address address;
    DeviceType deviceType;

    for (int i = 0; i < UsbHidDeviceCountMax; i++)
    {
        auto& driver = m_Drivers[i];

        // USB デバイスが接続されているかどうか
        if (driver.GetDeviceInfo(&address, &deviceType).IsSuccess())
        {
            // Nx の Hid プロトコルが有効化されているかどうか
            if (driver.IsNxDeviceHidActivated() == false)
            {
                // USB接続されたデバイスが BT 接続されているかどうかチェック
                bool isBtConnected = false;
                for (int btIndex = 0; btIndex < m_BluetoothDeviceCount; btIndex++)
                {
                    if (address == m_BluetoothDevices[btIndex].address)
                    {
                        isBtConnected = true;
                        break;
                    }
                }

                // BT 接続されていない場合のみ対処する Nx の Hid プロトコルを有効にする
                if (isBtConnected == false ||
                    (deviceType == DeviceType_FullKey && m_IsFullKeyUsbEnabled == true) ||
                    (deviceType == DeviceType_MiyabiLeft)||
                    (deviceType == DeviceType_MiyabiRight))
                {
                    // Nx Hid Protocol の有効化回数をリセット
                    driver.ResetNxHidProtocolActivationCount();
                    driver.ActivateUsbHidOnController();
                }

            }
        }
    }
}

Result UsbHidManager::SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceInfo info;
    NN_RESULT_DO(GetDeviceInfo(&info, handle));

    // 必要な情報はアドレスのみなので、差し当たり全て埋める必要はない
    m_FirmwareUpdatingDevice.deviceHandle = handle;
    m_FirmwareUpdatingDevice.address      = info.address;
    m_FirmwareUpdatingDevice.deviceType   = info.deviceType;

    NN_RESULT_SUCCESS;
}

void UsbHidManager::UnsetFirmwareUpdatingDevice() NN_NOEXCEPT
{
    m_FirmwareUpdatingDevice = HidDeviceInfo();
}

void UsbHidManager::TriggerSuspend(::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_IsSuspending = true;
    m_pSuspendCompletedEvent = pEvent;

    bool isSuspended = true;
    for (int i = 0; i < UsbHidDeviceCountMax; i++)
    {
        if (m_Drivers[i].IsNxDeviceHidActivated() == true)
        {
            m_Drivers[i].DeactivateUsbHidOnController();
            isSuspended = false;
        }
    }
    if (isSuspended)
    {
        m_IsSuspending = false;
        m_IsSuspended = true;
        ::nn::os::SignalLightEvent(m_pSuspendCompletedEvent);
    }
}

void UsbHidManager::Resume() NN_NOEXCEPT
{
    m_IsSuspended = false;

    // Resume 直後の状態をチェック
    UpdateUsbHidDeviceLists();
}

Result UsbHidManager::GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT
{
    auto& driver = m_Drivers[index];
    return driver.GetKuinaVersion(pOutMcuVersionData);
}

Result UsbHidManager::RequestKuinaVersion(int index) NN_NOEXCEPT
{
    auto& driver = m_Drivers[index];

    nn::Result result = driver.RequestKuinaVersion();
    if (result.IsSuccess())
    {
        NN_DETAIL_XCD_INFO("%s Success.\n", NN_CURRENT_FUNCTION_NAME);
    }
    else
    {
        NN_DETAIL_XCD_INFO("%s Fail.\n", NN_CURRENT_FUNCTION_NAME);
    }
    NN_RESULT_SUCCESS;
}

Result UsbHidManager::SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT
{
    auto& driver = m_Drivers[index];

    nn::Result result = driver.SetKuinaToFirmwareUpdateMode();
    if (result.IsSuccess())
    {
        NN_DETAIL_XCD_INFO("%s Success.\n", NN_CURRENT_FUNCTION_NAME);
    }
    else
    {
        NN_DETAIL_XCD_INFO("%s Fail.\n", NN_CURRENT_FUNCTION_NAME);
    }
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::xcd::detail
