﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/pinmux/pinmux.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_UsbHidDriver-os.horizon.h"
#include "../xcd_ReportTypes.h"
#include "../xcd_CommandTypes.h"
#include "../xcd_TaskManager.h"

#define ENABLE_START_HID_DATA

namespace nn { namespace xcd { namespace detail {

namespace {

    //!< Product Id の定義です。
struct ProductId final
{
    static const uint16_t ExternalGrip       = 0x2008;
    static const uint16_t FullKeyUsb         = 0x2009;
    static const uint16_t ExternalGripUpdate = 0x200e;
    static const uint16_t FullKeyUsbUpdate   = 0x200f;
};

//!< Vendor Id の定義です。
struct VendorId final
{
    static const uint16_t Nintendo = 0x57e;
};

//!< NxDevice の扱う ReportId かどうかをチェックする
bool IsNxHidReport(uint8_t reportId)
{
    if(reportId == Report_AudioIn::Id ||
       reportId == Report_AttachmentIn::Id ||
       reportId == Report_BasicIn::Id ||
       reportId == Report_CommandIn::Id ||
       reportId == Report_GenericIn::Id ||
       reportId == Report_McuIn::Id)
    {
        return true;
    }

    return false;
}

// Inquiry 受信待ちの Timeout 値
const auto InquiryTimeout = ::nn::TimeSpan::FromMilliSeconds(1500);

// DeleteHidConnection レスポンス待ちの Timeout 値
const auto DeleteHidConnectionTimeout = ::nn::TimeSpan::FromMilliSeconds(500);
}

UsbHidDriver::UsbHidDriver() NN_NOEXCEPT :
    m_Mutex(false),
    m_Activated(false),
    m_IsAttached(false),
    m_pInputReportParserFunc(nullptr),
    m_pInputReportParserArg(nullptr),
    m_IsAvailableKuinaFirmwareVersion(false)
{
    ResetOutputBuffer();
}

UsbHidDriver::~UsbHidDriver() NN_NOEXCEPT
{
    // 何もしない
}

void UsbHidDriver::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void UsbHidDriver::PeriodicEventFunction() NN_NOEXCEPT
{
    if (::nn::os::TryWaitTimerEvent(&m_ExtGripOutTimeoutEvent))
    {
        ::nn::os::ClearTimerEvent(&m_ExtGripOutTimeoutEvent);

        if (!m_IsNxDeviceAttached)
        {
            // Inquiry が来ない場合は明示的に要求する
            // セルフパワーハブで Hid が維持されてしまう場合がある
            m_IsPacketReceived = true;
            SendStopHidData();
            SendInquiry();
        }
        else
        {
            if (m_RetryCount == 0)
            {
                // 切断失敗しているので再度切断を試みる
                ::nn::os::StartOneShotTimerEvent(&m_ExtGripOutTimeoutEvent, DeleteHidConnectionTimeout);
                SendStopHidData();
                m_RetryCount++;
                SendDeleteHidConnection();
            }
            else
            {
                // コマンド失敗したら強制切断
                m_IsExtGripOutBusy = false;
                m_IsNxDeviceAvailable = false;
                ResetOutputBuffer();
                m_RetryCount = 0;
                nn::os::SignalLightEvent(m_pConnectionEvent);
            }
        }
        return;
    }
}

void UsbHidDriver::StartMonitoring(nn::os::LightEventType* pDeviceUpdateEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);
    NN_SDK_REQUIRES_NOT_NULL(pDeviceUpdateEvent);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    ::nn::os::InitializeTimerEvent(&m_ExtGripOutTimeoutEvent, ::nn::os::EventClearMode_ManualClear);

    GetTaskManager().RegisterPeriodicTask(this);

    m_pConnectionEvent = pDeviceUpdateEvent;
    m_IsAttached = false;
    m_IsExtGripOutBusy = false;
    m_IsNxDeviceAvailable = false;
    m_NxHidProtocolActivationCount = 0;
    m_IsPacketReceived = false;

    m_Activated = true;

    m_RetryCount = 0;
}

void UsbHidDriver::StopMonitoring() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    GetTaskManager().UnregisterPeriodicTask(this);

    ::nn::os::FinalizeTimerEvent(&m_ExtGripOutTimeoutEvent);
    m_pConnectionEvent = nullptr;
    m_Activated = false;
}

void UsbHidDriver::SetSampleParserFunction(InputReportParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);

    {
        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        m_pInputReportParserFunc = func;
        m_pInputReportParserArg  = pArg;
    }

    // 受信済み InputReport が存在する場合は解析
    if (m_HidInputReportLength > 0)
    {
        m_pInputReportParserFunc(m_pInputReportParserArg, m_HidInputReportBuffer, m_HidInputReportLength);
    }
}

bool UsbHidDriver::IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    if (deviceInfo.vid == VendorId::Nintendo)
    {
        switch (deviceInfo.pid)
        {
        case ProductId::ExternalGrip:
        case ProductId::FullKeyUsb:
        case ProductId::ExternalGripUpdate:
        case ProductId::FullKeyUsbUpdate:
            return true;
        default:
            return false;
        }
    }

    return false;
}

Result UsbHidDriver::AddDevice(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsAttached == false, ResultUsbHidAttachedDeviceExist());
    NN_RESULT_THROW_UNLESS(IsUsbHidSupported(deviceInfo), ResultUsbHidUnsupportedDevice());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_IsExtGripOutBusy = false;
    m_IsNxDeviceActivating = false;
    m_IsNxDeviceAvailable = false;
    m_HidInputReportLength = 0;
    m_NxHidProtocolActivationCount = 0;
    m_IsAttached = true;
    m_UsbHidDeviceInfo = deviceInfo;

    ::nn::os::StartOneShotTimerEvent(&m_ExtGripOutTimeoutEvent, InquiryTimeout);

    NN_RESULT_SUCCESS;
}

Result UsbHidDriver::RemoveDevice() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsAttached, ResultUsbHidDeviceDetached());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    HandleNxDeviceDetach();

    m_IsAttached = false;
    NN_RESULT_SUCCESS;
}

Result UsbHidDriver::GetDeviceInfo(::nn::bluetooth::Address* pOutAddress, DeviceType* pOutDeviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceType);
    NN_RESULT_THROW_UNLESS(m_IsNxDeviceAttached, ResultUsbHidDeviceDetached());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    *pOutAddress = m_Address;
    *pOutDeviceType = m_DeviceType;

    NN_RESULT_SUCCESS;
}

void UsbHidDriver::ActivateUsbHidOnController() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    if (!m_IsNxDeviceAvailable && !m_IsNxDeviceActivating && !m_IsExtGripOutBusy)
    {
        m_IsNxDeviceActivating = true;
        m_IsExtGripOutBusy = true;
        SendCreateHidConnection();
    }
}

void UsbHidDriver::DeactivateUsbHidOnController() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    if (m_IsNxDeviceAvailable && !m_IsExtGripOutBusy)
    {
        m_IsExtGripOutBusy = true;
        ::nn::os::StartOneShotTimerEvent(&m_ExtGripOutTimeoutEvent, DeleteHidConnectionTimeout);
        SendStopHidData();
        m_RetryCount = 0;
        SendDeleteHidConnection();
    }
}

Result UsbHidDriver::GetKuinaVersion(KuinaVersionData* pOutMcuVersionData) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsAttached, ResultUsbHidDeviceDetached());
    NN_RESULT_THROW_UNLESS(m_IsAvailableKuinaFirmwareVersion, ResultUsbHidFirmwareVersionNotAvailable());

    if (m_IsAvailableKuinaFirmwareVersion)
    {
        pOutMcuVersionData->major = m_KuinaFirmwareVersion.major;
        pOutMcuVersionData->minor = m_KuinaFirmwareVersion.minor;
        pOutMcuVersionData->micro = m_KuinaFirmwareVersion.micro;
        pOutMcuVersionData->type = m_KuinaFirmwareVersion.type;
    }

    NN_RESULT_SUCCESS;
}

Result UsbHidDriver::RequestKuinaVersion() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsAttached, ResultUsbHidDeviceDetached());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    //FWバージョンは未取得
    if (m_IsAvailableKuinaFirmwareVersion == false)
    {
        // KuinaデバイスであればFW要求コマンドを送る
        if (m_UsbHidDeviceInfo.pid == ProductId::ExternalGrip
            || m_UsbHidDeviceInfo.pid == ProductId::FullKeyUsb)
        {
            SendGetFwVersion();
        }
    }

    NN_RESULT_SUCCESS;
}

Result UsbHidDriver::SetKuinaToFirmwareUpdateMode() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsAttached, ResultUsbHidDeviceDetached());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    // Kuinaデバイスであればコマンドを送る
    if (m_UsbHidDeviceInfo.pid == ProductId::ExternalGrip
        || m_UsbHidDeviceInfo.pid == ProductId::FullKeyUsb)
    {
        SendGotoFwUpdateMode();
    }

    NN_RESULT_SUCCESS;
}

bool UsbHidDriver::IsNxDeviceHidActivated() NN_NOEXCEPT
{
    return m_IsNxDeviceAvailable;
}

int UsbHidDriver::GetNxHidProtocolActivationCount() NN_NOEXCEPT
{
    return m_NxHidProtocolActivationCount;
}

void UsbHidDriver::ResetNxHidProtocolActivationCount() NN_NOEXCEPT
{
    m_NxHidProtocolActivationCount = 0;
}

Result UsbHidDriver::SetInputReport(uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_RESULT_THROW_UNLESS(length <= UsbHidReportLengthMax, ResultUsbHidInvalidReportLength());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    NN_RESULT_DO(SetInputReportImpl(pBuffer, length));
    NN_RESULT_SUCCESS;
}

size_t UsbHidDriver::GetInputReport(uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(length, UsbHidReportLengthMax);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    std::memcpy(pOutBuffer, m_HidInputReportBuffer, m_HidInputReportLength);
    return m_HidInputReportLength;
}

Result UsbHidDriver::SetOutputReport(const uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(length, UsbHidReportLengthMax);
    NN_RESULT_THROW_UNLESS(m_IsAttached, ResultUsbHidDeviceDetached());
    NN_RESULT_THROW_UNLESS(m_HidOutputReportBufferCount < OutputReportRingBufferCountMax, ResultUsbHidOutputRingBufferFull());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    NN_RESULT_DO(SetOutputReportImpl(pBuffer, length));
    NN_RESULT_SUCCESS;
}

size_t UsbHidDriver::GetOutputReport(uint8_t *pOutBuffer, size_t length) NN_NOEXCEPT
{
    size_t hidReportLength = 0;

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_IsPacketReceived)
    {
        if (m_HidOutputReportBufferCount > 0)
        {
            // 送信対象のリングバッファのインデックス
            auto bufferIndex = ((m_HidOutputReportBufferIndex + OutputReportRingBufferCountMax) - m_HidOutputReportBufferCount) % OutputReportRingBufferCountMax;
            m_HidOutputReportBufferCount--;

            hidReportLength = m_HidOutputReportBuffer[bufferIndex].size;
            std::memcpy(pOutBuffer, m_HidOutputReportBuffer[bufferIndex].buffer, hidReportLength);

            if (pOutBuffer[ReportByte_ReportId] == 0x00)
            {
                // ダミーパケットは何もしない
            }
            else if (pOutBuffer[ReportByte_ReportId] == Report_ExtGripOut::Id)
            {
                if (pOutBuffer[ReportByte_ReportId + 1] == ExtGripStatus_StartHidData ||
                    pOutBuffer[ReportByte_ReportId + 1] == ExtGripStatus_StopHidData)
                {
                    // レスポンスがないコマンドでは受信フラグをクリアしない
                    m_IsPacketReceived = true;
                    m_ProcessingExtGripStatus = static_cast<ExtGripStatus>(0);
                }
                else
                {
                    // 受信フラグをクリア
                    m_IsPacketReceived = false;
                    m_ProcessingExtGripStatus = static_cast<ExtGripStatus>(pOutBuffer[ReportByte_ReportId + 1]);
                }
            }
            else
            {
                // 受信フラグをクリア
                m_IsPacketReceived = false;
                m_ProcessingExtGripStatus = static_cast<ExtGripStatus>(0);
            }
        }
#ifndef ENABLE_START_HID_DATA
        else if (m_IsNxDeviceAvailable)
        {
            // 送信データがないときは POLL 用パケットを送信
            hidReportLength = Report_BasicOut::Size;
            std::memset(pOutBuffer, 0, Report_BasicOut::Size);
            pOutBuffer[ReportByte_ReportId] = Report_BasicOut::Id;

            // 受信フラグをクリア
            m_IsPacketReceived = false;
        }
#endif
    }

    return hidReportLength;
}

::nn::TimeSpan UsbHidDriver::GetInterval() NN_NOEXCEPT
{
    return m_UsbHidDeviceInfo.interval;
}

Result UsbHidDriver::SetInputReportImpl (uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_RESULT_THROW_UNLESS(length <= UsbHidReportLengthMax, ResultUsbHidInvalidReportLength());
    NN_RESULT_THROW_UNLESS(length != 0, ResultUsbHidInvalidReportLength());

    // 受信フラグをセット
    m_IsPacketReceived = true;

    if (pBuffer[ReportByte_ReportId] == Report_ExtGripIn::Id)
    {
        // デバイスの状態検出処理
        switch (pBuffer[ExtGripInReportByte_Status])
        {
        case ExtGripStatus_Inquiry:
            ::nn::os::StopTimerEvent(&m_ExtGripOutTimeoutEvent);
            ::nn::os::ClearTimerEvent(&m_ExtGripOutTimeoutEvent);
            if (pBuffer[ExtGripInReportByte_InquiryResult] == 0)
            {
                for (int i = 0; i < ::nn::bluetooth::AddressLength; i++)
                {
                    m_Address.address[i] = pBuffer[ExtGripInReportByte_InquiryBdAddr + 5 - i];
                }
                switch (pBuffer[ExtGripInReportByte_InquiryDeviceType])
                {
                case 0x01: m_DeviceType = DeviceType_Left;        break;
                case 0x02: m_DeviceType = DeviceType_Right;       break;
                case 0x03: m_DeviceType = DeviceType_FullKey;     break;
                case 0x04: m_DeviceType = DeviceType_MiyabiLeft;  break;
                case 0x05: m_DeviceType = DeviceType_MiyabiRight; break;
                default:   m_DeviceType = DeviceType_Unknown;
                }

                NN_DETAIL_XCD_INFO("New UsbHid Device Attached DeviceType=%02x BdAddr=%02x:%02x:%02x:%02x:%02x:%02x\n", m_DeviceType,
                                                                                                                        m_Address.address[0],
                                                                                                                        m_Address.address[1],
                                                                                                                        m_Address.address[2],
                                                                                                                        m_Address.address[3],
                                                                                                                        m_Address.address[4],
                                                                                                                        m_Address.address[5]);

                nn::os::SignalLightEvent(m_pConnectionEvent);

                m_IsNxDeviceAttached = true;
            }
            else
            {
                // デバイスが切断された
                HandleNxDeviceDetach();
            }
            break;
        case ExtGripStatus_CreateHidConnection:
            m_IsExtGripOutBusy = false;
            // KUINA が BasicIn 以外の Data Format に対応できないのでここで初期化しておく
            // Bluetooth 通信から切り替わった場合に McuIn/McuUpdateIn になっている可能性がある
            ResetDataFormat();
            break;
        case ExtGripStatus_DeleteHidConnection:
            ::nn::os::StopTimerEvent(&m_ExtGripOutTimeoutEvent);
            ::nn::os::ClearTimerEvent(&m_ExtGripOutTimeoutEvent);
            m_RetryCount = 0;
            m_IsExtGripOutBusy = false;
            // バッファされたコマンドもクリアする
            m_pInputReportParserFunc = nullptr;
            m_pInputReportParserArg = nullptr;
            m_IsNxDeviceAvailable = false;
            ResetOutputBuffer();
            nn::os::SignalLightEvent(m_pConnectionEvent);
            break;
        case ExtGripStatus_GetVersion:

            //Uart_Status_Success
            if (pBuffer[ExtGripInReportByte_Status + 1] == 0x00)
            {
                m_KuinaFirmwareVersion.type = pBuffer[ExtGripInReportByte_Status + 2];
                m_KuinaFirmwareVersion.major = pBuffer[ExtGripInReportByte_Status + 3];
                m_KuinaFirmwareVersion.minor = pBuffer[ExtGripInReportByte_Status + 4];
                m_KuinaFirmwareVersion.micro = pBuffer[ExtGripInReportByte_Status + 5];
                m_IsAvailableKuinaFirmwareVersion = true;
            }
            /*
            NN_DETAIL_XCD_INFO("GetFwVersion %x %x %x (%x.%x.%x)\n",
                pBuffer[ExtGripInReportByte_Status + 0],//command
                pBuffer[ExtGripInReportByte_Status + 1],//status
                pBuffer[ExtGripInReportByte_Status + 2],//type
                pBuffer[ExtGripInReportByte_Status + 3],//Major
                pBuffer[ExtGripInReportByte_Status + 4],//Minor
                pBuffer[ExtGripInReportByte_Status + 5]//Micro
            );
            */
            break;
        default:
            ;
            // 何もしない
        }
    }
    else if (IsNxHidReport(pBuffer[ReportByte_ReportId]))
    {
        if (m_IsNxDeviceActivating)
        {
            if (pBuffer[ReportByte_ReportId] == Report_CommandIn::Id &&
                pBuffer[CommandInReportByte_Payload + CommandInByte_OutputId] == Command_SetDataFormat.Id)
            {
                // 接続シーケンスの続き
                m_IsNxDeviceActivating = false;
                m_IsNxDeviceAvailable = true;
                m_NxHidProtocolActivationCount++;
                nn::os::SignalLightEvent(m_pConnectionEvent);

#ifdef ENABLE_START_HID_DATA
                SendStartHidData();
#endif

                // ダミーデータを Output Buffer に突っ込む
                SetDummyHidOutputReport();
            }

            // 接続確立前のパケットは無視する
        }
        else
        {
            std::memcpy(m_HidInputReportBuffer, pBuffer, length);
            m_HidInputReportLength = length;
            if (m_pInputReportParserFunc != nullptr)
            {
                // パーサー関数内で SetOutputReport を呼び出しており、Mutex の多重ロックにつながるので、
                // 呼出し前に Mutex をアンロックしておく
                bool isLocked = m_Mutex.IsLockedByCurrentThread();
                if (isLocked)
                {
                    m_Mutex.unlock();
                }
                m_pInputReportParserFunc(m_pInputReportParserArg, pBuffer, length);
                if (isLocked)
                {
                    m_Mutex.lock();
                }
            }
        }
    }
    else
    {
        NN_RESULT_THROW(ResultUsbHidInvalidReportId());
    }

    NN_RESULT_SUCCESS;

}  // NOLINT(readability/fn_size)

Result UsbHidDriver::SetOutputReportImpl(const uint8_t *pBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(length, UsbHidReportLengthMax);
    NN_RESULT_THROW_UNLESS(m_IsAttached, ResultUsbHidDeviceDetached());
    NN_RESULT_THROW_UNLESS(m_HidOutputReportBufferCount < OutputReportRingBufferCountMax, ResultUsbHidOutputRingBufferFull());
    if (length > 0)
    {
        m_HidOutputReportBuffer[m_HidOutputReportBufferIndex].size = length;
        memcpy(m_HidOutputReportBuffer[m_HidOutputReportBufferIndex].buffer, pBuffer, length);

        m_HidOutputReportBufferIndex = (m_HidOutputReportBufferIndex + 1) % OutputReportRingBufferCountMax;
        m_HidOutputReportBufferCount++;
    }

    NN_RESULT_SUCCESS;
}

void UsbHidDriver::SendInquiry() NN_NOEXCEPT
{
    uint8_t buffer[2];
    buffer[0] = Report_ExtGripOut::Id;
    buffer[1] = ExtGripStatus_Inquiry;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::SendCreateHidConnection() NN_NOEXCEPT
{
    uint8_t buffer[2];
    buffer[0] = Report_ExtGripOut::Id;
    buffer[1] = ExtGripStatus_CreateHidConnection;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::SendDeleteHidConnection() NN_NOEXCEPT
{
    uint8_t buffer[2];

#ifdef ENABLE_START_HID_DATA
    // WORKAROUND: 前の処理が遅延してパケットが滞留されないようにダミーを挟む
    buffer[0] = 0;
    buffer[1] = 0;
    SetOutputReportImpl(buffer, sizeof(buffer));
    SetOutputReportImpl(buffer, sizeof(buffer));
#endif

    buffer[0] = Report_ExtGripOut::Id;
    buffer[1] = ExtGripStatus_DeleteHidConnection;
    SetOutputReportImpl(buffer, sizeof(buffer));

#ifdef ENABLE_START_HID_DATA
    // WORKAROUND: 前のパケットが上書きされるのを防ぐためにダミーを挟む
    buffer[0] = 0;
    buffer[1] = 0;
    SetOutputReportImpl(buffer, sizeof(buffer));
#endif
}

void UsbHidDriver::SendStartHidData() NN_NOEXCEPT
{
    uint8_t buffer[2];
    // StartHidData はレスポンスがない
    buffer[0] = Report_ExtGripOut::Id;
    buffer[1] = ExtGripStatus_StartHidData;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::SendStopHidData() NN_NOEXCEPT
{
    uint8_t buffer[2];

#ifdef ENABLE_START_HID_DATA
    // WORKAROUND: 前の処理が遅延してパケットが滞留されないようにダミーを挟む
    buffer[0] = 0;
    buffer[1] = 0;
    SetOutputReportImpl(buffer, sizeof(buffer));
    SetOutputReportImpl(buffer, sizeof(buffer));
#endif

    // StopHidData はレスポンスがない
    buffer[0] = Report_ExtGripOut::Id;
    buffer[1] = ExtGripStatus_StopHidData;
    SetOutputReportImpl(buffer, sizeof(buffer));

#ifdef ENABLE_START_HID_DATA
    // WORKAROUND: 前のパケットが上書きされるのを防ぐためにダミーを挟む
    buffer[0] = 0;
    buffer[1] = 0;
    SetOutputReportImpl(buffer, sizeof(buffer));
#endif
}

void UsbHidDriver::SendGetFwVersion() NN_NOEXCEPT
{
    uint8_t buffer[2];
    buffer[0] = Report_ExtGripOut::Id;
    buffer[1] = ExtGripStatus_GetVersion;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::SendGotoFwUpdateMode() NN_NOEXCEPT
{
    uint8_t buffer[2];
    buffer[0] = Report_ExtGripSys::Id;
    buffer[1] = ExtGripSysStatus_GotoFwUpdateMode;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::SendReboot() NN_NOEXCEPT
{
    uint8_t buffer[2];
    buffer[0] = Report_ExtGripSys::Id;
    buffer[1] = ExtGripSysStatus_Reboot;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::SetDummyHidOutputReport() NN_NOEXCEPT
{
    uint8_t buffer[Report_BasicIn::Size] = { 0 };
    buffer[0] = Report_BasicIn::Id;
    SetInputReportImpl(buffer, sizeof(buffer));
}

void UsbHidDriver::HandleNxDeviceDetach() NN_NOEXCEPT
{
    m_IsExtGripOutBusy = false;
    m_IsNxDeviceActivating = false;
    m_IsNxDeviceAvailable = false;
    m_NxHidProtocolActivationCount = 0;
    ResetOutputBuffer();
    m_IsNxDeviceAttached = false;
    m_pInputReportParserFunc = nullptr;
    m_pInputReportParserArg  = nullptr;
    m_IsAvailableKuinaFirmwareVersion = false;

    ::nn::os::SignalLightEvent(m_pConnectionEvent);
}

void UsbHidDriver::ResetOutputBuffer() NN_NOEXCEPT
{
    m_HidOutputReportBufferCount = 0;
    m_HidOutputReportBufferIndex = 0;
    for (auto& buffer : m_HidOutputReportBuffer)
    {
        buffer.size = 0;
    }
}

void UsbHidDriver::ResetDataFormat() NN_NOEXCEPT
{
    uint8_t buffer[Report_CommandOut::Size] = { 0 };
    buffer[ReportByte_ReportId] = Report_CommandOut::Id;
    buffer[CommandOutReportByte_Payload]     = Command_SetDataFormat.Id;
    buffer[CommandOutReportByte_Payload + 1] = Report_BasicIn::Id;
    SetOutputReportImpl(buffer, sizeof(buffer));
}

}}} // namespace nn::xcd::detail
