﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Tick.h>

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   指定時間が経過したことを検知する機能を提供するクラスです。
 *          イベント資源を消費せず、ポーリングでタイマー処理を行うために使用します。
 *          内部で排他制御を行っていないため、使用者側で排他する必要があります。(仮)
 */
class TimeCounter
{
    NN_DISALLOW_COPY(TimeCounter);

public:
    TimeCounter() NN_NOEXCEPT :
        m_IsStarted(false),
        m_StartTick(),
        m_Duration()
    {
    }

    ~TimeCounter() NN_NOEXCEPT {}

    /**
     * @brief   経過時間の計測を開始します。
     *
     * @param[in]   duration    満了までの時間
     */
    void Start(const nn::TimeSpan& duration) NN_NOEXCEPT
    {
        m_StartTick = nn::os::GetSystemTick();
        m_Duration  = duration;
        m_IsStarted = true;
    }

    /**
     * @brief   経過時間の計測を終了します。
     */
    void Stop() NN_NOEXCEPT
    {
        m_IsStarted = false;
    }

    /**
     * @brief   カウントを開始しているかどうか返します。
     */
    bool IsStarted() const NN_NOEXCEPT
    {
        return m_IsStarted;
    }

    /**
     * @brief   指定した時間が経過したか判定します。
     *
     * @retval  true    指定した時間が経過した
     * @retval  false   指定した時間が経過していない、または計測が終了している
     */
    bool IsExpired() const NN_NOEXCEPT
    {
        if (!m_IsStarted)
        {
            return false;
        }

        auto diffTime = (nn::os::GetSystemTick() - m_StartTick).ToTimeSpan();
        return diffTime >= m_Duration;
    }

private:
    bool            m_IsStarted;    //!< 開始済みフラグ
    nn::os::Tick    m_StartTick;    //!< 開始時刻を表す Tick
    nn::TimeSpan    m_Duration;     //!< タイマー満了までの期間
};

}}}  // nn::xcd::detail
