﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_Result.h>
#include "../xcd_ReportTypes.h"
#include "xcd_TeraUpdaterLegacy.h"
#include "xcd_TeraCommon.h"

namespace
{

// 起動処理中の進捗率
const uint16_t ProgressRateBoot = 0;

// ROM 消去開始時の進捗率
const uint16_t ProgressRateErase = 1;

// 書き込み開始時の進捗率
const uint16_t ProgressRateWriting = 2;

// CRC、ブロック数を含めて送出完了したときの進捗率
const uint16_t ProgressRateSendFinish = 98;

// 書き込み中に変動する進捗率
const uint16_t ProgressRateRangeForWrite = ProgressRateSendFinish - ProgressRateWriting;

// すべての書き込みが終了したときの進捗率
const uint16_t ProgressRateWriteFinish = 99;

// シーケンス完了時の進捗率
const uint16_t ProgressRateComplete = 100;

// 連続送信可能なコマンドの数
const int SendableCommandCountMax = 6;

// Queue empty 検知までのタイムアウト
const auto QueueEmptyTimeout = nn::TimeSpan::FromSeconds(30);

}  // anonymous

namespace nn { namespace xcd { namespace detail {

void TeraUpdaterLegacy::Activate(DeviceType deviceType, PhaseChangeCallbackType callback, void* pCallbackArgument) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);
    NN_SDK_REQUIRES_NOT_NULL(pCallbackArgument);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_IsActive)
    {
        return;
    }

    ClearInternalState();
    m_Phase               = UpdatePhaseLegacy::BootPrepare;
    m_Type                = deviceType;
    m_PhaseChangeCallback = callback;
    m_pCallbackArgument   = pCallbackArgument;
    m_ProgressRate        = 0;
    m_IsActive            = true;
}

void TeraUpdaterLegacy::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActive)
    {
        return;
    }

    ClearInternalState();
    m_IsActive = false;
}

void TeraUpdaterLegacy::ClearInternalState() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 完了後に進捗確認を行えるように、進捗状況は初期化しない
    m_Phase               = UpdatePhaseLegacy::Stopped;
    m_EraseState          = EraseState::None;
    m_Type                = DeviceType_Unknown;
    m_PhaseChangeCallback = nullptr;
    m_pCallbackArgument   = nullptr;
    m_ReceivedInput.Clear();
    m_OutputData.Clear();
    m_SentCommandCount  = 0;
    m_TotalCommandCount = 0;
    m_RestartCounter.Stop();
    m_TimeoutCounter.Stop();
}

nn::Result TeraUpdaterLegacy::SetFirmwareImage(
    const FirmwareImage& image,
    UpdateMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(mode == UpdateMode::Normal || mode == UpdateMode::Full);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    NN_RESULT_DO(m_Stream.Setup(image));

    // イメージの正当性チェック
    McuFirmwareImageInfo imageInfo;
    m_Stream.GetImageInfo(&imageInfo);

    switch (m_Type)
    {
    case DeviceType_Right:
        {
            NN_RESULT_THROW_UNLESS(
                imageInfo.type == McuFirmwareImageType_JoyRight,
                nn::xcd::ResultMcuInvalidFirmwareImage());
        }
        break;
    case DeviceType_FullKey:
        {
            NN_RESULT_THROW_UNLESS(
                imageInfo.type == McuFirmwareImageType_FullKey,
                nn::xcd::ResultMcuInvalidFirmwareImage());
        }
        break;
    default:
        NN_RESULT_THROW(nn::xcd::ResultMcuInvalidFirmwareImage());
    }

    // 予約領域は All 0x00
    NN_RESULT_THROW_UNLESS(
        detail::IsAllZero(imageInfo._reserve, sizeof(imageInfo._reserve)),
        nn::xcd::ResultMcuInvalidFirmwareImage());

    m_Mode  = mode;
    m_Crc32 = detail::Crc32InitialValue;

    NN_RESULT_SUCCESS;
}

void TeraUpdaterLegacy::GetRomFooter(RomFooter* pOutRomFooter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRomFooter);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // IAP 以外を更新ブロック数の計算対象とする
    uint32_t skipBytes;
    switch (m_Mode)
    {
    case UpdateMode::Normal:
        {
            // NFC Chip FW から書き込むので全域が対象
            skipBytes = 0;
        }
        break;
    case UpdateMode::Full:
        {
            // IAP 分のサイズをスキップ
            skipBytes = AddressOffsetNfcChipFirmware - AddressOffsetIap;
            NN_SDK_REQUIRES_GREATER(m_Stream.GetImageSize(), skipBytes);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 更新ブロック数は -1 した値を使用
    auto customerCodeSize = m_Stream.GetImageSize() - skipBytes;
    auto blockNumber = (customerCodeSize + McuUpdateDataSizeMax - 1) / McuUpdateDataSizeMax;

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("CRC32 = %08X\n", m_Crc32);

    // 更新失敗エミュレーション時は CRC を反転して Verify を失敗させる
    nn::Bit32 crc32 =
        g_DebugOptions.Test<DebugOption::UpdateFailureEmulation>() ?
        ~m_Crc32 :
        m_Crc32;

    // CRC32 と更新ブロック数はリトルエンディアンで格納
    pOutRomFooter->value[0] = static_cast<uint8_t>(crc32 & 0xFF);
    pOutRomFooter->value[1] = static_cast<uint8_t>((crc32 >> 8) & 0xFF);
    pOutRomFooter->value[2] = static_cast<uint8_t>((crc32 >> 16) & 0xFF);
    pOutRomFooter->value[3] = static_cast<uint8_t>((crc32 >> 24) & 0xFF);

    pOutRomFooter->value[4] = static_cast<uint8_t>(blockNumber & 0xFF);
    pOutRomFooter->value[5] = static_cast<uint8_t>((blockNumber >> 8) & 0xFF);
}

nn::Result TeraUpdaterLegacy::Abort() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    // アップデート動作中なら強制的にエラーにする
    if (m_Phase != UpdatePhaseLegacy::Stopped &&
        m_Phase != UpdatePhaseLegacy::BootPrepare &&
        m_Phase != UpdatePhaseLegacy::Error)
    {
        SetPhase(UpdatePhaseLegacy::Error);
        NotifyPhaseChanged();
    }

    NN_RESULT_SUCCESS;
}

bool TeraUpdaterLegacy::IsAborted() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_Phase == UpdatePhaseLegacy::Error;
}

void TeraUpdaterLegacy::SetPhase(UpdatePhaseLegacy phase) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_Phase = phase;
}

void TeraUpdaterLegacy::NotifyPhaseChanged() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES_NOT_NULL(m_PhaseChangeCallback);

    // Legacy phase を現行の phase に変換
    UpdatePhase phase;
    switch (m_Phase)
    {
    case UpdatePhaseLegacy::Stopped:
        phase = UpdatePhase::Stopped;
        break;

    case UpdatePhaseLegacy::BootPrepare:
        phase = UpdatePhase::BootPrepare;
        break;

    case UpdatePhaseLegacy::Boot:
        phase = UpdatePhase::Boot;
        break;

    case UpdatePhaseLegacy::BootFinish:
        phase = UpdatePhase::BootFinish;
        break;

    case UpdatePhaseLegacy::InvalidateSendCommand:
    case UpdatePhaseLegacy::InvalidateSendAddress:
    case UpdatePhaseLegacy::InvalidateSendData:
        phase = UpdatePhase::Invalidate;
        break;

    case UpdatePhaseLegacy::EraseSendCommand:
    case UpdatePhaseLegacy::EraseSendSectorCount:
    case UpdatePhaseLegacy::EraseSendSectors:
        phase = UpdatePhase::Erasing;
        break;

    case UpdatePhaseLegacy::EraseFinish:
        phase = UpdatePhase::EraseFinish;
        break;

    case UpdatePhaseLegacy::WriteSendCommand:
    case UpdatePhaseLegacy::WriteSendAddress:
    case UpdatePhaseLegacy::WriteSendData:
        phase = UpdatePhase::Write;
        break;

    case UpdatePhaseLegacy::FinalizeSendCommand:
    case UpdatePhaseLegacy::FinalizeSendAddress:
    case UpdatePhaseLegacy::FinalizeSendData:
        phase = UpdatePhase::Finalize;
        break;

    case UpdatePhaseLegacy::FinalizeWaiting:
        phase = UpdatePhase::Finalizing;
        break;

    case UpdatePhaseLegacy::FinalizeFinish:
        phase = UpdatePhase::FinalizeFinish;
        break;

    case UpdatePhaseLegacy::Error:
    default:
        phase = UpdatePhase::Error;
        break;
    }

    m_PhaseChangeCallback(phase, m_pCallbackArgument);
}

nn::Result TeraUpdaterLegacy::GetState(McuUpdateStateInfo* pOutStateInfo) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStateInfo);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    pOutStateInfo->progress = static_cast<uint16_t>(m_ProgressRate);

    if (!m_IsActive)
    {
        // 開始していないときは常に終了状態
        pOutStateInfo->state = McuUpdateState_End;
        NN_RESULT_SUCCESS;
    }

    // アップデートの進捗に応じてステートを決める
    switch (m_Phase)
    {
    case UpdatePhaseLegacy::BootPrepare:
    case UpdatePhaseLegacy::Boot:
    case UpdatePhaseLegacy::BootFinish:
        {
            pOutStateInfo->state = McuUpdateState_Boot;
        }
        break;
    case UpdatePhaseLegacy::InvalidateSendCommand:    // 起動フラグ無効化は消去扱い
    case UpdatePhaseLegacy::InvalidateSendAddress:
    case UpdatePhaseLegacy::InvalidateSendData:
    case UpdatePhaseLegacy::EraseSendCommand:
    case UpdatePhaseLegacy::EraseSendSectorCount:
    case UpdatePhaseLegacy::EraseSendSectors:
    case UpdatePhaseLegacy::EraseFinish:
        {
            pOutStateInfo->state = McuUpdateState_RomErasing;
        }
        break;
    case UpdatePhaseLegacy::WriteSendCommand:
    case UpdatePhaseLegacy::WriteSendAddress:
    case UpdatePhaseLegacy::WriteSendData:
    case UpdatePhaseLegacy::FinalizeSendCommand:
    case UpdatePhaseLegacy::FinalizeSendAddress:
    case UpdatePhaseLegacy::FinalizeSendData:
    case UpdatePhaseLegacy::FinalizeWaiting:
        {
            pOutStateInfo->state =
                (m_EraseState == EraseState::Finished) ?
                McuUpdateState_Writing :
                McuUpdateState_RomErasing;
        }
        break;
    case UpdatePhaseLegacy::Stopped:
        {
            pOutStateInfo->state = McuUpdateState_End;
        }
        break;
    case UpdatePhaseLegacy::FinalizeFinish:
    case UpdatePhaseLegacy::Error:
    default:
        {
            // 完了もしくは不正終了なのでリブートする
            // (更新の成否はリブート後に確認するため、この段階での通知は不要)
            pOutStateInfo->state = McuUpdateState_Reboot;
        }
        break;
    }

    NN_RESULT_SUCCESS;
}

void TeraUpdaterLegacy::ParseMcuUpdateInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_UNUSED(sampleNumber);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActive)
    {
        return;
    }

    std::memcpy(&m_ReceivedInput, pBuffer, std::min(size, sizeof(m_ReceivedInput)));

    auto previousPhase = m_Phase;
    ProcessMcuUpdateInput();

    if (previousPhase != m_Phase)
    {
        NotifyPhaseChanged();
    }
}

size_t TeraUpdaterLegacy::GetMcuUpdateOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsActive || IsAborted() || m_ReceivedInput.HasError())
    {
        return 0;
    }

    // コマンドを規定数以上は送らない
    if (m_SentCommandCount >= SendableCommandCountMax)
    {
        return 0;
    }

    if (!m_OutputData.HasData())
    {
        return 0;
    }

    auto payloadSize = std::min(m_OutputData.payloadSize, size);
    std::memcpy(pOutValue, m_OutputData.payload, payloadSize);

    m_OutputData.Clear();
    m_SentCommandCount++;
    m_TotalCommandCount++;

    return payloadSize;
}

void TeraUpdaterLegacy::ProcessMcuUpdateInput() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (IsAborted())
    {
        // 中断された場合は何もしない
        return;
    }
    else if (m_Phase == UpdatePhaseLegacy::BootPrepare)
    {
        // 準備中は何もしない
        return;
    }
    else if (m_ReceivedInput.HasError() || m_TimeoutCounter.IsExpired())
    {
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s occurred\n",
            m_ReceivedInput.HasError() ? "Error" : "Timeout");
        SetPhase(UpdatePhaseLegacy::Error);
        m_TimeoutCounter.Stop();
        NotifyPhaseChanged();
        return;
    }
    else if (m_ReceivedInput.IsQueueFull())
    {
        // キューが埋まっているときは何も送らない
        return;
    }
    else if (m_OutputData.HasData())
    {
        // 送信待ちデータがあるので何もしない
        return;
    }

    if (m_SentCommandCount >= SendableCommandCountMax)
    {
        ProcessMaximumSent();
    }
    else if (m_RestartCounter.IsStarted() && !m_RestartCounter.IsExpired())
    {
        // 再開時刻に到達するまで何もしない
    }
    else
    {
        Proceed();
    }
}

void TeraUpdaterLegacy::ProcessWaitErase() NN_NOEXCEPT
{
    switch (m_EraseState)
    {
    case EraseState::Started:
        {
            NN_XCD_TERA_UPDATER_LOG_VERBOSE("Waiting not empty\n");
            m_TimeoutCounter.Start(QueueEmptyTimeout);
            m_EraseState = EraseState::WaitingQueueNotEmpty;
        }
        return;
    case EraseState::WaitingQueueNotEmpty:
        if (!m_ReceivedInput.IsQueueEmpty() ||
            m_TimeoutCounter.IsExpired())
        {
            if (m_TimeoutCounter.IsExpired())
            {
                NN_XCD_TERA_UPDATER_LOG_VERBOSE("Waiting not empty is timed out\n");
            }
            else
            {
                NN_XCD_TERA_UPDATER_LOG_VERBOSE("Queue not empty\n");
            }
            m_EraseState = EraseState::WaitingQueueEmpty;
        }
        return;
    default:
        // 何もしない
        break;
    }
}

void TeraUpdaterLegacy::ProcessMaximumSent() NN_NOEXCEPT
{
    if (m_EraseState == EraseState::Started ||
        m_EraseState == EraseState::WaitingQueueNotEmpty)
    {
        ProcessWaitErase();
        return;
    }

    // キューが空になったら送信許可
    if (m_ReceivedInput.IsQueueEmpty())
    {
        NN_XCD_TERA_UPDATER_LOG_VERBOSE("Queue empty\n");
        m_SentCommandCount = 0;
        m_TimeoutCounter.Stop();

        if (m_EraseState == EraseState::WaitingQueueEmpty)
        {
            m_EraseState = EraseState::Finished;
        }
    }
}

void TeraUpdaterLegacy::Proceed() NN_NOEXCEPT
{
    switch (m_Phase)
    {
    case UpdatePhaseLegacy::Boot:
        {
            ProceedBootPhase();
        }
        break;
    case UpdatePhaseLegacy::InvalidateSendCommand:
    case UpdatePhaseLegacy::InvalidateSendAddress:
    case UpdatePhaseLegacy::InvalidateSendData:
        {
            ProceedInvalidatePhase();
        }
        break;
    case UpdatePhaseLegacy::EraseSendCommand:
    case UpdatePhaseLegacy::EraseSendSectorCount:
    case UpdatePhaseLegacy::EraseSendSectors:
        {
            ProceedErasePhase();
        }
        break;
    case UpdatePhaseLegacy::WriteSendCommand:
    case UpdatePhaseLegacy::WriteSendAddress:
    case UpdatePhaseLegacy::WriteSendData:
        {
            ProceedWritePhase();
        }
        break;
    case UpdatePhaseLegacy::FinalizeSendCommand:
    case UpdatePhaseLegacy::FinalizeSendAddress:
    case UpdatePhaseLegacy::FinalizeSendData:
    case UpdatePhaseLegacy::FinalizeWaiting:
        {
            ProceedFinalizePhase();
        }
        break;
    default:
        // ここに入るときはコマンド完了状態
        m_TimeoutCounter.Stop();
        return;
    }

    m_TimeoutCounter.Start(QueueEmptyTimeout);
}

void TeraUpdaterLegacy::ProceedBootPhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    switch (m_Phase)
    {
    case UpdatePhaseLegacy::Boot:
        {
            SetPhase(UpdatePhaseLegacy::BootFinish);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void TeraUpdaterLegacy::ProceedInvalidatePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    switch (m_Phase)
    {
    case UpdatePhaseLegacy::InvalidateSendCommand:
        {
            // 起動フラグアドレスの送信
            auto startAddress = FlashRomStartAddress + AddressOffsetActivationFlag;
            uint8_t sendData[5];
            SetBigEndianValue(sendData, sizeof(uint32_t), startAddress);
            sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::InvalidateSendAddress);
        }
        break;
    case UpdatePhaseLegacy::InvalidateSendAddress:
        {
            // 起動フラグ消去データの送信
            uint8_t sendData[ActivationFlagSize + 2] = {};
            sendData[0] = static_cast<uint8_t>(ActivationFlagSize - 1);
            sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::InvalidateSendData);
        }
        break;
    case UpdatePhaseLegacy::InvalidateSendData:
        {
            // ROM の消去を開始
            uint8_t sendData[] = { SofByte, EraseCommand, EraseCommand ^ 0xFF };
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::EraseSendCommand);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void TeraUpdaterLegacy::ProceedErasePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    // 消去対象セクタの情報
    const int* EraseSectors;
    int eraseSectorCount;
    switch (m_Mode)
    {
    case UpdateMode::Normal:
        {
            EraseSectors = EraseSectorsCustomer;
            eraseSectorCount = sizeof(EraseSectorsCustomer) / sizeof(EraseSectorsCustomer[0]);
        }
        break;
    case UpdateMode::Full:
        {
            EraseSectors = EraseSectorsFull;
            eraseSectorCount = sizeof(EraseSectorsFull) / sizeof(EraseSectorsFull[0]);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    switch (m_Phase)
    {
    case UpdatePhaseLegacy::EraseSendCommand:
        {
            // 送信するセクタ数は -1 する
            int sectorCountForSend = eraseSectorCount - 1;

            // 消去セクタ数の送信
            uint8_t sendData[3];
            SetBigEndianValue(sendData, 2, sectorCountForSend);
            sendData[2] = CalcCheckSum(sendData, sizeof(sendData) - 1);
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::EraseSendSectorCount);
        }
        break;
    case UpdatePhaseLegacy::EraseSendSectorCount:
        {
            // 消去セクタ番号の送信
            size_t sendDataSize = eraseSectorCount * 2 + 1;
            uint8_t sendData[EraseSectorCountMax * 2 + 1];
            for (int i = 0; i < eraseSectorCount; i++)
            {
                // 各セクタ番号は 16bit Big endian で格納
                SetBigEndianValue(sendData + i * 2, 2, EraseSectors[i]);
            }
            sendData[sendDataSize - 1] = CalcCheckSum(sendData, sendDataSize - 1);
            m_OutputData.SetPayload(sendData, sendDataSize);

            m_EraseState = EraseState::Started;
            SetPhase(UpdatePhaseLegacy::EraseSendSectors);
        }
        break;
    case UpdatePhaseLegacy::EraseSendSectors:
        {
            if (m_ReceivedInput.IsQueueEmpty())
            {
                SetPhase(UpdatePhaseLegacy::EraseFinish);
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void TeraUpdaterLegacy::ProceedWritePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    // 書き込み先アドレスの送信
    auto startAddress = FlashRomStartAddress;
    switch (m_Mode)
    {
    case UpdateMode::Normal:
        {
            startAddress += AddressOffsetNfcChipFirmware;
        }
        break;
    case UpdateMode::Full:
        {
            startAddress += AddressOffsetIap;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto nextAddress = static_cast<uint32_t>(startAddress + m_Stream.GetTotalReadBytes());

    switch (m_Phase)
    {
    case UpdatePhaseLegacy::WriteSendCommand:
        {
            uint8_t sendData[5];
            SetBigEndianValue(sendData, sizeof(uint32_t), nextAddress);
            sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::WriteSendAddress);
        }
        break;
    case UpdatePhaseLegacy::WriteSendAddress:
        {
            // 書き込むデータの送信
            uint8_t sendData[McuUpdateDataSizeMax + 2];
            size_t dataBytes;
            auto result = FetchNextWriteData(&dataBytes, sendData, sizeof(sendData), nextAddress);
            if (result.IsSuccess())
            {
                SetPhase(UpdatePhaseLegacy::WriteSendData);
            }
            else
            {
                // FW イメージ読み込み中にエラー
                SetPhase(UpdatePhaseLegacy::Error);
            }
        }
        break;
    case UpdatePhaseLegacy::WriteSendData:
        {
            uint8_t sendData[] = { SofByte, WriteCommand, WriteCommand ^ 0xFF };
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            m_ProgressRate =
                ProgressRateWriting +
                std::min<uint16_t>(
                    static_cast<uint16_t>(m_Stream.GetTotalReadBytes() * ProgressRateRangeForWrite / m_Stream.GetImageSize()),
                    ProgressRateRangeForWrite);

            if (m_Stream.IsEof())
            {
                // 全データの書き込みが完了したのでファイナライズへ
                SetPhase(UpdatePhaseLegacy::FinalizeSendCommand);
            }
            else
            {
                // まだデータが残っているので次のデータ送信へ
                SetPhase(UpdatePhaseLegacy::WriteSendCommand);
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result TeraUpdaterLegacy::FetchNextWriteData(
    size_t* pOutBytes,
    uint8_t* pOutBuffer,
    size_t bufferSize,
    uint32_t nextAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBytes);
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);

    // 実装ミス以外で引っかかることはないが、これを満たさないと配列の範囲外アクセスになるので abort
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSize, McuUpdateDataSizeMax + 2);

    size_t dataBytes;
    m_Stream.Read(&pOutBuffer[1], &dataBytes, McuUpdateDataSizeMax);

    *pOutBytes = dataBytes;
    if (dataBytes == 0)
    {
        NN_RESULT_SUCCESS;
    }

    // 先頭バイトは有効なデータサイズ - 1
    pOutBuffer[0] = static_cast<uint8_t>(dataBytes - 1);

    // 送信サイズに満たない分は、CRC32 計算のためにパディングデータで埋める
    auto remainSize = McuUpdateDataSizeMax - dataBytes;
    if (remainSize > 0)
    {
        std::memset(&pOutBuffer[dataBytes + 1], PaddingByte, remainSize);
    }

    // 書き込み先が IAP 外なら CRC32 を更新
    if (nextAddress >= FlashRomStartAddress + AddressOffsetNfcChipFirmware)
    {
        m_Crc32 = detail::CalcCrc32(&pOutBuffer[1], McuUpdateDataSizeMax, m_Crc32);
    }

    if (dataBytes % 2 != 0)
    {
        // 送信サイズが奇数のときはパディングを入れて 1 バイト余分に送る
        dataBytes++;
    }

    // チェックサムを付与して送信予約
    pOutBuffer[dataBytes + 1] = CalcCheckSum(pOutBuffer, dataBytes + 1);
    m_OutputData.SetPayload(pOutBuffer, dataBytes + 2);

    NN_RESULT_SUCCESS;
}

void TeraUpdaterLegacy::ProceedFinalizePhase() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s %d\n", NN_CURRENT_FUNCTION_NAME, m_Phase);

    switch (m_Phase)
    {
    case UpdatePhaseLegacy::FinalizeSendCommand:
        {
            // 書き込み先アドレスの送信
            auto startAddress = FlashRomStartAddress + AddressOffsetCrc32;
            uint8_t sendData[5];
            SetBigEndianValue(sendData, sizeof(uint32_t), startAddress);
            sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::FinalizeSendAddress);
        }
        break;
    case UpdatePhaseLegacy::FinalizeSendAddress:
        {
            // CRC32 + データブロック数の送信
            RomFooter romFooter;
            GetRomFooter(&romFooter);

            uint8_t sendData[McuFirmwareFooterSize + 2];
            sendData[0] = static_cast<uint8_t>(McuFirmwareFooterSize - 1);
            std::memcpy(sendData + 1, romFooter.value, McuFirmwareFooterSize);
            sendData[sizeof(sendData) - 1] = CalcCheckSum(sendData, sizeof(sendData) - 1);
            m_OutputData.SetPayload(sendData, sizeof(sendData));

            SetPhase(UpdatePhaseLegacy::FinalizeSendData);
        }
        break;
    case UpdatePhaseLegacy::FinalizeSendData:
        {
            m_ProgressRate = ProgressRateSendFinish;

            // 最終データブロックの書き込みが終わるまで待つ (時間は暫定)
            const auto WaitTimeForFinalData = nn::TimeSpan::FromMilliSeconds(1000);
            m_RestartCounter.Start(WaitTimeForFinalData);
            NN_XCD_TERA_UPDATER_LOG_VERBOSE("Waiting for finish...\n");
            SetPhase(UpdatePhaseLegacy::FinalizeWaiting);
        }
        break;
    case UpdatePhaseLegacy::FinalizeWaiting:
        {
            if (m_ReceivedInput.IsQueueEmpty())
            {
                m_ProgressRate = ProgressRateWriteFinish;
                NN_XCD_TERA_UPDATER_LOG_VERBOSE("Finalize complete!!!\n");
                SetPhase(UpdatePhaseLegacy::FinalizeFinish);
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void TeraUpdaterLegacy::StartSyncWithBootLoader() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    if (IsAborted())
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(m_Phase, UpdatePhaseLegacy::BootPrepare);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    m_ProgressRate = ProgressRateBoot;
    SetPhase(UpdatePhaseLegacy::Boot);

    uint8_t sendData[] = { SofByte };
    m_OutputData.SetPayload(sendData, sizeof(sendData));

    m_TotalCommandCount = 0;
    m_EraseState        = EraseState::None;
    m_TimeoutCounter.Start(QueueEmptyTimeout);
}

void TeraUpdaterLegacy::StartEraseRom() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    if (IsAborted())
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(m_Phase, UpdatePhaseLegacy::BootFinish);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    // 起動フラグの消去から開始する
    m_ProgressRate = ProgressRateErase;
    SetPhase(UpdatePhaseLegacy::InvalidateSendCommand);

    uint8_t sendData[] = { SofByte, WriteCommand, WriteCommand ^ 0xFF };
    m_OutputData.SetPayload(sendData, sizeof(sendData));

    m_TimeoutCounter.Start(QueueEmptyTimeout);
}

void TeraUpdaterLegacy::StartWriteRom() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);

    if (IsAborted())
    {
        return;
    }

    NN_SDK_REQUIRES_EQUAL(m_Phase, UpdatePhaseLegacy::EraseFinish);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    m_ProgressRate = ProgressRateWriting;
    SetPhase(UpdatePhaseLegacy::WriteSendCommand);

    uint8_t sendData[] = { SofByte, WriteCommand, WriteCommand ^ 0xFF };
    m_OutputData.SetPayload(sendData, sizeof(sendData));

    m_TimeoutCounter.Start(QueueEmptyTimeout);
}

void TeraUpdaterLegacy::Finish() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_SDK_REQUIRES(m_IsActive);
    NN_SDK_REQUIRES(
        m_Phase == UpdatePhaseLegacy::FinalizeFinish ||
        m_Phase == UpdatePhaseLegacy::Error);

    NN_XCD_TERA_UPDATER_LOG_VERBOSE("%s\n", NN_CURRENT_FUNCTION_NAME);

    // 最後まで書き込みが完了していれば 100% にする
    if (m_ProgressRate == ProgressRateWriteFinish)
    {
        m_ProgressRate = ProgressRateComplete;
    }

    NN_XCD_TERA_UPDATER_LOG("Total sent command: %d\n", m_TotalCommandCount);

    SetPhase(UpdatePhaseLegacy::Stopped);
}

}}} // namespace nn::xcd::detail
