﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_NfcTypes.h>
#include "xcd_TeraFirmwareVersion.h"

namespace
{

// Customer code 破損を確定するまでの判定回数
const int CorruptionCheckFixCount = 3;

}  // anonymous

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   バージョンを取得します。
 */
bool TeraFirmwareVersion::GetVersion(McuVersionData* pOutVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    if (!IsValid())
    {
        return false;
    }

    *pOutVersion = m_Version;

    return true;
}

/**
 * @brief   バージョンを取得します。 (NFC 向け)
 */
bool TeraFirmwareVersion::GetVersion(McuVersionDataForNfc* pOutVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    if (!IsValid())
    {
        return false;
    }

    pOutVersion->major      = m_Version.major;
    pOutVersion->minor      = m_Version.minor;
    pOutVersion->deviceType = m_Version.deviceType;
    std::memset(pOutVersion->_reserve, 0, sizeof(pOutVersion->_reserve));

    return true;
}

/**
 * @brief   バージョン情報を無効化します。
 */
void TeraFirmwareVersion::Invalidate() NN_NOEXCEPT
{
    m_HasValidVersion      = false;
    m_CorruptionCheckCount = 0;

    m_Version.isCorrupted    = false;
    m_Version.isIapCorrupted = false;
    m_Version.major          = 0;
    m_Version.minor          = 0;
    std::memset(m_Version._reserve, 0, sizeof(m_Version._reserve));
}

/**
 * @brief   Customer code のバージョンを設定します。
 */
void TeraFirmwareVersion::SetVersion(uint16_t major, uint16_t minor) NN_NOEXCEPT
{
    Invalidate();

    m_Version.isCorrupted    = false;
    m_Version.isIapCorrupted = false;
    m_Version.major          = major;
    m_Version.minor          = minor;

    m_HasValidVersion = true;
}

/**
 * @brief   Cusromter code 破損判定カウンタを進めます。
 */
void TeraFirmwareVersion::ProceedCustomerCodeCorruptionCount() NN_NOEXCEPT
{
    if (IsValid())
    {
        return;
    }

    m_CorruptionCheckCount++;

    if (m_CorruptionCheckCount >= CorruptionCheckFixCount)
    {
        SetCustomerCodeCorruption();
    }
}

/**
 * @brief   Cusromter code 破損判定カウンタを初期化します。
 */
void TeraFirmwareVersion::ClearCustomerCodeCorruptionCount() NN_NOEXCEPT
{
    m_CorruptionCheckCount = 0;
}

/**
 * @brief   Cusromter code 破損状態に設定します。
 */
void TeraFirmwareVersion::SetCustomerCodeCorruption() NN_NOEXCEPT
{
    Invalidate();

    m_Version.isCorrupted    = true;
    m_Version.isIapCorrupted = false;

    m_HasValidVersion = true;
}

/**
 * @brief   IAP 破損状態に設定します。
 */
void TeraFirmwareVersion::SetIapCorruption() NN_NOEXCEPT
{
    Invalidate();

    m_Version.isCorrupted    = true;
    m_Version.isIapCorrupted = true;

    m_HasValidVersion = true;
}

}}}
