﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Macro.h>  // type_traits 対応判定のため、先に必要
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)
    #include <type_traits>
#endif
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>

#include "../xcd_DeviceHandler.h"
#include "xcd_TeraCommon.h"

//#define VERBOSE

namespace
{

// CRC32 の計算に使用するテーブル
const nn::Bit32 Crc32Table[] =
{
    0x00000000, 0x04c11db7, 0x09823b6e, 0x0d4326d9,
    0x130476dc, 0x17c56b6b, 0x1a864db2, 0x1e475005,
    0x2608edb8, 0x22c9f00f, 0x2f8ad6d6, 0x2b4bcb61,
    0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd,
    0x4c11db70, 0x48d0c6c7, 0x4593e01e, 0x4152fda9,
    0x5f15adac, 0x5bd4b01b, 0x569796c2, 0x52568b75,
    0x6a1936c8, 0x6ed82b7f, 0x639b0da6, 0x675a1011,
    0x791d4014, 0x7ddc5da3, 0x709f7b7a, 0x745e66cd,
    0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
    0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5,
    0xbe2b5b58, 0xbaea46ef, 0xb7a96036, 0xb3687d81,
    0xad2f2d84, 0xa9ee3033, 0xa4ad16ea, 0xa06c0b5d,
    0xd4326d90, 0xd0f37027, 0xddb056fe, 0xd9714b49,
    0xc7361b4c, 0xc3f706fb, 0xceb42022, 0xca753d95,
    0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1,
    0xe13ef6f4, 0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d,
    0x34867077, 0x30476dc0, 0x3d044b19, 0x39c556ae,
    0x278206ab, 0x23431b1c, 0x2e003dc5, 0x2ac12072,
    0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16,
    0x018aeb13, 0x054bf6a4, 0x0808d07d, 0x0cc9cdca,
    0x7897ab07, 0x7c56b6b0, 0x71159069, 0x75d48dde,
    0x6b93dddb, 0x6f52c06c, 0x6211e6b5, 0x66d0fb02,
    0x5e9f46bf, 0x5a5e5b08, 0x571d7dd1, 0x53dc6066,
    0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
    0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e,
    0xbfa1b04b, 0xbb60adfc, 0xb6238b25, 0xb2e29692,
    0x8aad2b2f, 0x8e6c3698, 0x832f1041, 0x87ee0df6,
    0x99a95df3, 0x9d684044, 0x902b669d, 0x94ea7b2a,
    0xe0b41de7, 0xe4750050, 0xe9362689, 0xedf73b3e,
    0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2,
    0xc6bcf05f, 0xc27dede8, 0xcf3ecb31, 0xcbffd686,
    0xd5b88683, 0xd1799b34, 0xdc3abded, 0xd8fba05a,
    0x690ce0ee, 0x6dcdfd59, 0x608edb80, 0x644fc637,
    0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb,
    0x4f040d56, 0x4bc510e1, 0x46863638, 0x42472b8f,
    0x5c007b8a, 0x58c1663d, 0x558240e4, 0x51435d53,
    0x251d3b9e, 0x21dc2629, 0x2c9f00f0, 0x285e1d47,
    0x36194d42, 0x32d850f5, 0x3f9b762c, 0x3b5a6b9b,
    0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
    0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623,
    0xf12f560e, 0xf5ee4bb9, 0xf8ad6d60, 0xfc6c70d7,
    0xe22b20d2, 0xe6ea3d65, 0xeba91bbc, 0xef68060b,
    0xd727bbb6, 0xd3e6a601, 0xdea580d8, 0xda649d6f,
    0xc423cd6a, 0xc0e2d0dd, 0xcda1f604, 0xc960ebb3,
    0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7,
    0xae3afba2, 0xaafbe615, 0xa7b8c0cc, 0xa379dd7b,
    0x9b3660c6, 0x9ff77d71, 0x92b45ba8, 0x9675461f,
    0x8832161a, 0x8cf30bad, 0x81b02d74, 0x857130c3,
    0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640,
    0x4e8ee645, 0x4a4ffbf2, 0x470cdd2b, 0x43cdc09c,
    0x7b827d21, 0x7f436096, 0x7200464f, 0x76c15bf8,
    0x68860bfd, 0x6c47164a, 0x61043093, 0x65c52d24,
    0x119b4be9, 0x155a565e, 0x18197087, 0x1cd86d30,
    0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
    0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088,
    0x2497d08d, 0x2056cd3a, 0x2d15ebe3, 0x29d4f654,
    0xc5a92679, 0xc1683bce, 0xcc2b1d17, 0xc8ea00a0,
    0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb, 0xdbee767c,
    0xe3a1cbc1, 0xe760d676, 0xea23f0af, 0xeee2ed18,
    0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4,
    0x89b8fd09, 0x8d79e0be, 0x803ac667, 0x84fbdbd0,
    0x9abc8bd5, 0x9e7d9662, 0x933eb0bb, 0x97ffad0c,
    0xafb010b1, 0xab710d06, 0xa6322bdf, 0xa2f33668,
    0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4
};

// CRC8 の計算に使用するテーブル
const nn::Bit8 Crc8Table[] =
{
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
};

// CRC 計算用テーブルの要素数
const int Crc32TableLength = 256;
const int Crc8TableLength  = 256;

// CRC テーブルのサイズが適切か検査
NN_STATIC_ASSERT(sizeof(Crc32Table) / sizeof(Crc32Table[0]) == Crc32TableLength);
NN_STATIC_ASSERT(sizeof(Crc8Table)  / sizeof(Crc8Table[0])  == Crc8TableLength);

}

namespace nn { namespace xcd { namespace detail {

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)

// 構造体の POD 判定
NN_STATIC_ASSERT(std::is_pod<BluetoothTransferSize>::value);
NN_STATIC_ASSERT(std::is_pod<PacketModePackForSend>::value);
NN_STATIC_ASSERT(std::is_pod<ReceiveProtocolType>::value);
NN_STATIC_ASSERT(std::is_pod<TeraControlCommandType>::value);
NN_STATIC_ASSERT(std::is_pod<ExtControlCommandType>::value);
NN_STATIC_ASSERT(std::is_pod<CommonCommandType>::value);
NN_STATIC_ASSERT(std::is_pod<CommonResponseType>::value);

#endif  // defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_TYPE_TRAITS)

const BluetoothTransferSize BluetoothTransferSizeList[BluetoothMode_Max] =
{
    {  38, 313 },  // Mode 0
    { 308, 313 },  // Mode 1
    {  38,  34 }   // Mode 2
};

void DumpBinary(const void* pData, size_t dataSize) NN_NOEXCEPT
{
#if defined(NN_DETAIL_ENABLE_SDK_LOG)
    NN_SDK_REQUIRES_NOT_NULL(pData);

    const auto* pBinary = reinterpret_cast<const nn::Bit8*>(pData);
    for (size_t i = 0; i < dataSize; ++i)
    {
        NN_SDK_LOG("%02X ", pBinary[i]);
    }
    NN_SDK_LOG("\n");
#else
    NN_UNUSED(pData);
    NN_UNUSED(dataSize);
#endif  // if defined(NN_DETAIL_ENABLE_SDK_LOG)
}

void DumpBinaryFormatted(const char* pData, size_t dataSize) NN_NOEXCEPT
{
#if defined(NN_DETAIL_ENABLE_SDK_LOG)
    NN_SDK_REQUIRES_NOT_NULL(pData);

    for (size_t i = 0; i < 16; ++i)
    {
        NN_SDK_LOG("%02X ", i);
    }
    NN_SDK_LOG("\n");
    for (size_t i = 0; i < 16; ++i)
    {
        NN_SDK_LOG("---");
    }
    NN_SDK_LOG("\n");

    for (size_t i = 0; i < dataSize; ++i)
    {
        NN_SDK_LOG("%02X ", static_cast<nn::Bit8>(pData[i]));
        if ((i + 1) % 16 == 0)
        {
            NN_SDK_LOG("\n");
        }
    }
    NN_SDK_LOG("\n");
#else
    NN_UNUSED(pData);
    NN_UNUSED(dataSize);
#endif  // if defined(NN_DETAIL_ENABLE_SDK_LOG)
}

char CalcCheckSum(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    if (dataSize == 0)
    {
        return 0;
    }

    const auto* pBinary = reinterpret_cast<const nn::Bit8*>(pData);
    char sum = pBinary[0];
    for (size_t i = 1; i < dataSize; ++i)
    {
        sum ^= pBinary[i];
    }

    return sum;
}

nn::Bit32 CalcCrc32(void* pData, size_t dataSize, nn::Bit32 initialValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES_ALIGNED(dataSize, sizeof(nn::Bit32));

    const auto* pBinary = reinterpret_cast<const nn::Bit8*>(pData);
    auto crc = initialValue;
    for (size_t i = 0; i < dataSize / sizeof(nn::Bit32); ++i)
    {
        size_t offset = i * sizeof(nn::Bit32);
        crc = (crc << 8) ^ Crc32Table[((crc >> 24) ^ pBinary[offset + 3]) & 0xFF];
        crc = (crc << 8) ^ Crc32Table[((crc >> 24) ^ pBinary[offset + 2]) & 0xFF];
        crc = (crc << 8) ^ Crc32Table[((crc >> 24) ^ pBinary[offset + 1]) & 0xFF];
        crc = (crc << 8) ^ Crc32Table[((crc >> 24) ^ pBinary[offset + 0]) & 0xFF];
    }

    return crc;
}

nn::Bit8 CalcCrc8(const void* pData, size_t dataBytes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    const auto* pBinary = reinterpret_cast<const nn::Bit8*>(pData);
    nn::Bit32 r = 0;
    for (size_t i = 0; i < dataBytes; ++i)
    {
        r = Crc8Table[(r ^ pBinary[i]) & 0xFF];
    }

    return static_cast<nn::Bit8>(r);
}

uint8_t CreateMcuPacketModeByte(BluetoothMode bluetoothMode, CommandProtocol protocol) NN_NOEXCEPT
{
    nn::util::BitPack8 mode;
    mode.Set<PacketModePackForSend::BluetoothMode>(static_cast<uint8_t>(bluetoothMode));
    mode.Set<PacketModePackForSend::Protocol>(static_cast<uint8_t>(protocol));
    mode.Set<PacketModePackForSend::Reserved>(0);

    return static_cast<uint8_t>(mode.GetMaskedBits(0xFF));
}

size_t CreateTeraControlNopCommand(uint8_t* pOutCommand, size_t outSize, BluetoothMode mode) NN_NOEXCEPT
{
    NN_UNUSED(outSize);

    NN_SDK_REQUIRES_NOT_NULL(pOutCommand);
    NN_SDK_REQUIRES_LESS(mode, BluetoothMode_Max);

    const auto& TransferSize = BluetoothTransferSizeList[mode];

    // 最低限 State + CRC8 まで入る定義がされていることを保証
    NN_SDK_REQUIRES_GREATER(TransferSize.send, static_cast<size_t>(McuCommonOutputOffsetByte_State + 1));
    NN_SDK_REQUIRES_GREATER(TransferSize.receive, 0ul);
    NN_SDK_REQUIRES_GREATER_EQUAL(outSize, TransferSize.send);

    // TERA_CONTROL の NOP コマンドを生成
    std::memset(pOutCommand, 0, TransferSize.send);
    pOutCommand[McuCommonOutputOffsetByte_ResultCode] =
        CreateMcuPacketModeByte(
            mode,
            CommandProtocol_Common);
    pOutCommand[McuCommonOutputOffsetByte_Command] = CommonCommandId_TeraControl;
    pOutCommand[McuCommonOutputOffsetByte_State]   = InternalMcuState_Nop;
    pOutCommand[TransferSize.send - 1] =
        CalcCrc8(pOutCommand + 1, TransferSize.send - 2);

    return TransferSize.send;
}

size_t CreateTeraExtControlCommand(uint8_t* pOutCommand, size_t outSize, BluetoothMode mode, bool isOn) NN_NOEXCEPT
{
    NN_UNUSED(outSize);

    NN_SDK_REQUIRES_NOT_NULL(pOutCommand);
    NN_SDK_REQUIRES_LESS(mode, BluetoothMode_Max);

    const auto& TransferSize = BluetoothTransferSizeList[mode];

    // 最低限 State + CRC8 まで入る定義がされていることを保証
    NN_SDK_REQUIRES_GREATER(TransferSize.send, static_cast<size_t>(McuCommonOutputOffsetByte_State + 1));
    NN_SDK_REQUIRES_GREATER(TransferSize.receive, 0ul);
    NN_SDK_REQUIRES_GREATER_EQUAL(outSize, TransferSize.send);

    // EXT_CONTROL コマンドを生成
    std::memset(pOutCommand, 0, TransferSize.send);
    pOutCommand[McuExtControlOutputOffsetByte_Header] =
        CreateMcuPacketModeByte(
            mode,
            CommandProtocol_Common);
    pOutCommand[McuExtControlOutputOffsetByte_Command] = CommonCommandId_ExtControl;
    pOutCommand[McuExtControlOutputOffsetByte_Setting] = isOn ? 0x01 : 0x00;
    pOutCommand[TransferSize.send - 1] =
        CalcCrc8(pOutCommand + 1, TransferSize.send - 2);

    return TransferSize.send;
}

McuState ConvertInternalMcuStateToExternal(InternalMcuState mcuState) NN_NOEXCEPT
{
    switch (mcuState)
    {
    case InternalMcuState_Idle:         return McuState_Idle;
    case InternalMcuState_Standby:      return McuState_Standby;
    case InternalMcuState_Background:   return McuState_Background;
    case InternalMcuState_Nfc:          return McuState_Nfc;
    case InternalMcuState_Ir:           return McuState_Ir;
    case InternalMcuState_Initializing: return McuState_Initializing;
    default:                            return McuState_Unknown;
    }
}

InternalMcuState ConvertMcuStateToInternal(nn::xcd::McuState mcuState) NN_NOEXCEPT
{
    switch (mcuState)
    {
    case McuState_Idle:             return InternalMcuState_Idle;
    case McuState_Standby:          return InternalMcuState_Standby;
    case McuState_Background:       return InternalMcuState_Background;
    case McuState_Nfc:              return InternalMcuState_Nfc;
    case McuState_Ir:               return InternalMcuState_Ir;
    case McuState_Initializing:     return InternalMcuState_Initializing;
    case McuState_FirmwareUpdate:   return InternalMcuState_SystemBoot;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

bool IsAllZero(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    const auto* pBinary = reinterpret_cast<const nn::Bit8*>(pData);
    for (size_t i = 0; i < dataSize; i++)
    {
        if (pBinary[i] != 0)
        {
            return false;
        }
    }

    return true;
}

bool IsCrc8Match(const uint8_t* pData, size_t dataSize, BluetoothMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES_LESS(mode, BluetoothMode_Max);

    const auto& TransferSize = BluetoothTransferSizeList[mode];
    NN_SDK_REQUIRES_GREATER(TransferSize.send, 0ul);
    NN_SDK_REQUIRES_GREATER(TransferSize.receive, 0ul);

    if (TransferSize.receive != dataSize)
    {
        return false;
    }

    auto expectCrc = CalcCrc8(pData, TransferSize.receive - 1);
    auto actualCrc = static_cast<nn::Bit8>(pData[TransferSize.receive - 1]);
    if (expectCrc == actualCrc)
    {
        return true;
    }
    else
    {
#ifdef VERBOSE
        NN_SDK_LOG(
            "[xcd] CRC8 is not match (Expect: %02X, Actual: %02X)\n",
            expectCrc,
            actualCrc);
#endif
        return false;
    }
}

nn::Result CheckNfcDevicePowerImpl(nn::xcd::DeviceHandler* pHandler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandler);

    nn::xcd::DeviceStatus status;
    pHandler->GetDeviceStatus(&status);

    // 残量判定は給電中 or 充電中のみ
    //  * 非充電の無線接続中は電池切れによる切断直前に BatteryLevel が None になるタイミングがあり、
    //    オーバーレイ通知とエラービューア表示が重なる場合があるため判定から外す。
    if (!(status.powered || status.charged))
    {
        NN_RESULT_SUCCESS;
    }

    nn::xcd::DeviceInfo info;
    pHandler->GetDeviceInfo(&info);

    switch (info.deviceType)
    {
    case nn::xcd::DeviceType_FullKey:
    case nn::xcd::DeviceType_Right:
        // 残量 CriticalLow 以上なら使用可能
        NN_RESULT_THROW_UNLESS(
            status.batteryLevel >= nn::xcd::BatteryLevel_CriticalLow,
            nn::xcd::ResultLowBattery());
        break;
    default:
        NN_RESULT_THROW(nn::xcd::ResultNotSupported());
    }

    NN_RESULT_SUCCESS;
}

nn::Result CheckIrDevicePowerImpl(nn::xcd::DeviceHandler* pHandler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandler);

    nn::xcd::DeviceStatus status;
    pHandler->GetDeviceStatus(&status);

    // 残量判定は給電中 or 充電中のみ
    if (!(status.powered || status.charged))
    {
        NN_RESULT_SUCCESS;
    }

    nn::xcd::DeviceInfo info;
    pHandler->GetDeviceInfo(&info);

    switch (info.deviceType)
    {
    case nn::xcd::DeviceType_Right:
        // 残量 CriticalLow 以上なら使用可能
        NN_RESULT_THROW_UNLESS(
            status.batteryLevel >= nn::xcd::BatteryLevel_CriticalLow,
            nn::xcd::ResultLowBattery());
        break;
    default:
        NN_RESULT_THROW(nn::xcd::ResultNotSupported());
    }

    NN_RESULT_SUCCESS;
}

}}}  // nn::xcd::detail
