﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include "xcd_CreateSystemServer.h"
#include "xcd_ServiceName.h"

namespace nn { namespace xcd { namespace detail {

namespace
{

// セッションの最大数
const auto SessionCountMax = 30;  // XXX: 適当な値

// HIPC サーバーのオプション定義
struct SystemServerManagerOption
{
    // ポインタ転送のバッファサイズ
    static const size_t PointerTransferBufferSize = 4096;
};

// HIPC サーバーのポートとセッションを管理するクラス
// ポインタ転送を使用するので、オプションを指定する
class SystemServerManager : public nn::sf::HipcSimpleAllInOneServerManager<
    SessionCountMax, 1, SystemServerManagerOption>
{
    // 内部は空
};

// 繰り返しサーバーの起動と終了が可能になるように、placement new で初期化する
std::aligned_storage<sizeof(SystemServerManager), NN_ALIGNOF(SystemServerManager)>::type g_SystemServerManagerStorage;
SystemServerManager* g_pSystemServerManager = nullptr;

}  // anonymous namespace


void InitializeSystemServer() NN_NOEXCEPT
{
    if (g_pSystemServerManager != nullptr)
    {
        return;
    }

    g_pSystemServerManager = new(&g_SystemServerManagerStorage) SystemServerManager;

    // DFC 用のサービスオブジェクトを、サービス名で関連付けてシステムに登録する。
    // これで、ここで指定したサービス名に対して nn::sf::CreateHipcProxyByName() を呼んで
    // HIPC プロキシ参照を取得できるようになる。
    auto result = g_pSystemServerManager->RegisterObjectForPort(
        CreateSystemServerByDfc(),
        SessionCountMax,
        XcdSystemServiceName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サーバーの開始
    g_pSystemServerManager->Start();
}

void LoopSystemServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_pSystemServerManager);

    // サーバーのループ処理を開始
    g_pSystemServerManager->LoopAuto();
}

}}}  // nn::xcd::detail
