﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/btm/btm.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/uart/uart.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "../xcd_Bluetooth.h"
#include "../xcd_ReportTypes.h"
#include "../xcd_ParserUtil.h"
#include "xcd_NwcpUartDriver-os.horizon.h"
#include "xcd_NwcpProtocolTypes.h"

#if 0
#define NWCP_DUMP_LOG(...) NN_DETAIL_XCD_INFO(__VA_ARGS__)
#else
#define NWCP_DUMP_LOG(...)
#endif

#if 0
#define NWCP_DUMP_DATA(...) NN_DETAIL_XCD_INFO(__VA_ARGS__)
#else
#define NWCP_DUMP_DATA(...)
#endif

#define SEND_HID_PERIODICALLY

// #define ENABLE_UART_PAIRING

namespace nn { namespace xcd { namespace detail {

namespace
{
    const size_t ReceivePacketCommonSize = 5;

    //!< Sync 中の Timeout
    const nn::TimeSpan SyncTimeout = nn::TimeSpan::FromMilliSeconds(100);

    //!< DeviceCommand に対する Timeout
    const nn::TimeSpan DeviceCommandTimeout = nn::TimeSpan::FromMilliSeconds(2000);

    //!< Uart の Hid 通信中の Timeout
    const nn::TimeSpan HidUartTimeout = nn::TimeSpan::FromMilliSeconds(100);

    //!< Uart の FW 更新中の Hid Timeout
    const nn::TimeSpan HidUartTimeoutOnFwUpdate = nn::TimeSpan::FromMilliSeconds(2000);

    //!< Uart の Hid Connection 確立直後の Hid 通信の Timeout
    const nn::TimeSpan HidUartTimeoutAfterConnection = nn::TimeSpan::FromMilliSeconds(1000);

    //!< Suspend 時 Disconnect の Timeout
    const nn::TimeSpan DisconnectTimeout = nn::TimeSpan::FromMilliSeconds(500);

    //!< Sync 処理のリトライ回数の上限
    const int SyncRetryCountMax = 30;

    //!< Disconnect 処理のリトライ回数の上限
    const int DisconnectRetryCountMax = 1;

    //!< Hid Connection の最低維持期間
    const nn::TimeSpan MinimumDisconnectionTime = nn::TimeSpan::FromMilliSeconds(500);

    uint16_t ConvertLittleEndianToUint16(const uint8_t* pInput)
    {
        return static_cast<uint16_t>(((pInput[1] << 8) & 0xff00) | (pInput[0] & 0x00ff));
    }

    void SetUint16ToLittleEndian(uint8_t* pBuffer, uint16_t value)
    {
        pBuffer[0] = value & 0xff;
        pBuffer[1] = value >> 8 & 0xff;
    }

    void SetUint32ToLittleEndian(uint8_t* pBuffer, uint32_t value)
    {
        pBuffer[0] = value & 0xff;
        pBuffer[1] = value >> 8  & 0xff;
        pBuffer[2] = value >> 16 & 0xff;
        pBuffer[3] = value >> 24 & 0xff;
    }

    void StopAndClearTimerEvent(nn::os::TimerEventType* pEvent) NN_NOEXCEPT
    {
        nn::os::StopTimerEvent(pEvent);
        nn::os::ClearTimerEvent(pEvent);
    }
}

NwcpUartDriver::NwcpUartDriver(nn::uart::PortName portName, char* pReceiveBuffer, char* pSendBuffer) NN_NOEXCEPT
    : m_Event()
    , m_SendTimerEvent()
    , m_ResetTimerEvent()
    , m_HidTimeoutTimerEvent()
    , m_DeviceCommandTimeoutTimerEvent()
    , m_SyncTimeoutTimerEvent()
    , m_DisconnectTimeoutTimerEvent()
    , m_IsDriverEnabled(false)
    , m_PortName(portName)
    , m_pUpdateEvent(nullptr)
    , m_pInquiryCompleteEvent(nullptr)
    , m_UartSession()
    , m_pSendWorkBuffer(pSendBuffer)
    , m_pReceiveWorkBuffer(pReceiveBuffer)
    , m_ReceiveBuffer()
    , m_ReceivedDataSize(0)
    , m_Address()
    , m_DeviceType(NwcpDeviceType_Unknown)
    , m_HidInputReportLength(0)
    , m_HidInputReportBuffer()
    , m_IsPacketReceived(false)
    , m_HidOutputReportBuffer()
    , m_HidOutputReportBufferIndex(0)
    , m_HidOutputReportBufferCount(0)
    , m_UartState(UartState_Detached)
    , m_RetryCount(0)
    , m_IsCrcEnabled(false)
    , m_PairingDeviceInfo()
    , m_pBatteryLevelEvent(nullptr)
    , m_BatteryLevel(BatteryLevel_None)
    , m_IsCharging(false)
    , m_DeviceCommands()
    , m_DeviceCommandsQueueIndex(0)
    , m_IsDeviceCommandBusy(false)
    , m_pAwakeTriggerReasonReadCompleteEvent(nullptr)
    , m_AwakeTriggerReason(AwakeTriggerReason_None)
    , m_pHidListener(nullptr)
    , m_FirmwareUpdateMode(FirmwareUpdateMode_Disable)
    , m_UartHidInterval(UartHidInterval)
    , m_LastAttached()
    , m_LastSendTick()
    , m_IsSendDelayDetected(false)
    , m_IsRebootEnabled(false)
    , m_pInputReportParserFunc(nullptr)
    , m_pInputReportParserArg(nullptr)
{
    nn::uart::Initialize();
}

NwcpUartDriver::~NwcpUartDriver() NN_NOEXCEPT
{
    nn::uart::Finalize();
}

bool NwcpUartDriver::Activate() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("[xcd] UartDriver: Activated\n");

    if (m_IsDriverEnabled == false)
    {
        auto initializeTimer = [](nn::os::TimerEventType* pEvent)
        {
            nn::os::InitializeTimerEvent(pEvent, nn::os::EventClearMode_ManualClear);
        };
        m_SendTimerEvent.Initialize();
        initializeTimer(&m_ResetTimerEvent);
        initializeTimer(&m_HidTimeoutTimerEvent);
        initializeTimer(&m_DeviceCommandTimeoutTimerEvent);
        initializeTimer(&m_SyncTimeoutTimerEvent);
        initializeTimer(&m_DisconnectTimeoutTimerEvent);

        m_IsDriverEnabled = OpenUart(s_BaudRateDefault);
    }

    return m_IsDriverEnabled;
}

void NwcpUartDriver::Deactivate() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("[xcd] UartDriver: Deactivated\n");

    if (m_IsDriverEnabled == true)
    {
        CloseUart();

        m_pInquiryCompleteEvent = nullptr;

        m_SendTimerEvent.Finalize();
        nn::os::FinalizeTimerEvent(&m_ResetTimerEvent);
        nn::os::FinalizeTimerEvent(&m_HidTimeoutTimerEvent);
        nn::os::FinalizeTimerEvent(&m_DeviceCommandTimeoutTimerEvent);
        nn::os::FinalizeTimerEvent(&m_SyncTimeoutTimerEvent);
        nn::os::FinalizeTimerEvent(&m_DisconnectTimeoutTimerEvent);

        m_UartHidInterval = UartHidInterval;

        m_IsDriverEnabled = false;

        m_pInputReportParserFunc = nullptr;
        m_pInputReportParserArg  = nullptr;
    }
}

void NwcpUartDriver::StartUartCommunication(nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_ReceivedDataSize = 0;

    m_pUpdateEvent = pEvent;

    m_UartState = UartState_Sync;

    m_RetryCount = 0;

    m_FirmwareUpdateMode = FirmwareUpdateMode_Disable;

    m_UartHidInterval = UartHidInterval;

    m_AwakeTriggerReason = AwakeTriggerReason_None;

    // DeviceCommandQueue を初期化
    for (auto& command : m_DeviceCommands)
    {
        command.isAvailable = false;
        command.isSent = false;
    }
    m_DeviceCommandsQueueIndex = 0;

    m_IsCrcEnabled = false;

    // MagicPacket を送信
    SendMagicPacket();

    // Sync を送信
    SendSync();
}

void NwcpUartDriver::SetSampleParserFunction(InputReportParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);

    m_pInputReportParserFunc = func;
    m_pInputReportParserArg  = pArg;
}

void NwcpUartDriver::SetInquiryCompleteEvent(nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pInquiryCompleteEvent = pEvent;
}

Result NwcpUartDriver::GetDeviceInfo(::nn::bluetooth::Address* pOutAddress, NwcpDeviceType* pOutDeviceType) NN_NOEXCEPT
{
    *pOutAddress = m_Address;
    *pOutDeviceType = m_DeviceType;

    NN_RESULT_THROW_UNLESS(m_UartState == UartState_Attached, ResultNwcpDeviceDetached());
    NN_RESULT_SUCCESS;
}

void NwcpUartDriver::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    if (pMultiWaitHolder == m_Event.GetWaitId())
    {
        ReadUartData();
    }
    else if (pMultiWaitHolder == m_SendTimerEvent.GetWaitId())
    {
        if (m_IsPacketReceived == true)
        {
            // HidPacket を受信していなかったらタイムアウトは継続
            nn::os::StartOneShotTimerEvent(&m_HidTimeoutTimerEvent, HidUartTimeout);
            nn::os::ClearTimerEvent(&m_HidTimeoutTimerEvent);
        }
        if (m_DeviceType == NwcpDeviceType_Fifty)
        {
            SendFiftyPoll();
        }
        else
        {
            SendHidOutput();
        }
    }
}

void NwcpUartDriver::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitTimerEvent(&m_ResetTimerEvent))
    {
        nn::os::ClearTimerEvent(&m_ResetTimerEvent);
        NWCP_DUMP_LOG("[xcd] Reset Timeout\n");
        HandleResetComplete();
        return;
    }

    if (nn::os::TryWaitTimerEvent(&m_HidTimeoutTimerEvent))
    {
        nn::os::ClearTimerEvent(&m_HidTimeoutTimerEvent);
        NWCP_DUMP_LOG("[xcd] Hid Timeout\n");

        // Uart のイベントがシグナル状態の場合は、タイムアウトを再度セットする
        if (::nn::os::TryWaitSystemEvent(m_Event.GetBase()) == true)
        {
            nn::os::StartOneShotTimerEvent(&m_HidTimeoutTimerEvent, HidUartTimeout);
            return;
        }

        // HidUartTimeout 以上の間何も送信できていなければ送信遅延発生と見なす
        bool isSendDelayed = (nn::os::GetSystemTick() - m_LastSendTick).ToTimeSpan() >= HidUartTimeout;
        if (isSendDelayed && !m_IsSendDelayDetected)
        {
            // 送信遅延が検出された場合は、初回のみタイムアウトを再度セットする
            m_IsSendDelayDetected = true;
            nn::os::StartOneShotTimerEvent(&m_HidTimeoutTimerEvent, HidUartTimeout);
        }
        else
        {
            // Timeout を超えたら切断として扱う
            HandleDeviceDetach();

            // Reboot が実行されたときもここにくるので Timeout をキャンセル
            StopAndClearTimerEvent(&m_DisconnectTimeoutTimerEvent);
        }

        return;
    }
    if (nn::os::TryWaitTimerEvent(&m_DeviceCommandTimeoutTimerEvent))
    {
        nn::os::ClearTimerEvent(&m_DeviceCommandTimeoutTimerEvent);
        NWCP_DUMP_LOG("[xcd] DeviceCommand Timeout\n");
        // Uart のイベントがシグナル状態の場合は、タイムアウトを再度セットする
        if (::nn::os::TryWaitSystemEvent(m_Event.GetBase()) == true)
        {
            nn::os::StartOneShotTimerEvent(&m_DeviceCommandTimeoutTimerEvent, DeviceCommandTimeout);
            return;
        }
        else
        {
            HandleDeviceCommandTimeout();
        }

        return;
    }
    if (nn::os::TryWaitTimerEvent(&m_SyncTimeoutTimerEvent))
    {
        nn::os::ClearTimerEvent(&m_SyncTimeoutTimerEvent);
        NWCP_DUMP_LOG("[xcd] Sync Timeout\n");

        // Uart のイベントがシグナル状態の場合は、タイムアウトを再度セットする
        if (::nn::os::TryWaitSystemEvent(m_Event.GetBase()) == true)
        {
            nn::os::StartOneShotTimerEvent(&m_SyncTimeoutTimerEvent, SyncTimeout);
            return;
        }
        else
        {
            if (m_RetryCount < SyncRetryCountMax)
            {
                // Sync 処理中は応答がない場合は、再度 Sync を送る
                SendMagicPacket();
                SendSync();
                m_RetryCount++;
            }
            else
            {
                // Retry 回数を超えたら Detach
                HandleDeviceDetach();
            }
        }
        return;
    }
    if (nn::os::TryWaitTimerEvent(&m_DisconnectTimeoutTimerEvent))
    {
        nn::os::ClearTimerEvent(&m_DisconnectTimeoutTimerEvent);
        NWCP_DUMP_LOG("[xcd] Disconnect Timeout\n");
        if (m_RetryCount < DisconnectRetryCountMax)
        {
            // 再度 Disconnect する
            if (m_IsRebootEnabled)
            {
                DisconnectHidByReboot();
            }
            else
            {
                DisconnectHid();
            }
            m_RetryCount++;
        }
        else
        {
            // Retry 回数を超えたら Detach
            HandleDeviceDetach();
        }
        return;
    }

}  // NOLINT(readability/fn_size)

size_t NwcpUartDriver::GetInputReport(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(size, UartReceiveBufferSize);

    memcpy(pBuffer, &m_HidInputReportBuffer[0], m_HidInputReportLength);

    return m_HidInputReportLength;
}

Result NwcpUartDriver::SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(size, UartReceiveBufferSize);

    NN_RESULT_THROW_UNLESS(m_UartState == UartState_Attached, ResultNwcpDeviceDetached());

    NN_RESULT_THROW_UNLESS(m_HidOutputReportBufferCount < OutputReportRingBufferCountMax, ResultNwcpOutputRingBufferFull());

    if (size > 0)
    {
        m_HidOutputReportBuffer[m_HidOutputReportBufferIndex].size = size;
        memcpy(m_HidOutputReportBuffer[m_HidOutputReportBufferIndex].buffer, pBuffer, size);

        m_HidOutputReportBufferIndex = (m_HidOutputReportBufferIndex + 1) % OutputReportRingBufferCountMax;
        m_HidOutputReportBufferCount++;
    }

    NN_RESULT_SUCCESS;
}

void NwcpUartDriver::DetachUart() NN_NOEXCEPT
{
    if (m_UartState == UartState_Attached)
    {
        m_RetryCount = 0;
        m_IsRebootEnabled = false;

        auto diffTime = (nn::os::GetSystemTick() - m_LastAttached).ToTimeSpan();
        if (m_LastAttached != nn::os::Tick(0) && diffTime < MinimumDisconnectionTime)
        {
            // WORKAROUND: 切断まで最低 MinimumDisconnectionTime は待つ
            nn::os::StartOneShotTimerEvent(&m_DisconnectTimeoutTimerEvent, MinimumDisconnectionTime - diffTime);
        }
        else
        {
            DisconnectHid();
        }
    }
}

void NwcpUartDriver::DetachUartByReboot() NN_NOEXCEPT
{
    if (m_UartState == UartState_Attached)
    {
        m_RetryCount = 0;
        m_IsRebootEnabled = true;

        auto diffTime = (nn::os::GetSystemTick() - m_LastAttached).ToTimeSpan();
        if (m_LastAttached != nn::os::Tick(0) && diffTime < MinimumDisconnectionTime)
        {
            // WORKAROUND: 切断まで最低 MinimumDisconnectionTime は待つ
            nn::os::StartOneShotTimerEvent(&m_DisconnectTimeoutTimerEvent, MinimumDisconnectionTime - diffTime);
        }
        else
        {
            DisconnectHidByReboot();
        }
    }
}

::nn::TimeSpan NwcpUartDriver::GetInterval() NN_NOEXCEPT
{
    return m_UartHidInterval;
}

void NwcpUartDriver::SetReport(uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES(size > 0);
    NN_SDK_REQUIRES_LESS_EQUAL(size, NwcpSize_SetReportPayloadMax);

    uint8_t buffer[NwcpSize_HciHeaderSize + NwcpSize_SetReportPayloadMax];

    auto length = size + 1;

    GenerateHciHeader(buffer, length);
    buffer[NwcpByte_NnIndicator] = NwcpNnIndicator_SetReport;
    memcpy(&buffer[NwcpByte_NwcpOpCode], pBuffer, size);

    // コントローラーが切断しないために Dummy の HidReport を送る
    uint8_t DummyCommand[13] =
        {
            0x19, 0x01, 0x03,
            0x08, // サイズ下位バイト
            0x00, // サイズ上位バイト
            0x92, 0x00,
            0x01, // サイズ下位バイト
            0x00, // サイズ上位バイト
            0x00, 0x00,
            0x00,
            0x1F,
        };
    // コマンドを送信
    WriteUartData(DummyCommand, sizeof(DummyCommand));

    // HidPacket を受信していなかったらタイムアウトは継続
    nn::os::StartOneShotTimerEvent(&m_HidTimeoutTimerEvent, HidUartTimeoutOnFwUpdate);
    nn::os::ClearTimerEvent(&m_HidTimeoutTimerEvent);
    WriteUartData(buffer, NwcpSize_HciHeaderSize + length);

    m_pHidListener = pListener;
    m_FirmwareUpdateMode = FirmwareUpdateMode_SetReport;
}

void NwcpUartDriver::GetReport(uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    const auto size = NwcpSize_HciHeaderSize + NwcpSize_GetReportPayload;
    uint8_t buffer[size];

    auto length = NwcpSize_GetReportPayload;

    GenerateHciHeader(buffer, length);
    buffer[NwcpByte_NnIndicator] = NwcpNnIndicator_GetReport;
    buffer[NwcpByte_NwcpOpCode] = reportId;

    // HidPacket を受信していなかったらタイムアウトは継続
    nn::os::StartOneShotTimerEvent(&m_HidTimeoutTimerEvent, HidUartTimeoutOnFwUpdate);
    nn::os::ClearTimerEvent(&m_HidTimeoutTimerEvent);
    WriteUartData(buffer, size);

    m_pHidListener = pListener;
    m_FirmwareUpdateMode = FirmwareUpdateMode_GetReport;
}

void NwcpUartDriver::SetFirmwareUpdateModeEnabled(bool enabled) NN_NOEXCEPT
{
    if (enabled)
    {
        // Hid の定常送信処理を止める
        StopAndClearTimerEvent(m_SendTimerEvent.GetBase());
        m_FirmwareUpdateMode = FirmwareUpdateMode_Idle;
    }
    else
    {
        m_FirmwareUpdateMode = FirmwareUpdateMode_Disable;
        // Hid の定常送信処理を再開
        m_IsPacketReceived    = true;
        m_IsSendDelayDetected = false;
        nn::os::StartPeriodicTimerEvent(m_SendTimerEvent.GetBase(), 0, m_UartHidInterval);
    }
}

void NwcpUartDriver::SetFastModeEnabled(bool enabled) NN_NOEXCEPT
{
    auto newInterval = enabled ? UartHidIntervalFast : UartHidInterval;
    if (newInterval == m_UartHidInterval)
    {
        return;
    }

    // Hid の定常送信間隔を変更
    m_UartHidInterval = newInterval;
    StopAndClearTimerEvent(m_SendTimerEvent.GetBase());
    if (m_FirmwareUpdateMode == FirmwareUpdateMode_Disable)
    {
        nn::os::StartPeriodicTimerEvent(m_SendTimerEvent.GetBase(), 0, m_UartHidInterval);
    }

    SetHidInterval();
}

PadState NwcpUartDriver::GetFiftyPadState() NN_NOEXCEPT
{
    return m_FiftyPadState;
}

bool NwcpUartDriver::IsUartAttached() NN_NOEXCEPT
{
    return (m_UartState == UartState_Attached);
}

void NwcpUartDriver::SetBatteryCharing(bool enabled, bool is200mA) NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_WriteChargerSettingsCommandPayload;
    uint8_t buffer[length] = { 0 };
    SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], NwcpSize_WriteChargerSettingsCommandPayload);
    if (enabled == true)
    {
        buffer[NwcpByte_WriteChargerSettings] |= NwcpChargerSettingsMask_Enable;
        buffer[NwcpByte_WriteChargerSettings] |= (is200mA) ? NwcpChargerSettingsMask_200mA : NwcpChargerSettingsMask_100mA;
    }
    QueueDeviceCommand(buffer, length, NwcpOpCode_WriteChargerSettings);

    NWCP_DUMP_LOG("    -> WriteChargerSettings\n");
}

BatteryLevel NwcpUartDriver::GetBatteryLevel() NN_NOEXCEPT
{
    return m_BatteryLevel;
}

bool NwcpUartDriver::IsCharging() NN_NOEXCEPT
{
    return m_IsCharging;
}

void NwcpUartDriver::BindBatteryLevelChangeEvent(::nn::os::LightEventType* pEventType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventType);

    m_pBatteryLevelEvent = pEventType;
}

void NwcpUartDriver::UnbindBatteryLevelChangeEvent(::nn::os::LightEventType* pEventType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventType);

    if (m_pBatteryLevelEvent == pEventType)
    {
        m_pBatteryLevelEvent = nullptr;
    }
}

void NwcpUartDriver::StartReadingAwakeTriggerReason(::nn::os::LightEventType* pEventType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventType);

    if (m_UartState != UartState_Attached)
    {
        // デバイスが接続されていない場合は、シグナルして返す
        ::nn::os::SignalLightEvent(pEventType);
        return;
    }

    m_pAwakeTriggerReasonReadCompleteEvent = pEventType;
    ReadWakeUpReason();
}

AwakeTriggerReason NwcpUartDriver::GetAwakeTriggerReason() NN_NOEXCEPT
{
    return m_AwakeTriggerReason;
}

void NwcpUartDriver::ConnectWithBluetooth() NN_NOEXCEPT
{
    const uint8_t pageCommand[] = {0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                   0x06, 0x01};
    const uint8_t hidLength = 0x30;
    const uint8_t bufferSize = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + hidLength;

    uint8_t buffer[bufferSize] = { 0 };
    std::memcpy(&buffer[NwcpByte_NwcpPayload], pageCommand, sizeof(pageCommand));

    buffer[NwcpByte_NwcpPayloadLength] = hidLength;

    GenerateNwcpPacket(buffer, bufferSize, NwcpNnIndicator_Hid);

    WriteUartData(buffer, bufferSize);
}

void NwcpUartDriver::CloseUart() NN_NOEXCEPT
{
    UnbindUartReceiveEvent();

    nn::uart::ClosePort(&m_UartSession);
}


bool NwcpUartDriver::OpenUart(::nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    nn::uart::PortConfigType config;
    CreateUartConfig(&config, baudRate);

    NWCP_DUMP_LOG("    -> Uart Config Created\n");
    if (nn::uart::OpenPort(&m_UartSession, m_PortName, config) == false)
    {
        NWCP_DUMP_LOG("[xcd] Uart Open failed\n");
        return false;
    }

    NWCP_DUMP_LOG("    -> Bind Event\n");

    // データの受信のために SystemEvent をバインド
    BindUartReceiveEvent();

    return true;
}

void NwcpUartDriver::ReadUartData() NN_NOEXCEPT
{
    size_t receivedBytes;
    auto bufferIndex = m_ReceivedDataSize;
    size_t expectedPacketLength = 0;
    auto result = nn::uart::Receive(&receivedBytes, &m_ReceiveBuffer[bufferIndex], UartReceiveBufferSize - bufferIndex, &m_UartSession);
    if (!result.IsSuccess())
    {
        // TODO: receive エラー
        NWCP_DUMP_LOG("[xcd] Read Failed\n");
        HandleDeviceDetach();
        return;
    }
    if (receivedBytes == 0)
    {
        NWCP_DUMP_LOG("[xcd] 0 bytes read\n");
        return;
    }

    // 全てのデータが読み込まれたかどうかを確認
    m_ReceivedDataSize = bufferIndex + receivedBytes;

    NWCP_DUMP_DATA("\n[xcd] Recieved size:%d\n", receivedBytes);
    NWCP_DUMP_DATA("\n[xcd] Data ");
    for (size_t i = 0; i < m_ReceivedDataSize; i++)
    {
        if (i % 8 == 0) NWCP_DUMP_DATA("[%02d] ", i / 8);
        NWCP_DUMP_DATA("%02x ", m_ReceiveBuffer[i]);
        if (i % 8 == 7) NWCP_DUMP_DATA("\n");
    }
    NWCP_DUMP_DATA("\n");

    while (m_ReceivedDataSize > 0)
    {
        auto totalReceivedSize = m_ReceivedDataSize;
        for (size_t i = 0; i < totalReceivedSize; i++)
        {
            // データは 0x19 から始まる
            if (m_ReceiveBuffer[i] == 0x19)
            {
                if (i > 0)
                {
                    std::memmove(m_ReceiveBuffer, &m_ReceiveBuffer[i], m_ReceivedDataSize);
                }
                break;
            }

            m_ReceivedDataSize--;
            NWCP_DUMP_LOG("[xcd] Wrong Indicator. %02x Data Corrupted size:%d\n", m_ReceiveBuffer[i], m_ReceivedDataSize);

            if (m_ReceivedDataSize == 0)
            {
                return;
            }
        }

        // Hci パケット長がの記録されたバイトまでの読み込みが完了しているか
        if (m_ReceivedDataSize > NwcpByte_HciLength + (NwcpSize_HciLength - 1))
        {
            expectedPacketLength = CalculatePacketLength(&m_ReceiveBuffer[NwcpByte_HciLength]);

            // パケット長分のデータの読み込みが完了していない
            if (expectedPacketLength > m_ReceivedDataSize)
            {
                NWCP_DUMP_LOG("[xcd] Read Remaining Data total:%d current:%d\n", expectedPacketLength, m_ReceivedDataSize);
                // 残りのデータを読み込む
                return;
            }
        }
        else
        {
            NWCP_DUMP_LOG("[xcd] Wait for packet size\n");
            // 残りのデータを読み込む
            return;
        }

        // データをパース
        ParseUartData(m_ReceiveBuffer, expectedPacketLength);

        // パースしたデータを破棄
        m_ReceivedDataSize -= expectedPacketLength;
        std::memmove(m_ReceiveBuffer, &m_ReceiveBuffer[expectedPacketLength], m_ReceivedDataSize);
    }
}

void NwcpUartDriver::ParseUartData(uint8_t* pBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    // Callback 先をバックアップ
    auto listener = m_pHidListener;

    // CRC チェック / Firmware 更新中は CRC チェックを無効化
    if (!m_IsCrcEnabled ||
        CheckCrcInNwcpPacket(pBuffer, length) ||
        m_FirmwareUpdateMode != FirmwareUpdateMode_Disable)
    {
        // データのパース
        switch (pBuffer[NwcpByte_NnIndicator])
        {
        case NwcpNnIndicator_Sync:
            // Sync レスポンスに CRC が含まれていれば以降有効化
            if (CheckCrcInNwcpPacket(pBuffer, length))
            {
                m_IsCrcEnabled = true;
            }

            // Timeout をキャンセル
            StopAndClearTimerEvent(&m_SyncTimeoutTimerEvent);
            ReceivedSync();
            break;

        case NwcpNnIndicator_DeviceResult:
            // Timeout をキャンセル
            StopAndClearTimerEvent(&m_DeviceCommandTimeoutTimerEvent);
            ReceivedDeviceCommand(pBuffer, length);
            break;

        case NwcpNnIndicator_Hid:
            if (m_FirmwareUpdateMode == FirmwareUpdateMode_Disable)
            {
                // Timeout をキャンセル
                StopAndClearTimerEvent(&m_HidTimeoutTimerEvent);
                ReceivedHid(pBuffer, length);
            }
            else
            {
                NWCP_DUMP_LOG("Drop Hid Packet during FWUP\n");
            }
            break;

        case NwcpNnIndicator_GetReportResult:
            if (m_FirmwareUpdateMode == FirmwareUpdateMode_GetReport)
            {
                // Timeout をキャンセル
                StopAndClearTimerEvent(&m_HidTimeoutTimerEvent);
                m_FirmwareUpdateMode = FirmwareUpdateMode_Idle;
                if (listener != nullptr)
                {
                    m_pHidListener = nullptr;
                    listener->GetReportComplete(&pBuffer[NwcpByte_NnIndicator + 1], length - NwcpSize_HciLength);
                }
            }
            break;

        case NwcpNnIndicator_SetReportSuccess:
            if (m_FirmwareUpdateMode == FirmwareUpdateMode_SetReport)
            {
                // Timeout をキャンセル
                StopAndClearTimerEvent(&m_HidTimeoutTimerEvent);
                m_FirmwareUpdateMode = FirmwareUpdateMode_Idle;
                if(listener != nullptr)
                {
                    m_pHidListener = nullptr;
                    listener->SetReportComplete(nn::ResultSuccess());
                }
            }
            break;


        case NwcpNnIndicator_FiftyData:
            // Timeout をキャンセル
            StopAndClearTimerEvent(&m_HidTimeoutTimerEvent);
            HandleFiftyData(pBuffer, length);
            break;

        default:
            NWCP_DUMP_LOG("[xcd] Data corrupted\n");
        }
    }
    else
    {
        NWCP_DUMP_LOG("[xcd] Crc error\n");
    }
}

void NwcpUartDriver::WriteUartData(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
/*
    NWCP_DUMP_DATA("\n[xcd] this: %x send:", this);
    for (size_t i = 0; i < size; i++)
    {
        NWCP_DUMP_DATA("%02x ", pBuffer[i]);
    }
    NWCP_DUMP_DATA("\n");*/
    // コマンドを送信
    size_t doneBytes;
    nn::uart::Send(&doneBytes,
                   &m_UartSession,
                   pBuffer,
                   size);
//    NWCP_DUMP_DATA("[xcd] %d bytes sent\n", doneBytes);
    NN_ABORT_UNLESS_LESS_EQUAL(doneBytes, size);
}

void NwcpUartDriver::CreateUartConfig(::nn::uart::PortConfigType* pOutValue, ::nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    nn::uart::InitializePortConfig(pOutValue, baudRate, m_pSendWorkBuffer, UartWorkBufferSize, m_pReceiveWorkBuffer, UartWorkBufferSize);
    nn::uart::SetPortConfigFlowControlMode(pOutValue, nn::uart::FlowControlMode_Hardware);
    nn::uart::SetPortConfigInvertPins(pOutValue, true, false, true, false);
}

size_t NwcpUartDriver::CalculatePacketSizeFull(size_t payloadSize) NN_NOEXCEPT
{
    return payloadSize + ReceivePacketCommonSize;
}

void NwcpUartDriver::GenerateHciHeader(uint8_t* pOutValue, size_t payloadLength) NN_NOEXCEPT
{
    pOutValue[NwcpByte_HciIndicator] = 0x19;
    pOutValue[NwcpByte_HciConnectionHandle] = 0x01;
    pOutValue[NwcpByte_HciConnectionHandle + 1] = 0x03;
    SetUint16ToLittleEndian(&pOutValue[NwcpByte_HciLength], payloadLength);
}

void NwcpUartDriver::GenerateNwcpPacket(uint8_t *pOutValue, size_t bufferSize, uint8_t nnIndicator) NN_NOEXCEPT
{
    auto length = ConvertLittleEndianToUint16(&pOutValue[NwcpByte_NwcpPayloadLength]) + NwcpSize_NwcpHeaderSize;
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, CalculatePacketSizeFull(length));
    NN_UNUSED(bufferSize);

    GenerateHciHeader(pOutValue, length);
    pOutValue[NwcpByte_NnIndicator] = nnIndicator;

    if (m_IsCrcEnabled)
    {
        AddCrcForNwcpPacket(pOutValue, nnIndicator);
    }
}

void NwcpUartDriver::AddCrcForNwcpPacket(uint8_t *pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize);

    if (pOutValue[NwcpByte_NnIndicator] != NwcpNnIndicator_Sync &&
        ConvertLittleEndianToUint16(&pOutValue[NwcpByte_NwcpPayloadLength]))
    {
        pOutValue[NwcpByte_NwcpDataCrc] = CalculateCrc8(&pOutValue[NwcpByte_NwcpPayload], ConvertLittleEndianToUint16(&pOutValue[NwcpByte_NwcpPayloadLength]));
    }
    pOutValue[NwcpByte_NwcpHeaderCrc] = CalculateCrc8(&pOutValue[NwcpByte_NnIndicator], NwcpSize_NwcpHeaderSize - 1);
}

size_t NwcpUartDriver::CalculatePacketLength(uint8_t *pBuffer) NN_NOEXCEPT
{
    return static_cast<size_t>(ConvertLittleEndianToUint16(pBuffer)) + NwcpSize_HciHeaderSize;
}

void NwcpUartDriver::BindUartReceiveEvent() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("[xcd] UartDriver: BindUartSystemEvent\n");

    // Interrupt を bind する。
    NN_ABORT_UNLESS(
        nn::uart::BindPortEvent(
            m_Event.GetBase(),
            &m_UartSession,
            nn::uart::PortEventType_ReceiveBufferReady,
            1) == true
    );

    GetTaskManager().RegisterPeriodicTask(this);

    // 優先的にハンドリングするイベントを MultiWait に登録する
    GetTaskManager().RegisterEvent(&m_Event, this);
    GetTaskManager().RegisterEvent(&m_SendTimerEvent, this);
}

void NwcpUartDriver::UnbindUartReceiveEvent() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("[xcd] UartDriver: UnbindUartSystemEvent\n");

    GetTaskManager().UnregisterPeriodicTask(this);

    if (nn::uart::UnbindPortEvent(m_Event.GetBase(), &m_UartSession) == true)
    {
        NWCP_DUMP_LOG("    -> Unbinded\n");
        GetTaskManager().UnregisterEvent(&m_Event);
        m_Event.Destroy();
    }

    GetTaskManager().UnregisterEvent(&m_SendTimerEvent);

    StopAndClearTimerEvent(&m_HidTimeoutTimerEvent);
    StopAndClearTimerEvent(&m_DeviceCommandTimeoutTimerEvent);
    StopAndClearTimerEvent(&m_SyncTimeoutTimerEvent);
    StopAndClearTimerEvent(&m_DisconnectTimeoutTimerEvent);
}

void NwcpUartDriver::SendMagicPacket() NN_NOEXCEPT
{
    const size_t size = NwcpSize_MagicPacket;
    uint8_t buffer[size];
    memcpy(buffer, NwcpMagicPacket, size);

    WriteUartData(buffer, size);

    NWCP_DUMP_LOG("    -> SendMagicPacket\n");
}

void NwcpUartDriver::HandleDeviceAttach() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("[xcd] Device Attached\n");

    // Hid の確立完了
    m_UartState = UartState_Attached;
    nn::os::SignalLightEvent(m_pUpdateEvent);

    // 送受信関連フラグの削除
    m_IsPacketReceived    = true;
    m_IsSendDelayDetected = false;

    // リングバッファを初期化
    m_HidOutputReportBufferCount = 0;
    m_HidOutputReportBufferIndex = 0;

    //!< Fifty の入力状態初期化
    m_FiftyPadState.buttons.Reset();
    m_FiftyPadState.analogStickL = AnalogStickState();
    m_FiftyPadState.analogStickR = AnalogStickState();

    // 最終送信、接続時の Tick を初期化
    m_LastSendTick = m_LastAttached = nn::os::GetSystemTick();

    // 送信用のタイマーを開始
#ifdef SEND_HID_PERIODICALLY
    nn::os::StartPeriodicTimerEvent(m_SendTimerEvent.GetBase(), 0, m_UartHidInterval);
#else
    // 1回 Read を送信
    SendHidOutput();
#endif
}

void NwcpUartDriver::HandleDeviceDetach() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("[xcd] Device Detached\n");

#ifdef SEND_HID_PERIODICALLY
    if (m_UartState == UartState_Attached)
    {
        nn::os::StopTimerEvent(m_SendTimerEvent.GetBase());
    }
#endif

    // Uart が切断されたと判断
    m_UartState = UartState_Detached;

    //!< 電池状態の初期化
    m_BatteryLevel = BatteryLevel_None;
    m_IsCharging = false;

    //!< Fifty の入力状態初期化
    m_FiftyPadState.buttons.Reset();
    m_FiftyPadState.analogStickL = AnalogStickState();
    m_FiftyPadState.analogStickR = AnalogStickState();

    // 起動復帰要因を初期化
    m_AwakeTriggerReason = AwakeTriggerReason_None;
    if (m_pAwakeTriggerReasonReadCompleteEvent != nullptr)
    {
        ::nn::os::SignalLightEvent(m_pAwakeTriggerReasonReadCompleteEvent);
        m_pAwakeTriggerReasonReadCompleteEvent = nullptr;
    }

    // DeviceCommandQueue を初期化
    for (auto& command : m_DeviceCommands)
    {
        command.isAvailable = false;
        command.isSent = false;
    }
    m_DeviceCommandsQueueIndex = 0;
    m_IsDeviceCommandBusy = false;

    nn::os::SignalLightEvent(m_pUpdateEvent);
}

void NwcpUartDriver::ResetDevice() NN_NOEXCEPT
{
    const size_t size = NwcpSize_HciHeaderSize + NwcpSize_ResetPayload;
    uint8_t buffer[size];

    // Reset 完了のタイマーをセット
    nn::os::StartOneShotTimerEvent(&m_ResetTimerEvent, nn::TimeSpan::FromSeconds(2));

    memcpy(&buffer[NwcpSize_HciHeaderSize], NwcpReset, NwcpSize_ResetPayload);
    GenerateHciHeader(buffer, NwcpSize_ResetPayload);
    buffer[NwcpByte_NwcpHeaderCrc] = 0;

    WriteUartData(buffer, size);

    NWCP_DUMP_LOG("    -> SendReset\n");
}

void NwcpUartDriver::HandleResetComplete() NN_NOEXCEPT
{
    if (m_UartState == UartState_Sync)
    {
        // MagicPacket を送信
        SendMagicPacket();

        // Sync を送信
        SendSync();
    }
}

void NwcpUartDriver::SendSync() NN_NOEXCEPT
{
    const size_t size = NwcpSize_HciHeaderSize + NwcpSize_SyncPayload;
    uint8_t buffer[size];
    memcpy(&buffer[NwcpSize_HciHeaderSize], NwcpSync, NwcpSize_SyncPayload);
    GenerateHciHeader(buffer, NwcpSize_SyncPayload);
    buffer[NwcpByte_NwcpHeaderCrc] = 0;

    // Sync には CRC は含めない

    // Timeout を設定
    nn::os::StartOneShotTimerEvent(&m_SyncTimeoutTimerEvent, SyncTimeout);
    WriteUartData(buffer, size);

    NWCP_DUMP_LOG("    -> SendSync\n");
}

void NwcpUartDriver::ReceivedSync() NN_NOEXCEPT
{
    NWCP_DUMP_LOG("    -> Received Sync\n");

    if (m_UartState == UartState_Sync)
    {
        m_UartState = UartState_Inquiry;

        // デバイスの情報を読み出す
        SendInquiry();
    }
}

void NwcpUartDriver::QueueDeviceCommand(uint8_t* pBuffer, int length, uint8_t opCode) NN_NOEXCEPT
{
    auto& deviceCommand = m_DeviceCommands[m_DeviceCommandsQueueIndex];

    if (deviceCommand.isAvailable == true)
    {
        // コマンドキューがいっぱいのためコマンドを捨てる
        NN_DETAIL_XCD_ERROR("Nwcp Device Command Queue is full. Command Dropped\n");
    }

    deviceCommand.opCode = opCode;
    deviceCommand.length = length;
    std::memcpy(deviceCommand.buffer, pBuffer, length);
    deviceCommand.buffer[NwcpByte_NwcpOpCode] = opCode;
    GenerateNwcpPacket(deviceCommand.buffer, length, NwcpNnIndicator_DeviceCommand);
    deviceCommand.isAvailable = true;
    deviceCommand.isSent = false;
    m_DeviceCommandsQueueIndex = (m_DeviceCommandsQueueIndex + 1) % DeviceCommandQueueCountMax;

    // ビジーでなければ、即座に DeviceCommand を送信する
    TrySendDeviceCommandFromQueue();
}

void NwcpUartDriver::TrySendDeviceCommandFromQueue() NN_NOEXCEPT
{
    // ビジーの場合は送信しない
    if (m_IsDeviceCommandBusy == true)
    {
        return;
    }

    DeviceCommand* pCommand = nullptr;

    // もっとも古いコマンドを探索
    for (int i = 0; i < DeviceCommandQueueCountMax; i++)
    {
        if(m_DeviceCommands[(i + m_DeviceCommandsQueueIndex) % DeviceCommandQueueCountMax].isAvailable == true)
        {
            pCommand = &m_DeviceCommands[(i + m_DeviceCommandsQueueIndex) % DeviceCommandQueueCountMax];
            break;
        }
    }

    if (pCommand == nullptr)
    {
        // 送信するコマンドがなかった
        return;
    }

    if (!pCommand->isSent)
    {
        m_IsDeviceCommandBusy = true;
        pCommand->isSent = true;
        ::nn::os::StartOneShotTimerEvent(&m_DeviceCommandTimeoutTimerEvent, DeviceCommandTimeout);
        WriteUartData(pCommand->buffer, pCommand->length);
    }
}

void NwcpUartDriver::HandleDeviceCommandTimeout() NN_NOEXCEPT
{
    // 古いコマンドから送信中のコマンドを探索
    for (int i = 0; i < DeviceCommandQueueCountMax; i++)
    {
        auto& deviceCommand = m_DeviceCommands[(i + m_DeviceCommandsQueueIndex) % DeviceCommandQueueCountMax];
        if (deviceCommand.isAvailable == true &&
            deviceCommand.isSent == true)

        {
            deviceCommand.isAvailable = false;
            deviceCommand.isSent = false;
            auto opCode = deviceCommand.opCode;

            switch (opCode)
            {
            case NwcpOpCode_Inquiry:
            case NwcpOpCode_Pairing:
            case NwcpOpCode_UpdateUartBaudRate:
            case NwcpOpCode_CreateHidConnection:
            case NwcpOpCode_DisconnectHid:
            case NwcpOpCode_SetHidInterval:
                HandleDeviceDetach();
                return;

            case NwcpOpCode_WriteChargerSettings:
                if (m_UartState == UartState_WriteChargerSettings)
                {
                    // 初期化シーケンスのときはタイムアウト処理を実行する
                    HandleDeviceDetach();
                }
                else
                {
                    // 何もしない
                }
                break;

            case NwcpOpCode_ReadWakeUpReason:
                if (m_pAwakeTriggerReasonReadCompleteEvent != nullptr)
                {
                    ::nn::os::SignalLightEvent(m_pAwakeTriggerReasonReadCompleteEvent);
                    // 一度シグナルしたらイベントはクリアする
                    m_pAwakeTriggerReasonReadCompleteEvent = nullptr;
                }
                break;

            default:
                ;
                // タイムアウト対象のコマンドが見つからなかった
            }
            break;
        }
    }

    // キューされているコマンドがあれば送信
    m_IsDeviceCommandBusy = false;
    TrySendDeviceCommandFromQueue();
}

void NwcpUartDriver::ReceivedDeviceCommand(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    NWCP_DUMP_LOG("    -> Received Command Result opcode %02x status %02x size %02x\n", pBuffer[NwcpByte_NwcpOpCode], pBuffer[NwcpByte_NwcpStatus], size);

    // もっとも古いコマンドを探索
    for (int i = 0; i < DeviceCommandQueueCountMax; i++)
    {
        auto& deviceCommand = m_DeviceCommands[(i + m_DeviceCommandsQueueIndex) % DeviceCommandQueueCountMax];
        if (deviceCommand.isAvailable == true &&
            deviceCommand.isSent == true &&
            deviceCommand.opCode == pBuffer[NwcpByte_NwcpOpCode])
        {
            deviceCommand.isAvailable = false;
            deviceCommand.isSent = false;
        }
    }

    switch (pBuffer[NwcpByte_NwcpOpCode])
    {
    case NwcpOpCode_Inquiry:
        HandleInquiryResult(pBuffer, size);
        break;

    case NwcpOpCode_WriteChargerSettings:
        HandleWriteChargerSettings(pBuffer, size);
        break;

    case NwcpOpCode_UpdateUartBaudRate:
        HandleUpdateUartBaudRateResult(pBuffer, size);
        break;

    case NwcpOpCode_Pairing:
        HandlePairingResult(pBuffer, size);
        break;

    case NwcpOpCode_ReadWakeUpReason:
        HandleReadWakeUpReasonResult(pBuffer, size);
        break;

    case NwcpOpCode_CreateHidConnection:
        HandleCreateHidConnectionResult(pBuffer, size);
        break;

    case NwcpOpCode_DisconnectHid:
        HandleDisconnectHidResult(pBuffer, size);
        break;

    case NwcpOpCode_SetHidInterval:
        HandleSetHidIntervalResult(pBuffer, size);
        break;

    default:
        ;
        // 何もしない
    }

    // キューされているコマンドがあれば送信
    m_IsDeviceCommandBusy = false;
    TrySendDeviceCommandFromQueue();
}

void NwcpUartDriver::ReceivedHid(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    m_HidInputReportLength = size - NwcpSize_NwcpHeaderSize - NwcpSize_HciHeaderSize;

    auto reportBuffer = &pBuffer[NwcpByte_NwcpPayload];
    if (m_pInputReportParserFunc != nullptr)
    {
        m_pInputReportParserFunc(m_pInputReportParserArg, reportBuffer, m_HidInputReportLength);
    }

    // 電池残量をかすめ取る
    auto newBatteryLevel = static_cast<BatteryLevel>((reportBuffer[InputReportByte_Status] >> InputReportBit_BatteryLeast) & 0x07);
    auto newIsCharging = ((reportBuffer[InputReportByte_Status] & (1 << InputReportBit_Charge)) != 0);

    if (m_BatteryLevel != newBatteryLevel || m_IsCharging != newIsCharging)
    {
        m_BatteryLevel = newBatteryLevel;
        m_IsCharging = newIsCharging;
        if (m_pBatteryLevelEvent != nullptr)
        {
            ::nn::os::SignalLightEvent(m_pBatteryLevelEvent);
        }
    }

    // 最初の HID パケットの受信を確認して、接続完了
    if (m_UartState == UartState_CreateHid)
    {
        HandleDeviceAttach();
    }

    m_IsPacketReceived = true;
#ifndef SEND_HID_PERIODICALLY
    SendHidOutput();
#endif
}

void NwcpUartDriver::SendInquiry() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize;
    uint8_t buffer[length] = { 0 };
    QueueDeviceCommand(buffer, length, NwcpOpCode_Inquiry);

    NWCP_DUMP_LOG("    -> Send Inquiry\n");
}

void NwcpUartDriver::HandleInquiryResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_InquiryResponsePayload);

    NWCP_DUMP_LOG("    -> Received Inquiry\n");
    for (int i = 0; i < 6; i++)
    {
        m_Address.address[i] = pBuffer[NwcpByte_InquiryBdAddr + 5 - i];
    }

    // TODO DeviceType の設定
    switch (pBuffer[NwcpByte_InquiryDeviceType])
    {
    case 0x01:
        m_DeviceType = NwcpDeviceType_JoyLeft;
        break;
    case 0x02:
        m_DeviceType = NwcpDeviceType_JoyRight;
        break;
    case 0x21:
        NWCP_DUMP_LOG("    -> Fifty Attached\n");
        m_DeviceType = NwcpDeviceType_Fifty;
        break;
    default:
        m_DeviceType = NwcpDeviceType_Unknown;
    }

    if (m_DeviceType == NwcpDeviceType_Fifty)
    {
        // ポーリングを開始する
        HandleDeviceAttach();
    }
    else
    {
        m_UartState = UartState_WriteChargerSettings;
        // 確実に Charge Enable, 200mA に設定する
        SetBatteryCharing(true, true);
    }

    if (m_pInquiryCompleteEvent != nullptr)
    {
        ::nn::os::SignalLightEvent(m_pInquiryCompleteEvent);
    }
}

void NwcpUartDriver::HandleWriteChargerSettings(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_WriteChargerSettingsResponsePayload);

    NWCP_DUMP_LOG("    -> Write Charger Settings\n");

    switch (pBuffer[NwcpByte_NwcpStatus])
    {
    case 0x00:
        // 何もしない
        break;
    default:
        NWCP_DUMP_LOG("    -> WriteChargerSettings Error\n");
    }

    if (m_UartState == UartState_WriteChargerSettings)
    {
        m_UartState = UartState_CreateHid;
        UpdateUartBaudRate();
    }
}

void NwcpUartDriver::StartPairing() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_Pairing;
    uint8_t buffer[length] = { 0 };

    auto btDeviceInfo = GetHostBluetoothDeviceInfo();

    buffer[NwcpByte_NwcpStatus] = NwcpPairingStatus_Initiate;
    NWCP_DUMP_LOG("    -> Master BdAddr ");
    for (int i = 0; i < NwcpSize_PairingBdAddr; i++)
    {
        buffer[NwcpByte_PairingBdAddr + NwcpSize_PairingBdAddr - i - 1] = btDeviceInfo.address.address[i];
        NWCP_DUMP_LOG(":%02x", buffer[NwcpByte_PairingBdAddr + NwcpSize_PairingBdAddr - i - 1]);
    }
    NWCP_DUMP_LOG("\n");

    NWCP_DUMP_LOG("    -> Master BdName ");
    for (int i = 0; i < NwcpSize_PairingDeviceName; i++)
    {
        buffer[NwcpByte_PairingDeviceName + i] = btDeviceInfo.bdName.name[i];
        NWCP_DUMP_LOG("%c", buffer[NwcpByte_PairingDeviceName + i]);
    }
    NWCP_DUMP_LOG("\n");
    memcpy(&buffer[NwcpByte_PairingClassOfDevice], &btDeviceInfo.classOfDevice.cod, NwcpSize_PairingClassOfDevice);
    buffer[NwcpByte_PairingFeatureSet] = btDeviceInfo.featureSet;

    SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], NwcpSize_Pairing);

    QueueDeviceCommand(buffer, length, NwcpOpCode_Pairing);

    NWCP_DUMP_LOG("    -> Update Pairing\n");
}

void NwcpUartDriver::HandlePairingResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_Pairing);

    NWCP_DUMP_LOG("    -> Received PairingResult\n");

    auto result = nn::ResultSuccess();

    switch (pBuffer[NwcpByte_NwcpStatus])
    {
    case NwcpPairingStatus_Initiate:
    {
        // ペアリング情報の解析
        NWCP_DUMP_LOG("    -> Bd Addr : ");
        for (int i = 0; i < ::nn::btm::SIZE_OF_BDADDRESS; i++)
        {
            m_PairingDeviceInfo.address.address[i] = pBuffer[NwcpByte_PairingBdAddr + ::nn::btm::SIZE_OF_BDADDRESS - i - 1];
            NWCP_DUMP_LOG("%02x ", m_PairingDeviceInfo.address.address[i]);
        }
        NWCP_DUMP_LOG("\n");

        memcpy(&m_PairingDeviceInfo.bdName, &pBuffer[NwcpByte_PairingDeviceName], sizeof(m_PairingDeviceInfo.bdName));
        NWCP_DUMP_LOG("    -> DeviceName :%s\n", m_PairingDeviceInfo.bdName.name);
        memcpy(&m_PairingDeviceInfo.classOfDevice, &pBuffer[NwcpByte_PairingClassOfDevice], sizeof(m_PairingDeviceInfo.classOfDevice));

        // LinkKey シーケンス
        const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_Pairing;
        uint8_t buffer[length] = { 0 };

        buffer[NwcpByte_NwcpStatus] = NwcpPairingStatus_LinkKey;
        SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], NwcpSize_Pairing);

        QueueDeviceCommand(buffer, length, NwcpOpCode_Pairing);

        NWCP_DUMP_LOG("    -> Sending Pairing LinkKey\n");
        return;
    }

    case NwcpPairingStatus_LinkKey:
    {
        // ペアリング情報の解析
        NWCP_DUMP_LOG("    -> Link Key :");
        for (int i = 0; i < ::nn::btm::SIZE_OF_LINKKEY; i++)
        {
            m_PairingDeviceInfo.linkKey.key[i] = pBuffer[NwcpByte_PairingLinkKey + i] ^ 0xAA;
            NWCP_DUMP_LOG("%02x ", m_PairingDeviceInfo.linkKey.key[i]);
        }
        NWCP_DUMP_LOG("\n");

        // Ack の送信
        const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_Pairing;
        uint8_t buffer[length] = { 0 };

        buffer[NwcpByte_NwcpStatus] = NwcpPairingStatus_Ack;
        SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], NwcpSize_Pairing);

        QueueDeviceCommand(buffer, length, NwcpOpCode_Pairing);

        NWCP_DUMP_LOG("    -> Sending Pairing Ack\n");
        return;
    }

    case NwcpPairingStatus_Ack:
        NWCP_DUMP_LOG("    -> Received Ack\n");
        RegisterBluetoothDevice(m_PairingDeviceInfo);
        result = nn::ResultSuccess();
        break;

    default:
        // TODO: エラー用のResultを定義
        result = nn::ResultSuccess();
        break;
    }

    // Hid Commandの完了を通知
    m_PairingDeviceInfo = BluetoothDeviceInfo();
    m_UartState = UartState_CreateHid;
    DisconnectHid();
}

void NwcpUartDriver::UpdateUartBaudRate() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_UpdateUartBaudRatePayload;
    uint8_t buffer[length] = { 0 };
    SetUint32ToLittleEndian(&buffer[NwcpByte_UpdateUartBaudRateValue], s_BaudRateHigh);
    SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], NwcpSize_UpdateUartBaudRatePayload);
    QueueDeviceCommand(buffer, length, NwcpOpCode_UpdateUartBaudRate);

    NWCP_DUMP_LOG("    -> Update Uart Baud Rate\n");
}

void NwcpUartDriver::HandleUpdateUartBaudRateResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize);

    NWCP_DUMP_LOG("    -> Received Uart Baud Rate\n");

    CloseUart();
    OpenUart(s_BaudRateHigh);

#ifdef ENABLE_UART_PAIRING
    StartPairing();
#else
    DisconnectHid();
#endif
}

void NwcpUartDriver::ReadWakeUpReason() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize;
    uint8_t buffer[length] = { 0 };
    QueueDeviceCommand(buffer, length, NwcpOpCode_ReadWakeUpReason);

    NWCP_DUMP_LOG("    -> Read Wake Up Reason\n");
}

void NwcpUartDriver::HandleReadWakeUpReasonResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_ReadWakeUpReasonResponsePayload);

    NWCP_DUMP_LOG("    -> Received Read Wake Up Reason\n");
    uint8_t reason = pBuffer[NwcpByte_WakeUpReason];

    switch (reason)
    {
    case NwcpWakeUpReasonBitMask_Home:
        m_AwakeTriggerReason = AwakeTriggerReason_HomePressed;
        break;
    case NwcpWakeUpReasonBitMask_BatteryLevelChange:
        m_AwakeTriggerReason = AwakeTriggerReason_BatteryLevelChanged;
        break;
    default:
        m_AwakeTriggerReason = AwakeTriggerReason_None;
    }

    if (m_pAwakeTriggerReasonReadCompleteEvent != nullptr)
    {
        ::nn::os::SignalLightEvent(m_pAwakeTriggerReasonReadCompleteEvent);
        // 一度シグナルしたらイベントはクリアする
        m_pAwakeTriggerReasonReadCompleteEvent = nullptr;
    }
}

void NwcpUartDriver::CreateHidConnection() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize;
    uint8_t buffer[length] = { 0 };
    QueueDeviceCommand(buffer, length, NwcpOpCode_CreateHidConnection);

    NWCP_DUMP_LOG("    -> Create Hid Connection\n");
}

void NwcpUartDriver::HandleCreateHidConnectionResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize);

    NWCP_DUMP_LOG("    -> Received CreateHid\n");

    SetHidInterval();
}

void NwcpUartDriver::DisconnectHid() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize;
    uint8_t buffer[length] = { 0 };
    QueueDeviceCommand(buffer, length, NwcpOpCode_DisconnectHid);

    NWCP_DUMP_LOG("    -> Disconnect Hid Connection\n");
}

void NwcpUartDriver::DisconnectHidByReboot() NN_NOEXCEPT
{
    const uint8_t rebootCommand[] = {0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                     0x06, 0x00};
    const uint8_t hidLength = 0x30;
    const uint8_t bufferSize = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + hidLength;

    uint8_t buffer[bufferSize] = { 0 };
    std::memcpy(&buffer[NwcpByte_NwcpPayload], rebootCommand, sizeof(rebootCommand));

    buffer[NwcpByte_NwcpPayloadLength] = hidLength;

    GenerateNwcpPacket(buffer, bufferSize, NwcpNnIndicator_Hid);

    nn::os::StartOneShotTimerEvent(&m_DisconnectTimeoutTimerEvent, DisconnectTimeout);
    WriteUartData(buffer, bufferSize);
}

void NwcpUartDriver::HandleDisconnectHidResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize);

    NWCP_DUMP_LOG("    -> Received DisconnectHid\n");

    if (m_UartState == UartState_Sync)
    {
        SendMagicPacket();
        SendSync();
    }
    else if (m_UartState == UartState_CreateHid)
    {
        CreateHidConnection();
    }
    else if (m_UartState == UartState_Attached)
    {
        HandleDeviceDetach();
    }
}

void NwcpUartDriver::SetHidInterval() NN_NOEXCEPT
{
    const size_t length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + NwcpSize_SetHidIntervalPayload;
    uint8_t buffer[length] = { 0 };
    buffer[NwcpByte_SetHidIntervalValue] = m_UartHidInterval.GetMilliSeconds();
    SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], NwcpSize_SetHidIntervalPayload);
    QueueDeviceCommand(buffer, length, NwcpOpCode_SetHidInterval);

    NWCP_DUMP_LOG("    -> Set Hid Interval\n");
}

void NwcpUartDriver::HandleSetHidIntervalResult(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize);

    NWCP_DUMP_LOG("    -> Received SetHidInterval\n");

    // 1回 HidOutput を送信
    ::nn::os::StartOneShotTimerEvent(&m_HidTimeoutTimerEvent, HidUartTimeoutAfterConnection);
    ::nn::os::ClearTimerEvent(&m_HidTimeoutTimerEvent);
    SendHidOutput();
}

void NwcpUartDriver::SendHidOutput() NN_NOEXCEPT
{
    uint8_t buffer[UartReceiveBufferSize] = { 0 };
    size_t hidReportLength;

#ifndef SEND_HID_PERIODICALLY
    if (m_IsPacketReceived == false)
    {
        // HidPacket を受信していなかったら送信はしない
        // Send を連続して送信しない
        return;
    }
#endif // !SEND_HID_PERIODICALLY

    if (m_HidOutputReportBufferCount > 0)
    {
        // 送信対象のリングバッファののインデックス
        auto bufferIndex = ((m_HidOutputReportBufferIndex + OutputReportRingBufferCountMax) - m_HidOutputReportBufferCount) % OutputReportRingBufferCountMax;
        m_HidOutputReportBufferCount--;

        hidReportLength = m_HidOutputReportBuffer[bufferIndex].size;
        std::memcpy(&buffer[NwcpHidByte_ReportId], m_HidOutputReportBuffer[bufferIndex].buffer, hidReportLength);
    }
    else
    {
        // 送信データがひとつもないときは POLL 用パケットを送信
        hidReportLength = NwcpHidSize_Poll;
        buffer[NwcpHidByte_ReportId] = NwcpHidReportId_Poll;
    }

    SetUint16ToLittleEndian(&buffer[NwcpByte_NwcpPayloadLength], hidReportLength);
    auto length = NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize + hidReportLength;
    GenerateNwcpPacket(buffer, length, NwcpNnIndicator_Hid);

    WriteUartData(buffer, length);

    m_IsPacketReceived    = false;
    m_IsSendDelayDetected = false;
    m_LastSendTick        = nn::os::GetSystemTick();
}

void NwcpUartDriver::SendFiftyPoll() NN_NOEXCEPT
{
    uint8_t buffer[] =
    { 0x19, 0x01, 0x03, 0x07, 0x00, 0x9A, 0x01, 0x00, 0x00, 0x00, 0x00, 0x48
    };
    m_IsPacketReceived    = false;
    m_IsSendDelayDetected = false;
    m_LastSendTick        = nn::os::GetSystemTick();
    WriteUartData(buffer, 12);
}

void NwcpUartDriver::HandleFiftyData(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize);

    NWCP_DUMP_LOG("    -> Received Fifty Data\n");

    auto bit = nn::util::BitPack8();
    auto pPayload = &(pBuffer[NwcpByte_FiftyDataPayload]);
    bit.SetMaskedBits(0xFF, pPayload[3]);
    m_FiftyPadState.buttons.Set<PadButton::Y>       (bit.GetBit(0));
    m_FiftyPadState.buttons.Set<PadButton::X>       (bit.GetBit(1));
    m_FiftyPadState.buttons.Set<PadButton::B>       (bit.GetBit(2));
    m_FiftyPadState.buttons.Set<PadButton::A>       (bit.GetBit(3));
    m_FiftyPadState.buttons.Set<PadButton::SR>      (bit.GetBit(4));
    m_FiftyPadState.buttons.Set<PadButton::SL>      (bit.GetBit(5));
    m_FiftyPadState.buttons.Set<PadButton::R>       (bit.GetBit(6));
    m_FiftyPadState.buttons.Set<PadButton::ZR>      (bit.GetBit(7));

    bit.SetMaskedBits(0xFF, pPayload[4]);
    m_FiftyPadState.buttons.Set<PadButton::Select>  (bit.GetBit(0));
    m_FiftyPadState.buttons.Set<PadButton::Start>   (bit.GetBit(1));
    m_FiftyPadState.buttons.Set<PadButton::StickR>  (bit.GetBit(2));
    m_FiftyPadState.buttons.Set<PadButton::StickL>  (bit.GetBit(3));
    m_FiftyPadState.buttons.Set<PadButton::Home>    (bit.GetBit(4));
    m_FiftyPadState.buttons.Set<PadButton::Shot>    (bit.GetBit(5));

    bit.SetMaskedBits(0xFF, pPayload[5]);
    m_FiftyPadState.buttons.Set<PadButton::Down>    (bit.GetBit(0));
    m_FiftyPadState.buttons.Set<PadButton::Up>      (bit.GetBit(1));
    m_FiftyPadState.buttons.Set<PadButton::Right>   (bit.GetBit(2));
    m_FiftyPadState.buttons.Set<PadButton::Left>    (bit.GetBit(3));
    // SL,SR はorをByte1のbit4, 5とのORをとる
    m_FiftyPadState.buttons.Set<PadButton::SR>      (bit.GetBit(4) || m_FiftyPadState.buttons.Test<PadButton::SR>());
    m_FiftyPadState.buttons.Set<PadButton::SL>      (bit.GetBit(5) || m_FiftyPadState.buttons.Test<PadButton::SL>());
    m_FiftyPadState.buttons.Set<PadButton::L>       (bit.GetBit(6));
    m_FiftyPadState.buttons.Set<PadButton::ZL>      (bit.GetBit(7));

    ParseAnalogStick(&(m_FiftyPadState.analogStickL), &pPayload[6]);
    m_FiftyPadState.analogStickR.x = 2048;
    m_FiftyPadState.analogStickR.y = 2048;
    m_IsPacketReceived = true;
}

uint8_t NwcpUartDriver::CalculateCrc8(const uint8_t* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    const uint8_t crc8Poly = 0x8d;
    uint32_t r;
    uint32_t i, j;
    r = 0;
    for (i = 0; i < size; i++)
    {
        r = (r ^ pData[i]) & 0xff;
        for (j = 0 ; j < 8 ; j++)
        {
            if (r & 0x80)
            {
                r = (r << 1) ^ crc8Poly;
            }
            else
            {
                r <<= 1;
            }
        }
    }
    return static_cast<uint8_t>(r);
}

bool NwcpUartDriver::CheckCrcInNwcpPacket(const uint8_t *pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    if (size < NwcpSize_HciHeaderSize + NwcpSize_NwcpHeaderSize)
    {
        return false;
    }

    return (pData[NwcpByte_NwcpHeaderCrc] == CalculateCrc8(&pData[NwcpByte_NnIndicator], NwcpSize_NwcpHeaderSize - 1) &&
            (pData[NwcpByte_NnIndicator] == NwcpNnIndicator_Sync ||
             ConvertLittleEndianToUint16(&pData[NwcpByte_NwcpPayloadLength]) == 0 ||
             pData[NwcpByte_NwcpDataCrc] == CalculateCrc8(&pData[NwcpByte_NwcpPayload], ConvertLittleEndianToUint16(&pData[NwcpByte_NwcpPayloadLength]))));
}

const ::nn::TimeSpan NwcpUartDriver::UartHidInterval     = ::nn::TimeSpan::FromMilliSeconds(15);
const ::nn::TimeSpan NwcpUartDriver::UartHidIntervalFast = ::nn::TimeSpan::FromMilliSeconds(5);

}}} // namespace nn::xcd::detail
