﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/pinmux/pinmux.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Rail.h>
#include <nn/xcd/xcd_Input.h>

#include "xcd_LinkMonitorTypes.h"
#include "xcd_NwcpDriver-os.horizon.h"
#include "xcd_NwcpHidAccessor-os.horizon.h"
#include "xcd_PowerSupplyManager-os.horizon.h"
#include "../xcd_IEventTask.h"
#include "../xcd_MultiWaitEvent.h"

namespace nn { namespace xcd { namespace detail{

//!< Nwcp で接続できるデバイスの最大数
const int NwcpDeviceCountMax = 2;

//!< デバイスの接続状態の監視を行うためのプラットフォームごとの実装クラス
class NwcpManager final : public IEventTask
{
    NN_DISALLOW_MOVE(NwcpManager);
    NN_DISALLOW_COPY(NwcpManager);

private:
    //!< 左レールデバイスの接続状態に変化があった場合に受け取る通知
    nn::os::LightEventType m_LeftEvent;

    //!< 左レールデバイスの接続状態に変化があった場合に受け取る通知
    nn::os::LightEventType m_RightEvent;

    //!< デバイスの接続状態に変化があった際に通知するためのイベントオブジェクト
    nn::os::LightEventType* m_pDeviceUpdateEvent;

    NwcpDriver m_LeftDriver;
    NwcpDriver m_RightDriver;

    NwcpHidAccessor m_LeftAccessor;
    NwcpHidAccessor m_RightAccessor;

    //!< PowerSpplyManager
    PowerSupplyManager m_PowerSupplyManager;

    //!< アクティベートされているかどうか
    bool m_Activated;

    //!< サスペンドの完了の通知を受けるイベント
    nn::os::LightEventType m_SuspendCompleteEvent;

    //!< サスペンド処理が完了したときに上位に通知するイベント
    ::nn::os::LightEventType* m_pSuspendCompletedEvent;

public:
    NwcpManager() NN_NOEXCEPT;
    virtual ~NwcpManager() NN_NOEXCEPT NN_OVERRIDE;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態の監視を開始する
    void StartMonitoring(nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を停止する
    void StopMonitoring() NN_NOEXCEPT;

    //!< 接続済みデバイスのハンドルを取得する
    size_t GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT;

    //!< デバイスハンドルからHIDAccessorを取得する
    HidAccessor* GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< 右のレールへのデバイスの接続状態を取得する
    bool IsRightNwcpAttached() NN_NOEXCEPT;

    //!< 左のレールへのデバイスの接続状態を取得する
    bool IsLeftNwcpAttached() NN_NOEXCEPT;

    //!< レール部の Uart 通信の有効無効状態を変更する
    void SetNwcpEnabled(bool enabled) NN_NOEXCEPT;

    //!< ジョイコンのレール部の接続状態に変化があった場合に通知するイベントを登録します
    void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< ジョイコンのレール部の接続状態に変化について、変化の内容を取得します
    void GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddressRailUpdateEventType) NN_NOEXCEPT;

    //!< レール部の処理を Suspend します。処理は非同期で行われます
    void TriggerSuspend(::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    //!< レール部の処理を Resume します。
    void Resume() NN_NOEXCEPT;

    //!< 本体の給電状態を取得します。
    bool IsConsolePowered() NN_NOEXCEPT;

    //!< 左レール部のスリープ復帰要因を取得します
    AwakeTriggerReason GetAwakeTriggerReasonForLeftRail() NN_NOEXCEPT;

    //!< 右レール部のスリープ復帰要因を取得します
    AwakeTriggerReason GetAwakeTriggerReasonForRightRail() NN_NOEXCEPT;

    //!< 切断時再起動を有効にする
    void SetRebootEnabled(bool enabled) NN_NOEXCEPT;

    //!< Fifty が接続されているかどうかを取得します
    bool IsFiftyConnected() NN_NOEXCEPT;

    //!< Fifty の入力状態を取得します
    PadState GetFiftyPadState() NN_NOEXCEPT;

};

}}} // namespace nn::xcd::detail
