﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/pinmux/pinmux.h>
#include <nn/result/result_HandlingUtility.h>

#include "xcd_NwcpDriver-os.horizon.h"

namespace nn { namespace xcd { namespace detail {

namespace {

// Suspend 時に CTS が Low に下がるのを待つうえでの Timeout 値
const auto UartCtsTimeout = ::nn::TimeSpan::FromMilliSeconds(150);

// Device の Detach を待つうえでの Timeout 値
const auto UartDetachTimeout = ::nn::TimeSpan::FromMilliSeconds(150);

// Device の Detect 端子に対する debounce 値
const auto DetectDebounceTime = ::nn::TimeSpan::FromMilliSeconds(32);

}

NwcpDriver::NwcpDriver(nn::pinmux::AssignablePinGroupName pinmuxName,
                       nn::gpio::GpioPadName detectGpioPadName,
                       nn::gpio::GpioPadName uartCtsGpioPadName,
                       nn::uart::PortName uartPortName,
                       nn::gpio::GpioPadName chargerGpioPadName,
                       char* pUartRecivedBuffer,
                       char* pUartSendBuffer,
                       NwcpDeviceType deviceType) NN_NOEXCEPT :
    m_SupportedDeviceType(deviceType),
    m_CurrentDeviceType(NwcpDeviceType_Unknown),
    m_PinmuxName(pinmuxName),
    m_DetectGpioMonitor(detectGpioPadName),
    m_UartCtsGpioMonitor(uartCtsGpioPadName),
    m_UartDriver(uartPortName, pUartRecivedBuffer, pUartSendBuffer),
    m_Charger(chargerGpioPadName),
    m_Flags(),
    m_SuspendState(SuspendState_Suspended)
{
    // 何もしない
}

NwcpDriver::~NwcpDriver() NN_NOEXCEPT
{
    // 何もしない
}

void NwcpDriver::SetPowerSupplyManager(PowerSupplyManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pPowerSupplyManager = pManager;
}

void NwcpDriver::StartMonitoring(nn::os::LightEventType* pDeviceUpdateEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Flags.Test<NwcpDriverFlag::IsActivated>(), false);
    NN_SDK_REQUIRES_NOT_NULL(pDeviceUpdateEvent);
    NN_SDK_REQUIRES_NOT_NULL(m_pPowerSupplyManager);

    m_pDeviceUpdateEvent = pDeviceUpdateEvent;

    nn::os::InitializeLightEvent(&m_DetectGpioEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeLightEvent(&m_UartCtsGpioEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeLightEvent(&m_InquiryCompleteEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeLightEvent(&m_UartEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeTimerEvent(&m_UartCtsTimeoutEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeTimerEvent(&m_UartDetachTimeoutEvent, nn::os::EventClearMode_ManualClear);
    GetTaskManager().RegisterPeriodicTask(this);

    // Pinmux の初期化
    nn::pinmux::Initialize();

    // Charger の有効化
    m_Charger.SetPowerSupplyManager(m_pPowerSupplyManager);
    m_Charger.SetNwcpUartDriver(&m_UartDriver);
    m_Charger.Activate();
    m_Charger.ResetChargeState();

    // Gpio の監視を有効
    EnableGpioMonitoring();

    // 起動直後の状態をチェック
    UpdateGpioState();

    m_Flags.Set<NwcpDriverFlag::ReadAwakeTriggerReason>(false);

    // システム起動フラグを有効にする
    m_Flags.Set<NwcpDriverFlag::IsFromBoot>();

    m_SuspendState = SuspendState_Resumed;
    m_Flags.Set<NwcpDriverFlag::IsActivated>();
}

void NwcpDriver::StopMonitoring() NN_NOEXCEPT
{
    GetTaskManager().UnregisterPeriodicTask(this);

    nn::os::FinalizeLightEvent(&m_DetectGpioEvent);
    nn::os::FinalizeLightEvent(&m_UartCtsGpioEvent);
    nn::os::FinalizeLightEvent(&m_InquiryCompleteEvent);
    nn::os::FinalizeLightEvent(&m_UartEvent);
    nn::os::FinalizeTimerEvent(&m_UartCtsTimeoutEvent);
    nn::os::FinalizeTimerEvent(&m_UartDetachTimeoutEvent);

    m_Flags.Set<NwcpDriverFlag::IsActivated>(false);
}

void NwcpDriver::SetSampleParserFunction(InputReportParserFunc func, void* pArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(func);
    NN_SDK_REQUIRES_NOT_NULL(pArg);

    m_UartDriver.SetSampleParserFunction(func, pArg);
}

Result NwcpDriver::GetDeviceInfo(::nn::bluetooth::Address* pOutAddress, NwcpDeviceType* pOutDeviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceType);

    NN_RESULT_DO(m_UartDriver.GetDeviceInfo(pOutAddress, pOutDeviceType));

    NN_RESULT_SUCCESS;
}


size_t NwcpDriver::GetInputReport(uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    return m_UartDriver.GetInputReport(pBuffer, size);
}


Result NwcpDriver::SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_DO(m_UartDriver.SetOutputReport(pBuffer, size));

    NN_RESULT_SUCCESS;
}

void NwcpDriver::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMultiWaitHolder);

    // 各種イベントは PeriodicEventFunction でハンドリングするため、ここでは何もしない
}

void NwcpDriver::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitLightEvent(&m_DetectGpioEvent))
    {
        nn::os::ClearLightEvent(&m_DetectGpioEvent);
        UpdateGpioState();
    }
    else if (nn::os::TryWaitLightEvent(&m_UartCtsGpioEvent))
    {
        nn::os::ClearLightEvent(&m_UartCtsGpioEvent);
        HandleUartCtsUpdate();
    }
    else if (nn::os::TryWaitLightEvent(&m_UartEvent))
    {
        nn::os::ClearLightEvent(&m_UartEvent);
        UpdateUartState();
    }
    else if (nn::os::TryWaitTimerEvent(&m_UartCtsTimeoutEvent))
    {
        nn::os::ClearTimerEvent(&m_UartCtsTimeoutEvent);
        HandleUartCtsTimeout();
    }
    else if (nn::os::TryWaitTimerEvent(&m_UartDetachTimeoutEvent))
    {
        nn::os::ClearTimerEvent(&m_UartDetachTimeoutEvent);
        HandleUartDetachTimeout();
    }
    else if (nn::os::TryWaitLightEvent(&m_InquiryCompleteEvent))
    {
        nn::os::ClearLightEvent(&m_InquiryCompleteEvent);
        HandleInquiryComplete();
    }
}

void NwcpDriver::SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    m_pRailUpdateEvent = pEvent;
}

void NwcpDriver::GetRailUpdateEventType(RailUpdateEventType* pOutEventType, ::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT
{
    memcpy(pOutAddress, &(m_RailUpdateEventQueue[0].address), ::nn::bluetooth::AddressLength);
    *pOutEventType = m_RailUpdateEventQueue[0].eventType;

    // レールの先頭に何かイベントが突っ込まれている場合はキューを前づめ
    if (m_RailUpdateEventQueue[0].eventType != RailUpdateEventType_None)
    {
        for (int i = 1; i < RailUpdateEventQueueSize; i++)
        {
            m_RailUpdateEventQueue[i - 1] = m_RailUpdateEventQueue[i];
        }
        m_RailUpdateEventQueue[RailUpdateEventQueueSize - 1].eventType = RailUpdateEventType_None;
    }
}

void NwcpDriver::UpdateGpioState() NN_NOEXCEPT
{
    if (m_DetectGpioMonitor.IsLow())
    {
        if (m_Flags.Test<NwcpDriverFlag::IsNwcpEnabled>() == true)
        {
            // 電源制御を開始する
            m_Charger.StartPowerSupply();
        }

        if(m_Flags.Test<NwcpDriverFlag::IsUartAllowed>())
        {
            m_DetectGpioMonitor.Deactivate();

            if (m_UartDriver.Activate() == true)
            {
                m_UartDriver.SetInquiryCompleteEvent(&m_InquiryCompleteEvent);

                // Pinmux で　Uart に変更
                SwitchPinmux(true);

                m_UartDriver.StartUartCommunication(&m_UartEvent);

                m_Flags.Set<NwcpDriverFlag::IsUartUsed>(true);
            }
        }
        else
        {
            m_Flags.Set<NwcpDriverFlag::IsNwcpDeviceActivated>(true);
        }
    }
    else
    {
        // 充電制御状態をリセット
        m_Charger.ResetChargeState();

        // 起動以外の場合はレールからの切断を通知
        if (m_CurrentDeviceType == m_SupportedDeviceType && m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>() == true)
        {
            QueueNwcpAttachEvent(false, m_Address);
        }
        m_CurrentDeviceType = NwcpDeviceType_Unknown;

        m_Flags.Set<NwcpDriverFlag::IsFromBoot>(false);
        m_Flags.Set<NwcpDriverFlag::IsInquiryCompleted>(false);
        m_Flags.Set<NwcpDriverFlag::IsNwcpDeviceActivated>(false);
    }
}

void NwcpDriver::UpdateUartState() NN_NOEXCEPT
{
    NwcpDeviceType deviceType;
    ::nn::bluetooth::Address address;

    if (m_UartDriver.GetDeviceInfo(&address, &deviceType).IsSuccess())
    {
        if (m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>() == false)
        {
            // 充電管理を開始する
            m_Charger.StartChargeStateManagement();
        }

        m_Flags.Set<NwcpDriverFlag::IsNwcpDeviceActivated>(true);

        // 起動要因読み出し要求が来ている場合は、読み出しを行う
        if (m_Flags.Test<NwcpDriverFlag::ReadAwakeTriggerReason>())
        {
            m_UartDriver.StartReadingAwakeTriggerReason(&m_AwakeTriggerReasonReadCompleteEvent);
            m_Flags.Set<NwcpDriverFlag::ReadAwakeTriggerReason>(false);
        }

        // Suspend 処理中の場合は、Charger を Suspend してから Detach
        if (m_SuspendState != SuspendState_Resumed)
        {
            // 充電制御を終了
            m_Charger.StopChargeStateManagement();
            m_Charger.Suspend();
            Detach();
        }
        else
        {
            // スリープ中以外の場合は、FromBoot を落とす
            m_Flags.Set<NwcpDriverFlag::IsFromBoot>(false);
        }
    }
    else
    {
        // Uart を無効化
        m_UartDriver.Deactivate();
        m_Flags.Set<NwcpDriverFlag::IsUartUsed>(false);

        // 起動要因読み出し要求が来ている場合は、イベントをシグナルして待ち状態を解除
        if (m_Flags.Test<NwcpDriverFlag::ReadAwakeTriggerReason>())
        {
            ::nn::os::SignalLightEvent(&m_AwakeTriggerReasonReadCompleteEvent);
            m_Flags.Set<NwcpDriverFlag::ReadAwakeTriggerReason>(false);
        }

        // スリープ処理中以外の場合
        if (m_SuspendState == SuspendState_Resumed)
        {
            if (m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>())
            {
                // 充電管理を停止
                m_Charger.StopChargeStateManagement();

                // 強制的に充電/給電を停止
                m_Charger.StopPowerSupply();
            }

            // Gpio の監視を有効
            EnableGpioMonitoring();

            // 現在の Gpio の状態をチェック
            UpdateGpioState();
        }
        // スリープ処理中
        else
        {
            // Detach 待ち
            ::nn::os::StartOneShotTimerEvent(&m_UartDetachTimeoutEvent, UartDetachTimeout);
        }
        m_Flags.Set<NwcpDriverFlag::IsNwcpDeviceActivated>(false);
    }

    // 上位にデバイスの変更を通知
    nn::os::SignalLightEvent(m_pDeviceUpdateEvent);
}

void NwcpDriver::HandleUartCtsTimeout() NN_NOEXCEPT
{
    // GpioMonitor を終了
    m_UartCtsGpioMonitor.Deactivate();
    CompleteSuspend();
}

void NwcpDriver::HandleUartDetachTimeout() NN_NOEXCEPT
{
    // Detach 後の Charger Suspend 処理をここで実行する
    // Detach 前に給電を停止するとデバイスがバッテリレベル変動を複数回検出してしまう
    m_Charger.SuspendLow();
    // CTS が下りるのをまって Suspend 終了
    TryIsCtsLow();
}

void NwcpDriver::HandleInquiryComplete() NN_NOEXCEPT
{
    m_UartDriver.GetDeviceInfo(&m_Address, &m_CurrentDeviceType);

    // 起動時以外は Inquiry 完了した時点で通知を出す
    if (m_CurrentDeviceType == m_SupportedDeviceType && m_Flags.Test<NwcpDriverFlag::IsInquiryCompleted>() == false)
    {
        QueueNwcpAttachEvent(true, m_Address);
    }
    m_Flags.Set<NwcpDriverFlag::IsInquiryCompleted>();
}

void NwcpDriver::HandleUartCtsUpdate() NN_NOEXCEPT
{
    // CTS 変化通知がタイムアウトした場合は、CTS の状態によらずスリープを完了させる

    // GpioMonitor を終了
    m_UartCtsGpioMonitor.Deactivate();
    // スリープ処理完了
    CompleteSuspend();
}

void NwcpDriver::EnableGpioMonitoring() NN_NOEXCEPT
{
    // 最初は GPIO として使用
    SwitchPinmux(false);

    // GPIO の監視をアクティベート
    m_DetectGpioMonitor.Activate(&m_DetectGpioEvent, DetectDebounceTime.GetMilliSeconds());
}

void NwcpDriver::TriggerSuspend(::nn::os::LightEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pSuspendCompleteEvent = pEvent;

    if (m_Flags.Test<NwcpDriverFlag::IsActivated>() == false || m_SuspendState != SuspendState_Resumed)
    {
        return;
    }

    // スリープ中
    m_SuspendState = SuspendState_Suspending;

    if (m_Flags.Test<NwcpDriverFlag::IsNwcpEnabled>() == true)
    {
        // UART 通信を無効にする
        m_Flags.Set<NwcpDriverFlag::IsUartAllowed>(false);
    }

    if (m_Flags.Test<NwcpDriverFlag::IsUartUsed>())
    {
        if (m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>())
        {
            // 充電管理を停止
            m_Charger.StopChargeStateManagement();
            // Uart 接続されている場合は、Charger を Suspend してから Detach
            m_Charger.Suspend();
            Detach();
        }
        else
        {
            // Uart のセッションが確立もしくは、タイムアウトするのを待つ
        }
    }
    else
    {
        if (m_Flags.Test<NwcpDriverFlag::IsNwcpEnabled>() == true)
        {
            // 抜け状態のときは Charger の供給を停止してから終了
            m_Charger.StopPowerSupply();
        }

        // UART 接続されていない場合は、GPIO を Deactivate
        m_DetectGpioMonitor.Deactivate();

        CompleteSuspend();
    }
}

void NwcpDriver::Resume() NN_NOEXCEPT
{
    if (m_Flags.Test<NwcpDriverFlag::IsActivated>() == false || m_SuspendState == SuspendState_Resumed)
    {
        return;
    }

    if (m_Flags.Test<NwcpDriverFlag::IsNwcpEnabled>() == true)
    {
        m_Flags.Set<NwcpDriverFlag::IsUartAllowed>();
    }
    m_SuspendState = SuspendState_Resumed;

    // Gpio の監視を有効
    EnableGpioMonitoring();

    // 起動直後の状態をチェック
    UpdateGpioState();
}

void NwcpDriver::SetRebootEnabled(bool enabled) NN_NOEXCEPT
{
    m_Flags.Set<NwcpDriverFlag::IsRebootEnabled>(enabled);
}

::nn::TimeSpan NwcpDriver::GetInterval() NN_NOEXCEPT
{
    return m_UartDriver.GetInterval();
}

void NwcpDriver::SetReport(uint8_t* pBuffer, size_t size, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    m_UartDriver.SetReport(pBuffer, size, pListener);
}

void NwcpDriver::GetReport(uint8_t reportId, IHidListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    m_UartDriver.GetReport(reportId, pListener);
}

void NwcpDriver::SetFirmwareUpdateModeEnabled(bool enabled) NN_NOEXCEPT
{
    m_UartDriver.SetFirmwareUpdateModeEnabled(enabled);
}

void NwcpDriver::SetFastModeEnabled(bool enabled) NN_NOEXCEPT
{
    m_UartDriver.SetFastModeEnabled(enabled);
}

PadState NwcpDriver::GetFiftyPadState() NN_NOEXCEPT
{
    return m_UartDriver.GetFiftyPadState();
}

bool NwcpDriver::IsSuspended() NN_NOEXCEPT
{
    return (m_SuspendState == SuspendState_Suspended);
}

void NwcpDriver::Detach() NN_NOEXCEPT
{
    if (m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>() == true)
    {
        if (m_Flags.Test<NwcpDriverFlag::IsRebootEnabled>() == true)
        {
            m_Flags.Set<NwcpDriverFlag::IsRebootEnabled>(false);
            m_UartDriver.DetachUartByReboot();
        }
        else
        {
            m_UartDriver.DetachUart();
        }
    }
}

bool NwcpDriver::IsDeviceAttached() NN_NOEXCEPT
{
    return m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>();
}

void NwcpDriver::SetNwcpEnabled(bool enabled) NN_NOEXCEPT
{
    if (m_Flags.Test<NwcpDriverFlag::IsNwcpEnabled>() != enabled)
    {
        m_Flags.Set<NwcpDriverFlag::IsNwcpEnabled>(enabled);

        if (m_Flags.Test<NwcpDriverFlag::IsUartAllowed>() != enabled)
        {
            m_Flags.Set<NwcpDriverFlag::IsUartAllowed>(enabled);

            if (m_Flags.Test<NwcpDriverFlag::IsUartAllowed>() == true)
            {
                m_Flags.Set<NwcpDriverFlag::IsNwcpDeviceActivated>(false);
                UpdateGpioState();
            }
        }
    }
}

AwakeTriggerReason NwcpDriver::GetAwakeTriggerReason() NN_NOEXCEPT
{
    if (m_Flags.Test<NwcpDriverFlag::IsUartUsed>() == false)
    {
        // Uart有効になっていない場合(デバイス検出していない場合は、None)
        return AwakeTriggerReason_None;
    }

    NN_SDK_REQUIRES(m_Flags.Test<NwcpDriverFlag::ReadAwakeTriggerReason>() == false);

    m_Flags.Set<NwcpDriverFlag::ReadAwakeTriggerReason>();
    ::nn::os::InitializeLightEvent(&m_AwakeTriggerReasonReadCompleteEvent, false, ::nn::os::EventClearMode_ManualClear);

    if (m_Flags.Test<NwcpDriverFlag::IsNwcpDeviceActivated>())
    {
        // Uart 接続済みの場合は、Reason の読み出しを行う
        m_UartDriver.StartReadingAwakeTriggerReason(&m_AwakeTriggerReasonReadCompleteEvent);
    }
    else
    {
        // Uart 接続中の場合は、Uart 接続完了後に Reason の読み出しを行う
    }

    // タイムアウトつきで Read の完了を待つ
    ::nn::os::TimedWaitLightEvent(&m_AwakeTriggerReasonReadCompleteEvent, ::nn::TimeSpan::FromSeconds(5));
    ::nn::os::FinalizeLightEvent(&m_AwakeTriggerReasonReadCompleteEvent);

    m_Flags.Set<NwcpDriverFlag::ReadAwakeTriggerReason>(false);
    return m_UartDriver.GetAwakeTriggerReason();
}

void NwcpDriver::TryIsCtsLow() NN_NOEXCEPT
{
    // Pinmux を Gpio に変更
    SwitchPinmux(false);

    // Cts Gpio を Activate
    m_UartCtsGpioMonitor.Activate(&m_UartCtsGpioEvent);

    // UartCts が High になっていればスリープ完了
    if (m_UartCtsGpioMonitor.IsLow() == false)
    {
        // GpioMonitor を終了
        m_UartCtsGpioMonitor.Deactivate();
        // Suspend 処理完了
        CompleteSuspend();
    }
    else
    {
        // m_UartCtsGpioEvent のシグナルで、CTS が下りるのを待つ
        // Timeout だけ設定
        ::nn::os::StartOneShotTimerEvent(&m_UartCtsTimeoutEvent, UartCtsTimeout);
    }
}

void NwcpDriver::SwitchPinmux(bool useUart) NN_NOEXCEPT
{
    nn::pinmux::PinmuxSession session;
    nn::pinmux::OpenSession(&session, m_PinmuxName);
    nn::pinmux::SetPinAssignment(&session,
        useUart ? nn::pinmux::PinAssignment_ExtConTxUart : nn::pinmux::PinAssignment_ExtConTxGpio);
    nn::pinmux::CloseSession(&session);
}

void NwcpDriver::CompleteSuspend() NN_NOEXCEPT
{
    // Timer を解除
    ::nn::os::StopTimerEvent(&m_UartCtsTimeoutEvent);
    ::nn::os::ClearTimerEvent(&m_UartCtsTimeoutEvent);

    m_SuspendState = SuspendState_Suspended;

    // サスペンド処理の完了を通知
    ::nn::os::SignalLightEvent(m_pSuspendCompleteEvent);
}

void NwcpDriver::QueueNwcpAttachEvent(bool attach, ::nn::bluetooth::Address address) NN_NOEXCEPT
{
    // 起動時以外は Inquiry 完了した時点で通知を出す
    if (m_Flags.Test<NwcpDriverFlag::IsFromBoot>() == false)
    {
        RailUpdateEventType type = RailUpdateEventType_None;
        switch (m_SupportedDeviceType)
        {
        case NwcpDeviceType_JoyLeft:
            type = (attach) ? RailUpdateEventType_LeftAttached : RailUpdateEventType_LeftDetached;
            break;
        case NwcpDeviceType_JoyRight:
            type = (attach) ? RailUpdateEventType_RightAttached : RailUpdateEventType_RightDetached;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        for (auto& event : m_RailUpdateEventQueue)
        {
            if (event.eventType == RailUpdateEventType_None)
            {
                event.eventType = type;
                event.address = address;
                if (m_pRailUpdateEvent != nullptr)
                {
                    nn::os::SignalSystemEvent(m_pRailUpdateEvent);
                }
                return;
            }
        }

        // ここまで来た場合はキューがいっぱいだった
        // キューがいっぱいのときは古いものから捨てる
        for (int i = RailUpdateEventQueueSize - 1; i > 0; i--)
        {
            m_RailUpdateEventQueue[i - 1] = m_RailUpdateEventQueue[i];
        }
        m_RailUpdateEventQueue[RailUpdateEventQueueSize - 1].eventType = type;
        m_RailUpdateEventQueue[RailUpdateEventQueueSize - 1].address = address;
        if (m_pRailUpdateEvent != nullptr)
        {
            nn::os::SignalSystemEvent(m_pRailUpdateEvent);
        }
    }
}

}}} // namespace nn::xcd::detail
