﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_NfcCommandHandler.h"
#include "xcd_TeraCommon.h"
#include "xcd_TeraNfc.h"

//#define VERBOSE

/*
 * 引数に指定された文字列を、NfcCommandHandler のログとして出力します。
 */
#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_NFC_LOG(...)             NN_DETAIL_XCD_INFO("[NfcCmd] " ##__VA_ARGS__)
#else
#define NN_XCD_NFC_LOG(format, ...)     NN_DETAIL_XCD_INFO("[NfcCmd] " format, ##__VA_ARGS__)
#endif

/*
 * VERBOSE モード時のみ、引数に指定された文字列を、NfcCommandHandler のログとして出力します。
 */
#ifdef VERBOSE
#define NN_XCD_NFC_LOG_VERBOSE(...)     NN_XCD_NFC_LOG(__VA_ARGS__)
#else
#define NN_XCD_NFC_LOG_VERBOSE(...)     static_cast<void>(0)
#endif

/*
 * コマンドキューが空いているか確認し、空いていなければ Busy を返します。
 */
#define NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL() \
    NN_RESULT_THROW_UNLESS(!m_CommandQueue.IsFull(), nn::xcd::ResultMcuBusy())

namespace nn { namespace xcd { namespace detail {

namespace
{

// 単発系コマンドのタイムアウト時間
const nn::TimeSpan SingleCommandTimeout = nn::TimeSpan::FromMilliSeconds(500);

// 一時的な NOP コマンド送信のもととなる固定データ
const NfcCommandType NfcNopCommandData =
{
    // Header
    {
        BluetoothMode_Nfc,      // Bluetooth mode
        CommandProtocol_Nfc,    // プロトコル
        NfcCommand_Nop          // NFC コマンドコード
    }

    // 以降のメンバ (SQN, ACKN, ペイロード, ペイロード長) は All 0.
    // ACKN はコマンド送出時に適切に設定する。
};

/**
 * @brief   MIFARE コマンド用の NUID を設定
 */
void SetMifareNuid(char* pOutNuid, size_t outSize, const nn::xcd::NfcTagId& tagId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNuid);
    NN_SDK_REQUIRES_GREATER_EQUAL(outSize, MifareNuidLength);

    // 渡されたバッファや UID の長さを超えないようにする
    // (NUID は 4 バイトだが、Release ビルドで事前条件をすり抜けた場合のために決め打ちではコピーしない)
    auto idLength = std::min<size_t>(
        std::min<size_t>(tagId.length, MifareNuidLength),
        outSize);

    // NUID は UID の末尾を使用
    std::memcpy(
        pOutNuid,
        tagId.uid + (tagId.length - idLength),
        idLength);
}

}  // anonymous

void NfcCommandHandler::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_SendTimeoutCounter.Stop();
    m_SequenceNumber  = 0;
    m_AckNumber       = 0;
    m_ProcessingState = CommandProcessingState::Idle;
    m_ExpectedState   = NfcState_CommandWaiting;
    CancelAllCommand();
}

void NfcCommandHandler::CancelAllCommand() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_CommandQueue.Clear();
    m_ProcessingState = CommandProcessingState::Idle;
}

void NfcCommandHandler::StartWaitingReceive(NfcState expectedState) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_ProcessingState = CommandProcessingState::WaitingReceiveFinish;
    m_ExpectedState   = expectedState;
}

bool NfcCommandHandler::CheckReceiveFinished(InternalNfcEventType* pOutEventType, NfcState state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);

    if (m_ProcessingState != CommandProcessingState::WaitingReceiveFinish)
    {
        return false;
    }

    // 期待するステートへの遷移を以て完了とする
    if (state != m_ExpectedState)
    {
        return false;
    }

    switch (m_ExpectedState)
    {
    case NfcState_ReadFinish:
        *pOutEventType = InternalNfcEventType::DataReceived;
        break;
    case NfcState_PassThruFinish:
        *pOutEventType = InternalNfcEventType::PassThru;
        break;
    case NfcState_MifareFinish:
        *pOutEventType = InternalNfcEventType::Mifare;
        break;
    default:
        // m_ExpectedState は内部で設定するため、ここには入らない
        NN_UNEXPECTED_DEFAULT;
    }

    m_ProcessingState  = CommandProcessingState::Idle;
    m_ExpectedState    = NfcState_CommandWaiting;

    return true;
}

bool NfcCommandHandler::CheckPacketTransferSuccess(
    const NfcPackedPacketHeader& packetHeader,
    const NfcPackedCommonResponseHeader& commonHeader) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    CheckSendSuccess(packetHeader, commonHeader);

    return CheckReceiveSuccess(packetHeader);
}

void NfcCommandHandler::CheckSendSuccess(
    const NfcPackedPacketHeader& packetHeader,
    const NfcPackedCommonResponseHeader& commonHeader) NN_NOEXCEPT
{
    if (m_CurrentCommand.IsResendRequired())
    {
        // 要再送コマンド

        if (IsMultiPacketSendSuccess(packetHeader.ackNumber))
        {
            if (packetHeader.ackNumber == GetCurrentCommandPacketCount())
            {
                // 最終パケットの送信が完了していればコマンド完了
                FinishSendingPacket();
            }
        }
        else
        {
            NN_XCD_NFC_LOG_VERBOSE(
                "Send packet lost (Console SQN: %d, Tera ACKN: %d)\n",
                GetLastSequenceNumber(),
                packetHeader.ackNumber);

            // ACKN の次のパケットから再送する
            m_SequenceNumber = packetHeader.ackNumber;
        }
    }
    else if (m_ProcessingState == CommandProcessingState::WaitingSendFinish)
    {
        // 送信完了待ち

        auto nfcState = static_cast<NfcState>(commonHeader.state);
        if (IsExpectedNfcStateForSendFinish(m_CurrentCommand.header.command, nfcState))
        {
            // 期待したステートに遷移していればコマンド完了
            NN_XCD_NFC_LOG_VERBOSE("Command %d is completed. Current state is %d (%s)\n",
                m_CurrentCommand.header.command,
                commonHeader.state,
                GetNfcStateString(nfcState));
            FinishSendingPacket();
        }
    }
}

bool NfcCommandHandler::CheckReceiveSuccess(
    const NfcPackedPacketHeader& packetHeader) NN_NOEXCEPT
{
    if (packetHeader.sequenceNumber == 0)
    {
        return true;
    }

    if (!IsReceiveSuccess(
        packetHeader.sequenceNumber,
        packetHeader.IsEndOfCommand()))
    {
        NN_XCD_NFC_LOG_VERBOSE(
            "Receive packet lost (Tera SQN: %d, Console ACKN: %d)\n",
            packetHeader.sequenceNumber,
            m_AckNumber);

        // 受信失敗時はパケットを解析しない
        return false;
    }

    NN_XCD_NFC_LOG_VERBOSE(
        "Tera SQN: %d, Console ACKN: %d\n",
        packetHeader.sequenceNumber,
        m_AckNumber);

    // 受信した分割パケットに対する ACK 応答
    m_AckNumber = packetHeader.sequenceNumber;

    return true;
}

void NfcCommandHandler::FinishSendingPacket() NN_NOEXCEPT
{
    switch (m_CurrentCommand.header.command)
    {
    case NfcCommand_ReadStart:
    case NfcCommand_PassThru:
    case NfcCommand_MifareRequest:
        if (m_ProcessingState != CommandProcessingState::WaitingReceiveFinish)
        {
            // NFC_READ_START, NFC_PASS_THRU_REQ, NFC_MIFARE_REQ の場合は受信完了を待機
            m_ProcessingState = CommandProcessingState::NeedWaitReceive;
        }
        break;
    default:
        m_ProcessingState = CommandProcessingState::Idle;
        break;
    }

    m_SendTimeoutCounter.Stop();
}

bool NfcCommandHandler::IsMultiPacketSendSuccess(int mcuAckNumber) const NN_NOEXCEPT
{
    if (GetLastSequenceNumber() <= mcuAckNumber)
    {
        // 前回送信時の SQN と今回の ACKN が一致する場合は成功
        return true;
    }
    // 複数パケット送信時は、XCD の実装および通信の仕様上、通信が正常であっても
    // 前回の SQN と今回の ACKN が 1 or 2 ズレるため、その状態も成功として扱う
    else if (GetLastSequenceNumber() >= 1 &&
        GetLastSequenceNumber() - 1 <= mcuAckNumber)
    {
        return true;
    }
    else if (GetLastSequenceNumber() >= 2 &&
        GetLastSequenceNumber() - 2 <= mcuAckNumber)
    {
        return true;
    }

    return false;
}

bool NfcCommandHandler::IsReceiveSuccess(int mcuSequenceNumber, bool isLastPacket) const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (mcuSequenceNumber == m_AckNumber + 1)
    {
        // SQN の方が 1 だけ進んだ状態なら受信成功
        return true;
    }
    else if (isLastPacket && mcuSequenceNumber == m_AckNumber)
    {
        // 最終パケットは SQN == ACKN の場合も成功
        return true;
    }

    return false;
}

void NfcCommandHandler::PrepareNextCommand(bool* pOutNeedsClearTagData, NfcState currentState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsClearTagData);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutNeedsClearTagData = false;

    // 現在のコマンドの送信が完了したら、次のコマンドを送信準備
    if (m_ProcessingState == CommandProcessingState::Idle)
    {
        if (IsNextCommandReady(currentState))
        {
            if (!m_CommandQueue.TryDequeue(&m_CurrentCommand))
            {
                // 送信予約されたコマンドがなければ NOP を送る
                ConfigureNfcNopCommand(&m_CurrentCommand);
            }
        }
        else
        {
            // コマンド送信準備ができていない間は、予約されたコマンドを保持したまま NOP を送る
            ConfigureNfcNopCommand(&m_CurrentCommand);
        }
        SetupCommandForSend(pOutNeedsClearTagData, m_CurrentCommand);
    }
    else if (m_SendTimeoutCounter.IsExpired())
    {
        // コマンド送信がタイムアウトした場合は再送する
        SetupCommandForSend(pOutNeedsClearTagData, m_CurrentCommand);
    }
}

bool NfcCommandHandler::IsNextCommandReady(NfcState currentState) const NN_NOEXCEPT
{
    // コマンド処理中判定
    const auto IsBusy = [this]
    {
        return m_ProcessingState == CommandProcessingState::NeedWaitReceive ||
            m_ProcessingState == CommandProcessingState::NeedWaitSend ||
            m_ProcessingState == CommandProcessingState::WaitingReceiveFinish ||
            m_ProcessingState == CommandProcessingState::WaitingSendFinish;
    };

    if (currentState == NfcState_Initializing)
    {
        // NFC の初期化中
        return false;
    }
    else if (IsBusy())
    {
        // 次のコマンドを送信可能になっていない
        return false;
    }

    return true;
}

void NfcCommandHandler::SetupCommandForSend(bool* pOutNeedsClearTagData, const NfcCommandType& command) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsClearTagData);

    *pOutNeedsClearTagData = false;

    m_CurrentCommand = command;
    m_SendTimeoutCounter.Stop();

    // 新しいコマンドの送信時は SQN をクリア
    ClearSequenceNumber();

    if (command.header.command == NfcCommand_Nop)
    {
        m_ProcessingState = CommandProcessingState::Idle;
        return;
    }

    // 新たにコマンドを発行するときは ACKN を初期化
    m_AckNumber = 0;

    switch (command.header.command)
    {
    case NfcCommand_WriteStart:
    case NfcCommand_PassThru:
    case NfcCommand_MifareKeyStartWrite:
    case NfcCommand_MifareRequest:
        {
            m_ProcessingState = CommandProcessingState::SendingMultiPacket;

            // 以前受信したタグのデータは無効にする
            *pOutNeedsClearTagData = true;
        }
        break;
    default:
        {
            m_ProcessingState = CommandProcessingState::NeedWaitSend;
            m_SendTimeoutCounter.Start(SingleCommandTimeout);

            if (command.header.command == NfcCommand_ReadStart)
            {
                // 以前受信したタグのデータは無効にする
                *pOutNeedsClearTagData = true;
            }
        }
        break;
    }

    m_ExpectedState = NfcState_CommandWaiting;
}

/*
 * 次に送信する NFC コマンドパケットを生成
 */
void NfcCommandHandler::CreateNextCommandPacket(
    size_t* pOutPacketSize,
    uint8_t* pOutPacketData,
    size_t maxSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPacketSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutPacketData);
    NN_SDK_REQUIRES_EQUAL(maxSize, GetTransferSize().send);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    const auto* pCommandToSend = &m_CurrentCommand;

    // 最終パケットまで送出済みか判定
    const auto IsFinalPacketSent = [&pCommandToSend, this]
    {
        return pCommandToSend->IsResendRequired() &&
            GetSequenceNumber() >= GetCurrentCommandPacketCount();
    };

    // 送信待ち中判定
    const auto IsWaitingForSent = [this]
    {
        return m_ProcessingState == CommandProcessingState::NeedWaitReceive ||
            m_ProcessingState == CommandProcessingState::WaitingSendFinish ||
            m_ProcessingState == CommandProcessingState::WaitingReceiveFinish;
    };

    if (IsFinalPacketSent())
    {
        // 最終パケットまで送出済みの場合は一時的に NOP を送る
        pCommandToSend = &NfcNopCommandData;
        NN_XCD_NFC_LOG_VERBOSE("Final packet was sent. Temporarily sending NOP.\n");
    }
    else if (IsWaitingForSent())
    {
        // 送信待ち中は一時的に NOP を送る
        pCommandToSend = &NfcNopCommandData;
        NN_XCD_NFC_LOG_VERBOSE("Waiting for sent. Temporarily sending NOP.\n");
    }
    else if (m_ProcessingState == CommandProcessingState::NeedWaitSend)
    {
        m_ProcessingState = CommandProcessingState::WaitingSendFinish;
    }

    // パケットの中身を生成
    *pOutPacketSize = CreateCommandPacketPayload(pOutPacketData, maxSize, *pCommandToSend);
}

size_t NfcCommandHandler::CreateCommandPacketPayload(
    uint8_t* pOutPacketData,
    size_t maxSize,
    const NfcCommandType& commandToSend) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPacketData);

    const auto& header = commandToSend.header;
    NfcPackedPacketHeader packedHeader = {};
    packedHeader.SetBluetoothMode(header.bluetoothMode);
    packedHeader.SetProtocol(header.protocol);
    packedHeader.command   = header.command;
    packedHeader.ackNumber = static_cast<uint8_t>(m_AckNumber);
    packedHeader.payloadAttribute.Set<NfcPacketPayloadAttributePack::Reserved>(0);

    const size_t payloadLengthMax = maxSize - sizeof(NfcPackedPacketHeader) - 1;
    size_t nextPayloadOffset;
    if (commandToSend.IsResendRequired())
    {
        UpdateSequenceNumber();
        packedHeader.sequenceNumber = GetSequenceNumber();

        // 次に送るペイロードの先頭位置を算出
        nextPayloadOffset = (packedHeader.sequenceNumber - 1) * payloadLengthMax;

        // 送信するサイズを算出
        const size_t nextPayloadLength = std::min<size_t>(
            commandToSend.payloadLength - nextPayloadOffset,
            payloadLengthMax);
        packedHeader.SetPayloadLength(nextPayloadLength);
        packedHeader.SetEndOfCommand(
            packedHeader.sequenceNumber == GetCurrentCommandPacketCount());
    }
    else
    {
        // 再送不要の場合は単一パケットで全て送る
        packedHeader.sequenceNumber = 0;
        nextPayloadOffset = 0;

        // 単一パケットで payloadLengthMax を超えることはないが、念のためチェック
        const size_t nextPayloadLength = std::min<size_t>(
            commandToSend.payloadLength,
            payloadLengthMax);
        NN_SDK_REQUIRES_EQUAL(nextPayloadLength, commandToSend.payloadLength);

        packedHeader.SetPayloadLength(nextPayloadLength);
        packedHeader.SetEndOfCommand(true);
    }

    // パケットのバイト列を生成
    std::memcpy(
        reinterpret_cast<char*>(&pOutPacketData[sizeof(packedHeader)]),
        &commandToSend.payload[nextPayloadOffset],
        packedHeader.GetPayloadLength());
    std::memcpy(pOutPacketData, &packedHeader, sizeof(packedHeader));

    // PayloadLength 以降のペイロードは all 0x00
    size_t remainPayloadLength = payloadLengthMax - packedHeader.GetPayloadLength();
    if (remainPayloadLength > 0)
    {
        std::memset(
            &pOutPacketData[sizeof(packedHeader) + packedHeader.GetPayloadLength()],
            0,
            remainPayloadLength);
    }

    // 末尾に CRC8 を付与
    //  * CRC は BT mode + protocol と CRC の 2 バイトを除いたパケット全体に対して計算
    pOutPacketData[maxSize - 1] = CalcCrc8(pOutPacketData + 1, maxSize - 2);

    return maxSize;
}

nn::Result NfcCommandHandler::TryRegisterNextCommand(NfcCommandType& command) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_CommandQueue.TryEnqueue(command),
        nn::xcd::ResultMcuBusy());

    NN_RESULT_SUCCESS;
}

nn::Result NfcCommandHandler::StartDiscovery(const NfcDiscoveryParameter& nfcDiscoveryParam) NN_NOEXCEPT
{
    NN_UNUSED(nfcDiscoveryParam);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_START_DISCOVERY コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_StartDiscovery);

    // PollMask の値をチェックするべき？

    auto* payload = command.payload;

    // [0] ポーリング対象タグのマスク
    payload[0] = static_cast<char>(nfcDiscoveryParam.pollingMask);

    // [1..2] タグ検出までのタイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(
        payload + 1,
        sizeof(uint16_t),
        nfcDiscoveryParam.activationTimeout);

    // [3..4] ポーリング間隔 (Little endian)
    SetLittleEndianValue<char, uint16_t>(
        payload + 3,
        sizeof(uint16_t),
        nfcDiscoveryParam.discoveryPeriod);

    command.payloadLength = 5;

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::StopDiscovery() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NfcCommandType command = {};

    ConfigureNfcCommand(&command, NfcCommand_StopDiscovery);

    // ペイロードなし
    command.payloadLength = 0;

    // StopDiscovery は優先して発行するため、キューをクリアして追加する
    m_CommandQueue.Clear();
    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    // 強制的にコマンド発行可能状態にする
    m_ProcessingState = CommandProcessingState::Idle;
    m_SendTimeoutCounter.Stop();

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::StartNtagRead(const NtagReadParameter& ntagReadParam) NN_NOEXCEPT
{
    const int PayloadUidLengthMax = 7;
    NN_SDK_REQUIRES_LESS_EQUAL(ntagReadParam.tagId.length, PayloadUidLengthMax);
    NN_SDK_REQUIRES_MINMAX(ntagReadParam.blockCount, 1, NtagReadBlockCountMax);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_READ_START コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_ReadStart);

    auto* payload = command.payload;

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(payload, sizeof(uint16_t), ntagReadParam.timeoutMsec);

    // [2..8] 読み取り対象タグの UID
    std::memcpy(
        &payload[2],
        ntagReadParam.tagId.uid,
        std::min<size_t>(ntagReadParam.tagId.length, PayloadUidLengthMax));

    // [9] パスワード認証要否
    payload[9]  = ntagReadParam.isPasswordRequired ? 1 : 0;

    // [10] 読み込みコマンド (読み込むデータブロック) の数
    const auto blockCount = std::min<int>(
        std::max<int>(ntagReadParam.blockCount, 0),
        NtagReadBlockCountMax);
    payload[10] = static_cast<char>(blockCount);

    // [11..] 読み込むデータブロックのアドレス
    for (int i = 0; i < blockCount; ++i)
    {
        auto& address = ntagReadParam.addresses[i];
        NN_SDK_REQUIRES_RANGE(address.endPage - address.startPage, 0, NtagReadBlockPageCountMax);

        auto* pAddressStart = &payload[11 + i * 2];
        pAddressStart[0] = address.startPage;
        pAddressStart[1] = address.endPage;
    }

    command.payloadLength = 19;

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::StartNtagWrite(const NtagWriteParameter& ntagWriteParam) NN_NOEXCEPT
{
    const int PayloadUidLengthMax = 7;
    NN_SDK_REQUIRES_MINMAX(ntagWriteParam.tagId.length, 0, PayloadUidLengthMax);
    NN_SDK_REQUIRES_MINMAX(ntagWriteParam.ntagWriteData.blockCount, 1, NtagWriteBlockCountMax);
    for (int i = 0; i < ntagWriteParam.ntagWriteData.blockCount; ++i)
    {
        NN_SDK_REQUIRES_ALIGNED(ntagWriteParam.ntagWriteData.dataBlocks[i].dataSize, 4);
        NN_SDK_REQUIRES_MINMAX(ntagWriteParam.ntagWriteData.dataBlocks[i].dataSize, 4ul, NtagWriteBlockSizeMax);
    }

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_WRITE_START コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_WriteStart);

    const char ActivationDataAddress = 0x04;
    auto* payload = command.payload;

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(payload, sizeof(uint16_t), ntagWriteParam.timeoutMsec);

    // [2..8] 書き込み対象タグの UID
    std::memcpy(
        &payload[2],
        ntagWriteParam.tagId.uid,
        std::min<size_t>(ntagWriteParam.tagId.length, PayloadUidLengthMax));

    // [9] パスワード認証要否
    payload[9]  = ntagWriteParam.isPasswordRequired ? 1 : 0;

    // [10] Type2 タグバージョン
    payload[10] = ntagWriteParam.type2TagVersion;

    // [11] アクティベーション要否
    payload[11] = ntagWriteParam.ntagWriteData.isActivationNeeded ? 1 : 0;

    // [12] アクティベーション情報のアドレス
    payload[12] = ActivationDataAddress;

    // [13..16] アクティベーション情報クリア時の値
    std::memcpy(&payload[13], ntagWriteParam.ntagWriteData.clearData, NtagActivationAreaSize);

    // [17..20] アクティベーション成功時の値
    std::memcpy(&payload[17], ntagWriteParam.ntagWriteData.activationData, NtagActivationAreaSize);

    // [21] 書き込みブロック数
    const auto blockCount = std::min<int>(
        std::max<int>(ntagWriteParam.ntagWriteData.blockCount, 0),
        NtagWriteBlockCountMax);
    payload[21] = static_cast<char>(blockCount);

    // [22..] 書き込み先アドレスとデータ
    size_t totalPayloadSize = 22;
    for (int i = 0; i < blockCount; ++i)
    {
        auto& block = ntagWriteParam.ntagWriteData.dataBlocks[i];
        auto* pDataStart = &payload[totalPayloadSize];

        totalPayloadSize += block.dataSize + 2;
        NN_SDK_REQUIRES_LESS_EQUAL(totalPayloadSize, PayloadLengthMax);

        pDataStart[0] = block.startPageAddress;
        pDataStart[1] = static_cast<char>(block.dataSize);
        std::memcpy(
            &pDataStart[2],
            block.data,
            std::min<size_t>(block.dataSize, NtagWriteBlockSizeMax));
    }

    // 使用しないブロックのアドレスとサイズは 0
    for (int i = blockCount; i < NtagWriteBlockCountMax; i++)
    {
        auto* pDataStart = &payload[totalPayloadSize];

        totalPayloadSize += 2;
        NN_SDK_REQUIRES_LESS_EQUAL(totalPayloadSize, PayloadLengthMax);

        pDataStart[0] = 0;  // アドレス
        pDataStart[1] = 0;  // サイズ
    }

    command.payloadLength = std::min<size_t>(totalPayloadSize, PayloadLengthMax);

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::SendRawData(const NfcPassThruParameter& passThruParam) NN_NOEXCEPT
{
    const int PayloadUidLengthMax = 7;
    NN_SDK_REQUIRES_MINMAX(passThruParam.tagId.length, 0, NfcUidLengthMax);
    NN_SDK_REQUIRES_LESS_EQUAL(passThruParam.sendDataSize, NfcPassThruDataSizeMax);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_PASS_THRU_REQ コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_PassThru);

    auto* payload = command.payload;

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(payload, sizeof(uint16_t), passThruParam.timeoutMsec);

    // [2..8] 対象タグの UID
    // 指定された UID のうち先頭 7 バイトまでを使用する。
    std::memcpy(
        &payload[2],
        passThruParam.tagId.uid,
        std::min<size_t>(passThruParam.tagId.length, PayloadUidLengthMax));

    auto dataSize = std::min<size_t>(passThruParam.sendDataSize, NfcPassThruDataSizeMax);

    // [9..10] 送信データサイズ (Little endian)
    SetLittleEndianValue<char, uint16_t>(
        payload + 9,
        sizeof(uint16_t),
        static_cast<uint16_t>(dataSize));

    // [11..] 送信データ
    auto* pDataStart = &payload[11];
    std::memcpy(pDataStart, passThruParam.sendData, dataSize);

    command.payloadLength = 11 + dataSize;

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::RegisterMifareKey(const MifareKeyWriteParameter& keyWriteParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(keyWriteParameter.keyCount, 1, MifareKeyCountMax);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_MIFARE_KEY_START_WRITE コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_MifareKeyStartWrite);

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(command.payload, sizeof(uint16_t), keyWriteParameter.timeoutMsec);

    // [2..] 鍵リスト
    auto* keyList = reinterpret_cast<NfcMifareKeyListForWrite*>(command.payload + 2);

    keyList->keyCount = std::min<uint8_t>(keyWriteParameter.keyCount, MifareKeyCountMax);
    for (decltype(keyList->keyCount) i = 0; i < keyList->keyCount; i++)
    {
        std::memcpy(keyList->keys[i], keyWriteParameter.keys[i].value, MifareEncryptedKeyLength);
    }

    command.payloadLength = 3 + keyList->keyCount * MifareEncryptedKeyLength;

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::ClearMifareKey(const MifareKeyClearParameter& keyClearParameter) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_MIFARE_KEY_START_WRITE コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_MifareKeyStartWrite);

    auto* payload = command.payload;

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(payload, sizeof(uint16_t), keyClearParameter.timeoutMsec);

    // [2] キー数 を 0 にすると消去扱いになる
    payload[2] = 0;

    command.payloadLength = 3;

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::StartMifareRead(const MifareReadParameter& readParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(readParameter.blockCount, 1, MifareReadBlockCountMax);
    NN_SDK_REQUIRES_GREATER_EQUAL(
        static_cast<size_t>(readParameter.tagId.length),
        MifareNuidLength);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_MIFARE_REQ コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_MifareRequest);

    bool isRawKey = readParameter.keyFormat == MifareKeyValueFormat_Raw;
    auto* payload = command.payload;

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(payload, sizeof(uint16_t), readParameter.timeoutMsec);

    const auto blockCount = std::min<int>(readParameter.blockCount, MifareReadBlockCountMax);

    // [2] Command status
    NfcMifareCommandConfig config = {};
    config.SetWriteMode(false);
    config.SetUseRawKey(isRawKey);
    config.SetCommandCount(static_cast<uint8_t>(blockCount));
    payload[2] = config.GetValue();

    // [3..6] タグの NUID
    SetMifareNuid(payload + 3, MifareNuidLength, readParameter.tagId);

    // [7..] データブロック指定
    int totalPayloadLength = 3 + MifareNuidLength;
    for (int i = 0; i < blockCount; i++)
    {
        const auto& block = readParameter.blocks[i];

        totalPayloadLength += SetMifareCommonBlockParameter(
            &payload[totalPayloadLength],
            isRawKey,
            block.key,
            block.blockAddress);
    }

    command.payloadLength = totalPayloadLength;

    return TryRegisterNextCommand(command);
}

nn::Result NfcCommandHandler::StartMifareWrite(const MifareWriteParameter& writeParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(writeParameter.blockCount, 1, MifareReadBlockCountMax);
    NN_SDK_REQUIRES_GREATER_EQUAL(
        static_cast<size_t>(writeParameter.tagId.length),
        MifareNuidLength);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_XCD_CHECK_COMMAND_QUEUE_NOT_FULL();

    NfcCommandType command = {};

    // NFC_MIFARE_REQ コマンドのパラメータ設定
    ConfigureNfcCommand(&command, NfcCommand_MifareRequest);

    bool isRawKey = writeParameter.keyFormat == MifareKeyValueFormat_Raw;
    auto* payload = command.payload;

    // [0..1] タイムアウト (Little endian)
    SetLittleEndianValue<char, uint16_t>(payload, sizeof(uint16_t), writeParameter.timeoutMsec);

    const auto blockCount = std::min<int>(writeParameter.blockCount, MifareReadBlockCountMax);

    // [2] Command status
    NfcMifareCommandConfig config = {};
    config.SetWriteMode(true);
    config.SetUseRawKey(isRawKey);
    config.SetCommandCount(static_cast<uint8_t>(blockCount));
    payload[2] = config.GetValue();

    // [3..6] タグの NUID
    SetMifareNuid(payload + 3, MifareNuidLength, writeParameter.tagId);

    // [7..] データブロック指定
    int totalPayloadLength = 3 + MifareNuidLength;
    for (int i = 0; i < blockCount; i++)
    {
        const auto& block = writeParameter.blocks[i];
        auto* pBlock = &payload[totalPayloadLength];

        auto parameterLength = SetMifareCommonBlockParameter(
            pBlock,
            isRawKey,
            block.key,
            block.blockAddress);

        // パラメータに続けて書き込むデータを格納
        std::memcpy(pBlock + parameterLength, block.data, MifareBlockBytes);

        totalPayloadLength += parameterLength + MifareBlockBytes;
    }

    command.payloadLength = totalPayloadLength;

    return TryRegisterNextCommand(command);
}

}}} // namespace nn::xcd::detail
