﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/gpio/gpio.h>
#include <nn/result/result_HandlingUtility.h>

#include "xcd_GpioMonitorTask-os.horizon.h"

namespace nn { namespace xcd { namespace detail {


GpioMonitorTask::GpioMonitorTask(nn::gpio::GpioPadName padName) NN_NOEXCEPT :
    m_PadName(padName),
    m_IsGpioEnabled(false)
{
    // 何もしない
}

GpioMonitorTask::~GpioMonitorTask() NN_NOEXCEPT
{
    // 何もしない
}

bool GpioMonitorTask::Activate(nn::os::LightEventType* pEvent, int debounceMsec) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    if (m_IsGpioEnabled == false)
    {
        m_pUpdateEvent = pEvent;

        nn::gpio::Initialize();

        nn::gpio::OpenSession(&m_GpioSession, m_PadName);

        nn::gpio::SetDirection(&m_GpioSession, nn::gpio::Direction_Input);

        nn::gpio::SetInterruptMode(&m_GpioSession, nn::gpio::InterruptMode_AnyEdge);

        if (debounceMsec > 0)
        {
            // 一度 0ms で設定し値を確定させる
            nn::gpio::SetDebounceTime(&m_GpioSession, 0);
            nn::gpio::SetDebounceEnabled(&m_GpioSession, true);
            // 目的の Debounce 時間を設定
            nn::gpio::SetDebounceTime(&m_GpioSession, debounceMsec);
        }
        else
        {
            nn::gpio::SetDebounceEnabled(&m_GpioSession, false);
        }

        // Interrupt を bind する。
        if (nn::gpio::BindInterrupt(&m_Event, &m_GpioSession).IsFailure() == true)
        {
            return false;
        }

        GetTaskManager().RegisterPeriodicTask(this);

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&m_GpioSession);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&m_GpioSession, true);

        m_IsGpioEnabled = true;
    }

    return true;
}

void GpioMonitorTask::Deactivate() NN_NOEXCEPT
{
    if (m_IsGpioEnabled == true)
    {
        GetTaskManager().UnregisterPeriodicTask(this);
        nn::gpio::UnbindInterrupt(&m_GpioSession);
        nn::gpio::SetDebounceEnabled(&m_GpioSession, false);
        nn::gpio::CloseSession(&m_GpioSession);
        nn::gpio::Finalize();
        m_IsGpioEnabled = false;
    }
}

void GpioMonitorTask::EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWaitHolder);
}

void GpioMonitorTask::PeriodicEventFunction() NN_NOEXCEPT
{
    if (nn::os::TryWaitSystemEvent(&m_Event))
    {
        nn::os::ClearSystemEvent(&m_Event);

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&m_GpioSession);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&m_GpioSession, true);

        // Gpio の変化を上位に通知する
        nn::os::SignalLightEvent(m_pUpdateEvent);
    }
}

bool GpioMonitorTask::IsLow() NN_NOEXCEPT
{
    return (nn::gpio::GetValue(&m_GpioSession) == nn::gpio::GpioValue_Low);
}

}}} // namespace nn::xcd::detail
