﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/btm/user/btm_UserTypes.h>

#include "xcd_BleGattUuids.h"
#include "../xcd_IBleCommandListener.h"
#include "xcd_IBleNbatClient.h"
#include "xcd_IBleGattClient-hardware.nx.h"
#include "xcd_BleGattClientCommon-hardware.nx.h"

namespace nn { namespace xcd { namespace detail{

//!< Nintendo Hid over Gatt サービスのクライアント
class BleNbatClient final : IBleGattClient, public IBleNbatClient
{
    NN_DISALLOW_MOVE(BleNbatClient);
    NN_DISALLOW_COPY(BleNbatClient);

private:
    //!< GATT クライアントで取り扱う Characteristic のリスト
    enum CharacteristicIndex
    {
        CharacteristicIndex_BatteryLevel,
        CharacteristicIndex_BatteryState,

        CharacteristicIndex_Max
    };
    //!< Characteristic にアクセスするための情報
    GattClientAttributeParameter m_CharacteristicParameterList[CharacteristicIndex_Max];

    //!< GATT クライアントで取り扱う Descriptor のリスト
    enum DescriptorIndex
    {
        DescriptorIndex_BatteryLevel_ClientConfig,
        DescriptorIndex_BatteryState_ClientConfig,

        DescriptorIndex_Max
    };
    //!< Descriptor にアクセスするための情報
    GattClientAttributeParameter m_DescriptorParameterList[DescriptorIndex_Max];

    //!< Activate されているかどうか
    bool m_Activated;

    //!< BLE デバイスのコネクションハンドル
    uint32_t m_ConnectionHandle;
    //!< GATT サービスのハンドル
    uint16_t m_ServiceHandle;

    //!< GattOperation のキュー
    GattOperationConfig m_OperationQueue[MaxOperationCount];
    //!< キューにある GattOperation の数
    int m_OperationCount;
    //!< キューの先頭
    int m_OperationQueueHead;
    //!< キューの最後尾
    int m_OperationQueueTail;

public:
    BleNbatClient() NN_NOEXCEPT;
    virtual ~BleNbatClient() NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT サーバからの応答のハンドラ
    virtual bool GattOperationCompletedHandler(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT クライアントを有効にする
    virtual void Activate(uint32_t connectionHandle, const nn::btm::user::GattService& service) NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT クライアントを無効にする
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    //!< Activate されているかどうか
    virtual bool IsActivated() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Activated;
    }

    //!< サービスの UUID を返す
    virtual const nn::bluetooth::GattAttributeUuid* GetServiceUuid() NN_NOEXCEPT NN_OVERRIDE
    {
        return &NbatService.Uuid;
    }

    //!< サービスのハンドルを返す
    virtual uint16_t GetServiceHandle() NN_NOEXCEPT NN_OVERRIDE;

    //!< 電池の状態を再取得します
    virtual void UpdateBatteryStatus() NN_NOEXCEPT NN_OVERRIDE;

    //!< GATT クライアントの処理をポーリングする
    Result Proceed() NN_NOEXCEPT;

private:
    //!< GattClientAttributeParameter の値をセットする
    void SetGattClientAttributeParameterValue(bool isValid, CharacteristicIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic) NN_NOEXCEPT;

    void SetGattClientAttributeParameterValue(bool isValid, DescriptorIndex index, const nn::btm::user::GattService& service, const nn::btm::user::GattCharacteristic& characteristic, const nn::btm::user::GattDescriptor& descriptor) NN_NOEXCEPT;

    //!< GattOperation キューにタスクを積む
    Result Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& status, IBleCommandListener* pListener) NN_NOEXCEPT;

    Result Enqueue(GattOperationType opType, uint8_t authType, const GattClientAttributeParameter& status, const GattOperationPayload& payload, IBleCommandListener* pListener) NN_NOEXCEPT;

    //!< キューの先頭から要素を削除する
    void RemoveHead() NN_NOEXCEPT;

    //!< GATT_OPERATION_TYPE_READ をハンドルする
    bool HandleReadCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT;

    //!< GATT_OPERATION_TYPE_WRITE をハンドルする
    bool HandleWriteCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT;

    //!< GATT_OPERATION_TYPE_NOTIFY をハンドルする
    bool HandleNotifyCompletedEvent(nn::bluetooth::InfoFromLeGattOperationCallback* pGattResult) NN_NOEXCEPT;

    const char* GetName(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

    //!< BatteryLevel の READ 操作をハンドルする
    bool HandleReadBatteryLevel(uint8_t* pValue, int length) NN_NOEXCEPT;

    //!< BatteryState の READ 操作をハンドルする
    bool HandleReadBatteryState(uint8_t* pValue, int length) NN_NOEXCEPT;

    //!< GATT サーバから BatteryLevel を読み込む。
    void ReadBatteryLevelAsync() NN_NOEXCEPT;

    //!< GATT サーバから BatteryState を読み込む。
    void ReadBatteryStateAsync() NN_NOEXCEPT;

    //!< BatteryLevel の Indication を有効にする。
    void EnableBatteryLevelIndication() NN_NOEXCEPT;

    //!< BatteryState の Indication を有効にする。
    void EnableBatteryStateIndication() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
