﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/btm/user/btm_UserTypes.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_Event.h>
#include "xcd_BleNhogClient-hardware.nx.h"
#include "xcd_BleNbatClient-hardware.nx.h"
#include "xcd_BtmUtil-hardware.nx.h"
#include "xcd_BleHidAccessor.h"

namespace nn { namespace xcd { namespace detail{

//!< 汎用的なHidデバイスを扱うクラス
class BleHidAccessorImpl final : public BleHidAccessor
{
    NN_DISALLOW_MOVE(BleHidAccessorImpl);
    NN_DISALLOW_COPY(BleHidAccessorImpl);

private:
    //!< BLE デバイスのコネクションハンドル
    uint32_t m_ConnectionHandle;

    //!< Nintendo Hid over Gatt サービスの Accessor
    BleNhogClient m_NhogClient;

    //!< Nintendo Battery サービスの Accessor
    BleNbatClient m_NbatClient;

    //!< InputReport を格納するバッファ
    uint8_t m_Buffer[GattOperationPayloadMaxLength];
    size_t  m_ReceivedSize;

    //!< InputReport を解析する関数
    BleInputReportParserFunc m_pInputReportParserFunc;
    void* m_pInputReportParserArg;

    //!< CommandResponse を解析する関数
    BleCommandResponseParserFunc m_pCommandResponseParserFunc;
    void* m_pCommandResponseParserArg;

    //!< コマンドリスナー
    IBleCommandListener* m_pListener;

    //!< Activateされているかどうか
    bool m_Activated;

    //!< GATT Service のリスト
    nn::btm::user::GattService m_Services[nn::bluetooth::GattAttributeCountMaxClient];

    //!< GATT Service の数
    uint16_t m_ServiceNum;

    //!< BD アドレス
    ::nn::bluetooth::Address m_Address;

    //!< MTU 変更のパラメータ
    uint16_t m_ConfigureMtuValue;

    //!< BTM のリトライをするためのタイマー
    nn::os::TimerEventType m_BtmRetryEvent;

    //!< BTM のリトライで使用するフラグの集合
    BtmRetryFlagSet m_RetryFlags;

    //!< Gatt Client が Activate されているかどうか
    bool m_IsGattClientActivated;

public:
    BleHidAccessorImpl() NN_NOEXCEPT;
    virtual ~BleHidAccessorImpl() NN_NOEXCEPT NN_OVERRIDE;

    //!< Accessorを有効にする (環境依存)
    void Activate(uint32_t connectionHandle) NN_NOEXCEPT;
    //!< Accessorを無効にする
    void Deactivate() NN_NOEXCEPT;
    //!< Activateされているかどうか
    bool IsActivated() NN_NOEXCEPT
    {
        return m_Activated;
    }

    uint32_t GetConnectionHandle() NN_NOEXCEPT
    {
        return m_ConnectionHandle;
    }

    //!< サンプリングを開始
    virtual void StartInputReport(BleInputReportParserFunc func, void* pArg) NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリングを停止
    virtual void StopInputReport() NN_NOEXCEPT NN_OVERRIDE;

    //!< CommandResponse を有効化
    virtual void ActivateCommandResponse(BleCommandResponseParserFunc func, void* pArg) NN_NOEXCEPT NN_OVERRIDE;

    //!< CommandResponse を無効化
    virtual void DeactivateCommandResponse() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスを切断する
    virtual void DetachDevice() NN_NOEXCEPT NN_OVERRIDE;

    //!< コマンドリスナーを設定する
    virtual void SetListener(IBleCommandListener* pListener) NN_NOEXCEPT NN_OVERRIDE;

    //!< MTU を変更する
    virtual void ConfigureBleMtu(uint16_t mtu) NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスアドレスを取得する
    virtual void GetDeviceAddress(::nn::bluetooth::Address* pAddress) NN_NOEXCEPT NN_OVERRIDE;

    //!< Nintendo Hid over Gatt サービスのクライアントを取得する
    virtual BleNhogClient* GetBleNhogClient() NN_NOEXCEPT NN_OVERRIDE;

    //!< Nintendo Battery サービスのクライアントを取得する
    virtual BleNbatClient* GetBleNbatClient() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスアドレスを設定する
    void SetDeviceAddress(const ::nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< GATT Service の一覧を列挙する
    void EnumerateGattServices() NN_NOEXCEPT;

    //!< GATT クライアント機能を有効にする
    bool ActivateGattClient() NN_NOEXCEPT;

    //!< BTM からの MtuConfigCallback をハンドルする
    void HandleMtuConfigCallback() NN_NOEXCEPT;

    //!< BTM からの GattOperationCallback をハンドルする
    void HandleGattOperationCallback(nn::bluetooth::InfoFromLeGattOperationCallback* pResult) NN_NOEXCEPT;

    //!< 定常的な処理を行う関数
    void PeriodicOperation() NN_NOEXCEPT;

    //!< Gatt Client が Activate されているかどうか
    bool IsGattClientActivated() NN_NOEXCEPT
    {
        return m_IsGattClientActivated;
    }

private:
    //!< Nintendo Hid over Gatt サービスの Accessor を有効にする
    bool ActivateNhogClient() NN_NOEXCEPT;

    //!< Nintendo Battery サービスの Accessor を有効にする
    bool ActivateNbatClient() NN_NOEXCEPT;

    //!< UUID から Service があるかどうか調べ、あれば Service のオブジェクトを返す
    bool GetGattService(nn::btm::user::GattService* pOutService, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

    //!< BTM のリトライをハンドリングする
    void HandleBtmRetryEvent() NN_NOEXCEPT;
};

}}} // namespace nn::xcd::detail
