﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <tchar.h>

#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TimerEvent.h>
#include "xcd_Win32HidAccessor-os.win32.h"
#include "aardvark/aardvark-os.win32.h"

namespace nn { namespace xcd { namespace detail{

/**
 * @brief   Aardvark の Vendor ID
 * XXX: 適当な値
 */
const int AardvarkVendorId = 0x3939;

/**
 * @brief   Aardvark の Device path
 */
extern const TCHAR * const AardvarkDevicePath;

/**
 * @brief   Aardvark を扱うクラス
 */
class AardvarkAccessor : public Win32HidAccessor
{
    NN_DISALLOW_MOVE(AardvarkAccessor);
    NN_DISALLOW_COPY(AardvarkAccessor);

public:
    AardvarkAccessor() NN_NOEXCEPT;
    virtual ~AardvarkAccessor() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   Accessorを有効にする (環境依存)
     */
    void Activate(DeviceHandle deviceHandle, aardvark::DeviceHandle aardvarkHandle) NN_NOEXCEPT;

    /**
     * @brief   Accessorを無効にする
     */
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   OutputReportをセットする
     */
    virtual void SetOutputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   InputReportを読み出す
     */
    virtual size_t GetInputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
     * @brief   ダミー応答の形式
     */
    enum class DummyType
    {
        Normal,                 //!< 定常通信
        Ack,                    //!< ACK を要求するコマンドに対する応答
        GetDeviceInfo,          //!< GetDeviceInfo に対する応答
        LrButtonElapsedTime,    //!< LRButtonDetection に対する応答
        SerialFlashRead,        //!< SerialFlashRead に対する応答
        McuIn,                  //!< McuOut に対する応答
        McuRead                 //!< McuWrite に対する応答
    };

private:
    virtual void ThreadFunctionImpl() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   BT MCU との通信を装うダミー入力の構築
     */
    void SetupDummyInput(DummyType type, uint8_t commandId, uint32_t address) NN_NOEXCEPT;

    void SetupDummyInput(DummyType type, uint8_t commandId) NN_NOEXCEPT;

    void SetupDummyInput(DummyType type) NN_NOEXCEPT;

    /**
     * @brief   Aardvark の初期化
     */
    void InitializeDevice() NN_NOEXCEPT;

    /**
     * @brief   Aardvark の終了
     */
    void FinalizeDevice() NN_NOEXCEPT;

    /**
     * @brief   Tera をリセット
     */
    void ResetMcu(bool isSystemBoot) NN_NOEXCEPT;

    /**
     * @brief   Tera と通信するための SPI の設定
     */
    void SetupSpi() NN_NOEXCEPT;

    /**
     * @brief   MCU への出力データを内部バッファに格納
     */
    void SetOutputDataForMcu(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   Standby 遷移コマンドを出力バッファに格納
     */
    void SetStandbyCommandForMcu() NN_NOEXCEPT;

private:
    aardvark::DeviceHandle m_AardvarkHandle;            //!< Aardvark のハンドル
    nn::os::TimerEventType m_PeriodicEvent;             //!< SPI 通信のタイミングを通知するイベント
    nn::os::Mutex          m_BufferMutex;               //!< バッファ操作を排他する Mutex

    uint8_t m_DummyHidBuffer[HidAccessorBufferSize];    //!< ダミーの HID レポートを格納するバッファ
    size_t  m_DummyInSize;                              //!< MCU から受け取る側のダミーデータサイズ
    size_t  m_DummyOutSize;                             //!< MCU に送る側のダミーデータサイズ

    uint8_t m_OutputBuffer[HidAccessorBufferSize];      //!< OutputReport を格納するバッファ
    size_t  m_SendSize;                                 //!< OutputReport のサイズ

    uint8_t m_SampleNumber;                             //!< HID サンプル番号
};

}}} // namespace nn::xcd::detail
