﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include "aardvark-os.win32.h"
#include "aardvark_Driver-os.win32.h"

namespace nn { namespace xcd { namespace detail { namespace aardvark {

namespace
{

// Aardvark の DLL 名
const char DllName[] = "aardvark.dll";

// Aardvark バージョン情報
struct AardvarkVersion
{
    uint16_t    software;
    uint16_t    firmware;
    uint16_t    hardware;

    // SW バージョンがこの値以上であることを FW が要求
    uint16_t    softwareRequiredByFirmware;

    // FW バージョンがこの値以上であることを SW が要求
    uint16_t    firmwareRequiredBySoftware;

    // API バージョンがこの値以上であることを SW が要求
    uint16_t    apiRequiredBySoftware;
};

}  // anonymous namespace


// DLL ロード関数はラップした方が良いかもしれないが、Windows 以外で
// 使用する予定がないので直接呼ぶ

bool Driver::Initialize() NN_NOEXCEPT
{
    if (IsInitialized())
    {
        return true;
    }

    m_DllHandle = ::LoadLibraryA(DllName);
    if (m_DllHandle == nullptr)
    {
        return false;
    }

    /*
    typedef uint32_t (*VersionFuncType)(DeviceHandle, AardvarkVersion*);
    VersionFuncType pVersionFunc;
    pVersionFunc = reinterpret_cast<VersionFuncType>(
        ::GetProcAddress(m_DllHandle, "aa_c_version"));
    NN_ABORT_UNLESS_NOT_NULL(pVersionFunc);

    AardvarkVersion version = {};
    pVersionFunc(0, &version);
    NN_SDK_LOG("Software version     : 0x%04X\n", version.software);
    NN_SDK_LOG("Requested API version: 0x%04X\n", version.apiRequiresBySoftware);
    */

    return true;
}

void Driver::Finalize() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    NN_ABORT_UNLESS(::FreeLibrary(m_DllHandle));
    m_DllHandle = nullptr;
}

DeviceHandle Driver::Open(int portNumber) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.open == nullptr)
    {
        m_Functions.open = reinterpret_cast<OpenFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_open"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.open);
    }

    return m_Functions.open(portNumber);
}

int Driver::Close(DeviceHandle handle) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.close == nullptr)
    {
        m_Functions.close = reinterpret_cast<CloseFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_close"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.close);
    }

    return m_Functions.close(handle);
}

int Driver::GetConfig(DeviceHandle handle) NN_NOEXCEPT
{
    return SetConfig(handle, Mode_Query);
}

int Driver::SetConfig(DeviceHandle handle, Mode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(mode == Mode_GpioOnly   ||
              mode == Mode_SpiAndGpio ||
              mode == Mode_GpioAndI2c ||
              mode == Mode_SpiAndI2c  ||
              mode == Mode_Query);

    AssertInitialized();

    if (m_Functions.configure == nullptr)
    {
        m_Functions.configure = reinterpret_cast<ConfigFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_configure"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.configure);
    }

    return m_Functions.configure(handle, mode);
}

int Driver::GetSpiBitrate(DeviceHandle handle) NN_NOEXCEPT
{
    // bitrate に 0 を渡すと現在の値が返る
    return SetSpiBitrate(handle, 0);
}

int Driver::SetSpiBitrate(DeviceHandle handle, int bitrateKhz) NN_NOEXCEPT
{
    NN_SDK_ASSERT(bitrateKhz >= 0);

    AssertInitialized();

    if (m_Functions.spiBitrate == nullptr)
    {
        m_Functions.spiBitrate = reinterpret_cast<SpiBitrateFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_spi_bitrate"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.spiBitrate);
    }

    return m_Functions.spiBitrate(handle, bitrateKhz);
}

int Driver::ConfigureSpi(DeviceHandle handle, const SpiConfig& config) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.spiConfigure == nullptr)
    {
        m_Functions.spiConfigure = reinterpret_cast<SpiConfigFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_spi_configure"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.spiConfigure);
    }

    int status = m_Functions.spiConfigure(
        handle,
        config.polarity,
        config.phase,
        config.bitOrder);

    return status;
}

int Driver::SetSpiMode(DeviceHandle handle, SpiMode mode) NN_NOEXCEPT
{
    AssertInitialized();

    SetSpiSlaveFuncType setSlaveModeFunc;
    switch (mode)
    {
    case SpiMode_Master:
        if (m_Functions.disableSpiSlave == nullptr)
        {
            m_Functions.disableSpiSlave = reinterpret_cast<SetSpiSlaveFuncType>(
                ::GetProcAddress(m_DllHandle, "c_aa_spi_slave_disable"));
            NN_ABORT_UNLESS_NOT_NULL(m_Functions.disableSpiSlave);
        }
        setSlaveModeFunc = m_Functions.disableSpiSlave;
        break;

    case SpiMode_Slave:
        if (m_Functions.enableSpiSlave == nullptr)
        {
            m_Functions.enableSpiSlave = reinterpret_cast<SetSpiSlaveFuncType>(
                ::GetProcAddress(m_DllHandle, "c_aa_spi_slave_enable"));
            NN_ABORT_UNLESS_NOT_NULL(m_Functions.enableSpiSlave);
        }
        setSlaveModeFunc = m_Functions.enableSpiSlave;
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    return setSlaveModeFunc(handle);
}

int Driver::SetSpiMasterSsPolarity(DeviceHandle handle, SpiMasterSsPolarity polarity) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.setMasterSsPolarity == nullptr)
    {
        m_Functions.setMasterSsPolarity = reinterpret_cast<SetSpiMasterSsPolarityFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_spi_master_ss_polarity"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.setMasterSsPolarity);
    }

    return m_Functions.setMasterSsPolarity(handle, polarity);
}

int Driver::ReadSpi(size_t* pOutDoneBytes,
                    char* pOutReadData,
                    size_t readDataBytes,
                    DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutDoneBytes);
    NN_SDK_ASSERT_NOT_NULL(pOutReadData);

    AssertInitialized();

    if (m_Functions.readSpi == nullptr)
    {
        m_Functions.readSpi = reinterpret_cast<ReadSpiFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_spi_slave_read"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.readSpi);
    }

    int status = m_Functions.readSpi(
        handle,
        static_cast<uint16_t>(readDataBytes),
        pOutReadData);
    if (status >= 0)
    {
        // 成功時は受信したバイト数を格納
        *pOutDoneBytes = status;
    }
    else
    {
        *pOutDoneBytes = 0;
    }

    return status;
}

int Driver::WriteSpi(
    char* pOutReceivedData,
    size_t receiveBytes,
    const char* pWriteData,
    size_t writeDataBytes,
    DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutReceivedData);
    NN_SDK_ASSERT_NOT_NULL(pWriteData);

    AssertInitialized();

    if (m_Functions.writeSpi == nullptr)
    {
        m_Functions.writeSpi = reinterpret_cast<WriteSpiFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_spi_write"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.writeSpi);
    }

    return m_Functions.writeSpi(
        handle,
        static_cast<uint16_t>(writeDataBytes),
        pWriteData,
        static_cast<uint16_t>(receiveBytes),
        pOutReceivedData);
}

int Driver::SetGpioDirection(DeviceHandle handle, nn::Bit8 directionMask) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.setGpioDirection == nullptr)
    {
        m_Functions.setGpioDirection = reinterpret_cast<SetGpioDirectionFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_gpio_direction"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.setGpioDirection);
    }

    return m_Functions.setGpioDirection(handle, directionMask);
}

int Driver::SetGpioPullup(DeviceHandle handle, nn::Bit8 pullupMask) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.setGpioPullup == nullptr)
    {
        m_Functions.setGpioPullup = reinterpret_cast<SetGpioPullupFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_gpio_pullup"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.setGpioPullup);
    }

    return m_Functions.setGpioPullup(handle, pullupMask);
}

int Driver::GetGpioPins(DeviceHandle handle) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.getGpio == nullptr)
    {
        m_Functions.getGpio = reinterpret_cast<GetGpioFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_gpio_get"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.getGpio);
    }

    return m_Functions.getGpio(handle);
}

int Driver::SetGpioPins(DeviceHandle handle, nn::Bit8 pinStatus) NN_NOEXCEPT
{
    AssertInitialized();

    if (m_Functions.setGpio == nullptr)
    {
        m_Functions.setGpio = reinterpret_cast<SetGpioFuncType>(
            ::GetProcAddress(m_DllHandle, "c_aa_gpio_set"));
        NN_ABORT_UNLESS_NOT_NULL(m_Functions.setGpio);
    }

    return m_Functions.setGpio(handle, pinStatus);
}

}}}}
