﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/sf/sf_Types.h>
#include "wlan_CreateWlanManagers.h"

namespace nn {
namespace wlan {

namespace {

nn::sf::SharedPointer<detail::ISocketManager> g_SocketManager;
nn::sf::HipcSimpleClientSessionManager g_Manager;
nn::sf::SharedPointer<detail::ISocketGetFrame> g_SocketGetFrameManager;
nn::sf::HipcSimpleClientSessionManager g_GetFrameManager;
bool g_TsfTimerFunctionEnabled;

// Initialize の参照カウント
int g_SocketInitializeCount = 0;
}

nn::Result InitializeSocketManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_SocketManager);
    NN_SDK_REQUIRES(!g_SocketGetFrameManager);
    if( g_SocketInitializeCount == 0 )
    {
        g_SocketManager = nn::wlan::CreateSocketManagerByHipc(&g_Manager);
        g_SocketGetFrameManager = nn::wlan::CreateSocketGetFrameManagerByHipc(&g_GetFrameManager);
        g_TsfTimerFunctionEnabled = false;
        g_SocketInitializeCount++;
    }
    NN_RESULT_SUCCESS;
}

nn::Result FinalizeSocketManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    NN_SDK_REQUIRES(g_SocketGetFrameManager);
    if( g_SocketInitializeCount > 0 )
    {
        g_SocketManager = nullptr;
        g_Manager.Finalize();
        g_SocketGetFrameManager = nullptr;
        g_GetFrameManager.Finalize();
        g_SocketInitializeCount--;
    }
    NN_RESULT_SUCCESS;
}

nn::Result RegisterSharedMemory(
       nn::os::NativeHandle memHandle,
       uint32_t memSize,
       nn::os::NativeHandle txEventHandle,
       nn::os::NativeHandle rxEventHandle,
       nn::os::NativeHandle txCbufEventHandle,
       nn::os::NativeHandle rxCbufEventHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);

    nn::Result result;

    result = g_SocketManager->RegisterSharedMemory(
                nn::sf::NativeHandle(memHandle, false),
                memSize,
                nn::sf::NativeHandle(txEventHandle, false),
                nn::sf::NativeHandle(rxEventHandle, false),
                nn::sf::NativeHandle(txCbufEventHandle, false),
                nn::sf::NativeHandle(rxCbufEventHandle, false));

    return result;
}

nn::Result UnregisterSharedMemory() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);

    nn::Result result;

    result = g_SocketManager->UnregisterSharedMemory();

    return result;
}

nn::Result EnableSharedMemory() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);

    nn::Result result;

    result = g_SocketManager->EnableSharedMemory();

    return result;
}

namespace Socket {


nn::Result PutFrameRaw(const uint8_t pInput[], size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( pInput == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pInput must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size >= 2048 )
    {
        NN_SDK_REQUIRES(false, "%s: PutFrame size(%d) must be less than 2048 bytes\n", __FUNCTION__, size);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pInput), size);
    auto result = g_SocketManager->PutFrameRaw(pInBuffer);

    return result;
}


nn::Result GetFrameRaw(uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketGetFrameManager);
    if( pOutBuf == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutBuf must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutSize == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutSize must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutBuf), size);
    uint32_t outSize = 0;  // 受信データサイズが32bit以上になることはないので、uint32_tで十分
    auto result = g_SocketGetFrameManager->GetFrameRaw(pOutBuffer, &outSize, rxId);
    *pOutSize = static_cast<size_t>(outSize);

    return result;
}


nn::Result CancelGetFrame(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    auto result = g_SocketManager->CancelGetFrame(rxId);
    return result;
}

nn::Result CreateRxEntry(uint32_t* pOutRxId, const uint16_t pEthertypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pEthertypes == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pEthertypes must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( count > EthertypeCountMax )
    {
        NN_SDK_REQUIRES(false, "%s: The number of ethertypes must be less than %d\n", __FUNCTION__, EthertypeCountMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }
    if( capacity > DataFrameRxCapacityMax )
    {
        NN_SDK_REQUIRES(false, "%s: Capacity of RxEntry must be less than %d\n", __FUNCTION__, DataFrameRxCapacityMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InArray<uint16_t> InArray(pEthertypes, count);
    uint32_t rxId = 0;
    auto result = g_SocketManager->CreateRxEntry(&rxId, InArray, capacity);
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}

nn::Result DeleteRxEntry(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);

    auto result = g_SocketManager->DeleteRxEntry(rxId);

    return result;
}


nn::Result AddEthertypeToRxEntry(uint32_t rxId, uint16_t ethertype) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);

    auto result = g_SocketManager->AddEthertypeToRxEntry(rxId, ethertype);

    return result;
}


nn::Result DeleteEthertypeFromRxEntry(uint32_t* pOutRxId, uint16_t ethertype) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t rxId = 0;
    auto result = g_SocketManager->DeleteEthertypeFromRxEntry(&rxId, ethertype);
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}

nn::Result GetMacAddress(MacAddress* mac) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( mac == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: mac must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlMacAddress pMac;
    auto result = g_SocketManager->GetMacAddress(nn::sf::Out<nn::wlan::detail::SfdlMacAddress>(&pMac));
    if( result.IsFailure() )
    {
        return result;
    }

    mac->Set(pMac.data);

    return result;
}

nn::Result EnableTsfTimerFunction() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( g_TsfTimerFunctionEnabled == false )
    {
        auto result = g_SocketManager->SwitchTsfTimerFunction(true);
        if( result.IsSuccess() )
        {
            g_TsfTimerFunctionEnabled = true;
        }
        return result;
    }
    NN_RESULT_SUCCESS;
}

nn::Result DisableTsfTimerFunction() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( g_TsfTimerFunctionEnabled == true )
    {
        auto result = g_SocketManager->SwitchTsfTimerFunction(false);
        if( result.IsSuccess() )
        {
            g_TsfTimerFunctionEnabled = false;
        }
        return result;
    }
    NN_RESULT_SUCCESS;
}

nn::Result GetDeltaTimeBetweenSystemAndTsf(int64_t* pOutDeltaTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( pOutDeltaTime == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutDeltaTime must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    if( g_TsfTimerFunctionEnabled != true )
    {
        return nn::wlan::ResultNotEnabled();
    }

    int64_t deltaTime;
    auto result = g_SocketManager->GetDeltaTimeBetweenSystemAndTsf(&deltaTime);
    if( result.IsSuccess() )
    {
        *pOutDeltaTime = deltaTime;
    }

    return result;
}

nn::Result SetMulticastFilter(const MulticastFilterInfo& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_SocketManager);
    if( info.count > MulticastFilterCountMax )
    {
        NN_SDK_REQUIRES(false, "%s: Multicast filter count(%d) must not be more than %d\n",
                        __FUNCTION__, info.count, MulticastFilterCountMax);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlMulticastFilterInfo sfdlInfo;
    for(int i = 0; i < info.count; i++)
    {
        std::memcpy(&sfdlInfo.address[i].data[0], info.address[i].GetMacAddressData(), MacAddress::MacAddressSize);
    }
    sfdlInfo.count = info.count;
    auto result = g_SocketManager->SetMulticastFilter(sfdlInfo);

    return result;
}

}
}
}

