﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <cctype>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/sf/sf_Types.h>
#include <nn/wlan/detail/wlan_InternalTypes.h>
#include "wlan_CreateWlanManagers.h"
#include "../../Processes/wlan/wlan_SignalStrength.h"

namespace nn {
namespace wlan {

namespace {

nn::sf::SharedPointer<detail::IInfraManager> g_InfraManager;
nn::sf::HipcSimpleClientSessionManager g_Manager;

// Initialize の参照カウント
int g_InitializeCount = 0;
}

nn::Result InitializeInfraManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_InfraManager);
    if( g_InitializeCount == 0 )
    {
        g_InfraManager = nn::wlan::CreateInfraManagerByHipc(&g_Manager);
        g_InitializeCount++;
    }
    NN_RESULT_SUCCESS;
}

nn::Result FinalizeInfraManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( g_InitializeCount > 0 )
    {
        g_InfraManager = nullptr;
        g_Manager.Finalize();
        g_InitializeCount--;
    }
    NN_RESULT_SUCCESS;
}

namespace Infra {

nn::Result OpenMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->OpenMode();
    return result;
}

nn::Result CloseMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->CloseMode();
    return result;
}

nn::Result GetMacAddress(MacAddress* pOutMac) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutMac == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutMac must not be NULL");
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlMacAddress pMac;
    auto result = g_InfraManager->GetMacAddress(nn::sf::Out<nn::wlan::detail::SfdlMacAddress>(&pMac));
    if( result.IsFailure() )
    {
        return result;
    }

    pOutMac->Set(pMac.data);

    return result;
}

nn::Result StartScan(void* pOutScanBuffer, size_t size, const ScanParameters& param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    // パラメータチェック
    if( pOutScanBuffer == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutScanBuffer must not be NULL");
        return nn::wlan::ResultInvalidArgument();
    }
    if( size <= sizeof(nn::wlan::ScanResultHeader) )
    {
        NN_SDK_REQUIRES(false, "scan buffer size must be larger than %d", sizeof(nn::wlan::ScanResultHeader));
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.scanType == nn::wlan::ScanType_Active || param.scanType == nn::wlan::ScanType_Passive) )
    {
        NN_SDK_REQUIRES(false, "scanType is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.channelScanTime == -1 || (param.channelScanTime >= 10 && param.channelScanTime <= 3000)) )
    {
        NN_SDK_REQUIRES(false, "channelScanTime is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(param.homeChannelTime == -1 || (param.homeChannelTime >= 0 && param.homeChannelTime <= 500)) )
    {
        NN_SDK_REQUIRES(false, "homeChannelTime is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( param.bssid == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
        return nn::wlan::ResultInvalidArgument();
    }

    nn::Result result;

    // IPC用構造体へ値をコピー
    nn::wlan::detail::SfdlScanParameters scanParams;
    scanParams.scanType = param.scanType;
    scanParams.channelCount = param.channelCount;
    for(int i = 0; i < param.channelCount; i++)
    {
        // チャンネルチェック
        if( param.channelList[i] < WirelessChannel_1ch || param.channelList[i] > WirelessChannel_165ch )
        {
            NN_SDK_REQUIRES(false, "channel is out of range");
            return nn::wlan::ResultInvalidArgument();
        }
        scanParams.channelList.channel[i] = param.channelList[i];
    }
    scanParams.channelScanTime = param.channelScanTime;
    scanParams.homeChannelTime = param.homeChannelTime;
    scanParams.ssidCount = param.ssidCount;
    // scan対象SSIDリストのコピー判断
    if( param.ssidCount > 0 && param.ssidList != NULL )
    {
        if( param.ssidCount > nn::wlan::ScanningSsidCountMax )
        {
            NN_SDK_LOG("[WLAN][WARN]Scanning SSID count should not be over %d.\n", nn::wlan::ScanningSsidCountMax);
            scanParams.ssidCount = nn::wlan::ScanningSsidCountMax;
        }
        for(int i = 0; i < scanParams.ssidCount; i++)
        {
            if( param.ssidList[i].GetLength() == 0 )
            {
                NN_SDK_LOG("[WLAN][WARN]The length of a filtering SSID must not be 0.\n");
                NN_SDK_ASSERT(false);
                return nn::wlan::ResultInvalidArgument();
            }
            scanParams.ssidList[i].length = param.ssidList[i].GetLength();
            std::memcpy(&scanParams.ssidList[i].data[0], param.ssidList[i].GetSsidData(), param.ssidList[i].GetLength());
        }
    }
    else
    {
        scanParams.ssidCount = 0;  // scanParams.ssidListはNULLではないので、ssidCountを0にしておくことでIPCサーバー側でSSIDフィルタ無しと判断する。
    }

    for(uint8_t i = 0; i < nn::wlan::MacAddress::MacAddressSize; i++)
    {
        scanParams.bssid.data[i] = param.bssid.GetMacAddressData()[i];
    }

    // Scan完了通知を受け取るためのSystemEventを設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_InfraManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ScanComplete));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::SystemEvent scanCompEvent;
    scanCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    result = g_InfraManager->StartScan(scanParams);
    if( result.IsFailure() )
    {
        return result;
    }

    // Scan完了待機
    scanCompEvent.Wait();

    // Scanが完了したので結果を取りに行く

    // 出力用バッファの作成
    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutScanBuffer), size);
    result = g_InfraManager->GetScanResult(pOutBuffer);

    return result;
} //NOLINT(impl/function_size)


nn::Result StopScan() NN_NOEXCEPT
{
     NN_SDK_REQUIRES(g_InfraManager);
     auto result = g_InfraManager->StopScan();
     return result;
}


nn::Result Connect(const Ssid& ssid, const MacAddress& bssid, int16_t channel, const Security& security, bool autoKeepAlive, int beaconLostTimeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    // パラメータチェック
    if( ssid.GetLength() == 0 )
    {
        NN_SDK_REQUIRES(false, "SSID must not be empty");
        return nn::wlan::ResultInvalidArgument();
    }
    if( bssid == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "bssid must not be ZeroMacAddress");
        return nn::wlan::ResultInvalidArgument();
    }
    if( !(channel == -1 || (channel >= nn::wlan::WirelessChannel_1ch && channel <= nn::wlan::WirelessChannel_165ch)) )
    {
        NN_SDK_REQUIRES(false, "channel is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( security.privacyMode > SecurityMode_Wpa2Aes )
    {
        NN_SDK_REQUIRES(false, "privacyMode is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( security.privacyMode >= SecurityMode_WpaTkip && security.groupPrivacyMode == SecurityMode_StaticAes )
    {
        NN_SDK_REQUIRES(false, "groupPrivacyMode is invalid");
        return nn::wlan::ResultInvalidArgument();
    }
    if( security.privacyMode >= nn::wlan::SecurityMode_Wep64Open && security.privacyMode <= nn::wlan::SecurityMode_Wep128Shared )
    {
        if(security.privacyMode == nn::wlan::SecurityMode_Wep64Open || security.privacyMode == nn::wlan::SecurityMode_Wep64Shared )
        {
            if( nn::util::Strnlen(security.key, 11) != 10 )
            {
                NN_SDK_REQUIRES(false, "WEP64 key length must be 10\n");
                return nn::wlan::ResultInvalidArgument();
            }
        }
        else if(security.privacyMode == nn::wlan::SecurityMode_Wep128Open || security.privacyMode == nn::wlan::SecurityMode_Wep128Shared )
        {
            if( nn::util::Strnlen(security.key, 27) != 26 )
            {
                NN_SDK_REQUIRES(false, "WEP128 key length must be 26\n");
                return nn::wlan::ResultInvalidArgument();
            }
        }
        if( security.keyIdx > 3 )
        {
            NN_SDK_REQUIRES(false, "keyIdx is invalid");
            return nn::wlan::ResultInvalidArgument();
        }
    }
    if( !(beaconLostTimeout >= 1 && beaconLostTimeout <= 30) )
    {
        NN_SDK_REQUIRES(false, "beaconLostTimeout is invalid");
        return nn::wlan::ResultInvalidArgument();
    }

    switch(security.privacyMode)
    {
    case SecurityMode_WpaTkip:
    case SecurityMode_WpaAes:
    case SecurityMode_Wpa2Tkip:
    case SecurityMode_Wpa2Aes:
    {
        int length;
        length = nn::util::Strnlen(security.key, 65);
        if( length < 8 || length > 64 )
        {
            NN_SDK_ASSERT(false, "security key (passphrase) length is out of range.\n");
            return ResultInvalidArgument();
        }
        if( length == 64 )
        {
            for(int i = 0; i < 64; i++)
            {
                if( isxdigit(security.key[i]) == false )
                {
                    NN_SDK_ASSERT(false, "security key (64 hexadecimal digits) accepts only [ 0~9 a~f A~F]\n");
                    return ResultInvalidArgument();
                }
            }
        }
        break;
    }
    default:
        break;
    }

    nn::Result result;

    // Connect完了通知を受け取るイベントの設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_InfraManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ConnectionComplete));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::SystemEvent connectCompEvent;
    connectCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    detail::SfdlSsid sfdlSsid;
    detail::SfdlMacAddress sfdlBssid;
    detail::SfdlSecurity sfdlSecurity;

    std::memcpy(&sfdlSsid.data[0], ssid.GetSsidData(), ssid.GetLength());
    sfdlSsid.length = static_cast<uint8_t>(ssid.GetLength());

    std::memcpy(&sfdlBssid.data[0], bssid.GetMacAddressData(), MacAddress::MacAddressSize);

    sfdlSecurity.privacyMode = static_cast<uint32_t>(security.privacyMode);
    sfdlSecurity.groupPrivacyMode = static_cast<uint32_t>(security.groupPrivacyMode);
    std::memcpy(&sfdlSecurity.key[0], &security.key[0], sizeof(security.key));
    sfdlSecurity.keyIdx = static_cast<uint32_t>(security.keyIdx);

    result = g_InfraManager->Connect(sfdlSsid, sfdlBssid, channel, sfdlSecurity, autoKeepAlive, beaconLostTimeout);
    if( result.IsFailure() )
    {
        return result;
    }

    // Connect完了待機
    connectCompEvent.Wait();

    // 4way-handshake失敗ケースに対応(SIGLO-64396)
    ConnectionStatus status;
    std::memset(&status, 0, sizeof(ConnectionStatus));
    GetConnectionStatus(&status);
    if( status.cause == CauseOfInfo_ConnectFailureAfterAssoc )
    {
        NN_SDK_LOG("[WLAN]Infra::Connect() ConnectFailureAfterAssoc happens. Call disconnect API from lib layer.\n");
        Disconnect();
    }

    return result;
} //NOLINT(impl/function_size)

nn::Result CancelConnect() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->CancelConnect();
    return result;
}

nn::Result Disconnect() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->Disconnect();
    return result;
}

nn::Result GetConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pSystemEvent == NULL )
    {
        NN_SDK_REQUIRES(false, "pSystemEvent must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::NativeHandle sfHandle;  // 出力用変数

    nn::Result result = g_InfraManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ConnectionEvent));
    if( result.IsFailure() )
    {
        return result;
    }

    nn::os::AttachReadableHandleToSystemEvent(pSystemEvent,
                                              sfHandle.GetOsHandle(),
                                              sfHandle.IsManaged(),
                                              nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    return result;
}

nn::Result GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutStatus == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutStatus must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlConnectionStatus sfConnectionStatus;  // 出力用変数

    nn::Result result = g_InfraManager->GetConnectionStatus(&sfConnectionStatus);
    if( result.IsFailure() )
    {
        return result;
    }

    // 結果のコピー
    pOutStatus->aid = sfConnectionStatus.aid;
    pOutStatus->beaconInterval = sfConnectionStatus.beaconInterval;
    pOutStatus->bssid.Set( sfConnectionStatus.bssid.data );
    pOutStatus->capabilityInfo = sfConnectionStatus.capabilityInfo;
    pOutStatus->cause = static_cast<CauseOfInfo>(sfConnectionStatus.cause);
    pOutStatus->channel = sfConnectionStatus.channel;
    pOutStatus->ssid.Set( sfConnectionStatus.ssid.data, static_cast<size_t>(sfConnectionStatus.ssid.length) );
    pOutStatus->state = static_cast<ConnectionState>(sfConnectionStatus.state);
    pOutStatus->statusReasonCode = sfConnectionStatus.statusReasonCode;

    return result;
}

nn::Result GetState(WlanState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutState == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutState must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t state;  // 出力用変数
    auto result = g_InfraManager->GetState(&state);
    if( result.IsFailure() )
    {
        return result;
    }

    *pOutState = static_cast<WlanState>(state);

    NN_RESULT_SUCCESS;
}

nn::Result GetAllowedChannels(int16_t (&outChannels)[WirelessChannelsCountMax], uint32_t* pOutCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutCount == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutCount must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    return g_InfraManager->GetAllowedChannels(nn::sf::OutArray<int16_t>(outChannels, WirelessChannelsCountMax), pOutCount);
}

nn::Result GetRssi(int32_t* pRssi) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pRssi == NULL )
    {
        NN_SDK_REQUIRES(false, "pRssi must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    int32_t rssi;  // 出力用変数
    auto result = g_InfraManager->GetRssi(&rssi);
    *pRssi = rssi;

    return result;
}

nn::Result GetLinkLevel(LinkLevel* pLinkLevel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pLinkLevel == NULL )
    {
        NN_SDK_REQUIRES(false, "pLinkLevel must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    int32_t rssi;
    auto result = GetRssi(&rssi);
    if( result.IsSuccess() )
    {
        *pLinkLevel = ConvertRssiToLinkLevel(rssi);
    }
    else
    {
        *pLinkLevel = LinkLevel_0;
    }

    return result;
}

LinkLevel ConvertRssiToLinkLevel(int32_t rssi) NN_NOEXCEPT
{
    LinkLevel level;
    // チャンネルの取得
    ConnectionStatus status;
    auto result = GetConnectionStatus(&status);
    if( result.IsSuccess() )
    {
        if( status.state == ConnectionState_Connected &&
                status.channel >= WirelessChannel_36ch )
        {
            level = SignalStrength(rssi).GetLinkLevelInfra5G();
        }
        else
        {
            level = SignalStrength(rssi).GetLinkLevelInfra2G();
        }
    }
    else
    {
        level = SignalStrength(rssi).GetLinkLevelInfra2G();
    }
    return level;
}

nn::Result ChangeRxAntenna(RxAntennaPattern rxAntenna) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( rxAntenna < RxAntennaPattern_0 || rxAntenna > RxAntennaPattern_Both )
    {
        NN_SDK_ASSERT(false, "%s: invalid argument", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    auto result = g_InfraManager->ChangeRxAntenna(static_cast<int32_t>(rxAntenna));

    return result;
}

nn::Result GetFwVersion(char* pOutStr, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutStr == NULL )
    {
        return nn::wlan::ResultInvalidArgument();
    }
    if( size < 256 )
    {
        return nn::wlan::ResultInvalidArgument();
    }

    // 出力用バッファの作成
    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutStr), size);
    auto result = g_InfraManager->GetFwVersion(pOutBuffer);
    return result;
}

nn::Result RequestSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->RequestSleep();
    return result;
}

nn::Result RequestWakeUp() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->RequestWakeUp();
    return result;
}

nn::Result ConnectWithWps(WpsMethod method, const char* pPin, size_t size, int beaconLostTimeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);

    if( method == WpsMethod_Pin )
    {
        if( pPin == NULL || size < WpsPinLength + 1 )
        {
            NN_SDK_ASSERT(false, "%s: invalid argument", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
        if( nn::util::Strnlen(pPin, WpsPinLength + 1) != WpsPinLength )
        {
            NN_SDK_ASSERT(false, "%s: Wps pin length is not %d\n", __FUNCTION__, WpsPinLength);
            return nn::wlan::ResultInvalidArgument();
        }
    }
    if( beaconLostTimeout < 1 || beaconLostTimeout > 30 )
    {
        NN_SDK_ASSERT(false, "%s: beaconLostTimeout is out of range. your input:[%d]\n", __FUNCTION__, beaconLostTimeout);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::Result result;

    // Connect完了通知を受け取るイベントの設定
    nn::sf::NativeHandle sfHandle;  // 出力用変数
    result = g_InfraManager->GetSystemEvent(&sfHandle, static_cast<uint32_t>(detail::SystemEventType_ConnectionComplete));
    NN_SDK_ASSERT( result.IsSuccess() );

    nn::os::SystemEvent connectCompEvent;
    connectCompEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    sfHandle.Detach();

    nn::sf::InArray<char> WpsPinArray(pPin, size);

    result = g_InfraManager->ConnectWithWps(static_cast<uint32_t>(method), WpsPinArray, beaconLostTimeout);
    if( result.IsFailure() )
    {
        return result;
    }

    // Connect完了待機
    connectCompEvent.Wait();

    return nn::ResultSuccess();
}

nn::Result GenerateWpsPin(char* pin, size_t size) NN_NOEXCEPT
{
    if( pin == NULL || size < WpsPinLength + 1 )
    {
        return nn::wlan::ResultInvalidArgument();
    }

    std::srand(static_cast<uint32_t>(nn::os::GetSystemTick().GetInt64Value()));
    uint32_t randomPin = static_cast<uint32_t>(std::rand() % 10000000); // 7桁整数を生成

    // check sum計算
    uint32_t accum = 0;
    uint32_t digit;
    uint32_t spin = randomPin * 10;

    accum += 3 * ((spin / 10000000) % 10);
    accum += 1 * ((spin / 1000000) % 10);
    accum += 3 * ((spin / 100000) % 10);
    accum += 1 * ((spin / 10000) % 10);
    accum += 3 * ((spin / 1000) % 10);
    accum += 1 * ((spin / 100) % 10);
    accum += 3 * ((spin / 10) % 10);
    digit = (accum % 10);

    accum = (10 - digit) % 10;

    nn::util::SNPrintf(pin, WpsPinLength + 1, "%.7u%u", randomPin, accum);

    return nn::ResultSuccess();
}

nn::Result GetWakeupReason(WowlWakeReason* pOutReason) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutReason == NULL )
    {
        return nn::wlan::ResultInvalidArgument();
    }

    // 出力用変数
    uint32_t out = 0;
    auto result = g_InfraManager->GetWakeupReason(&out);
    if( result.IsSuccess() )
    {
        *pOutReason = static_cast<WowlWakeReason>(out);
    }
    else
    {
        *pOutReason = WowlWakeReason_Nothing;
    }
    return result;
}

nn::Result SetTcpSessionInformation(MacAddress dstMac, WlanIpv4Address srcIp, WlanIpv4Address dstIp,
        uint16_t srcPort, uint16_t dstPort, uint32_t ackNum, uint16_t windowSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( dstMac == nn::wlan::MacAddress::CreateZeroMacAddress() || dstMac == nn::wlan::MacAddress::CreateBroadcastMacAddress() )
    {
        NN_SDK_REQUIRES(false, "dstMac must not be ZeroMacAddress or BroadcastMacAddress.\n");
        return nn::wlan::ResultInvalidArgument();
    }

    detail::SfdlMacAddress sfdlMac;
    std::memcpy(&sfdlMac.data[0], dstMac.GetMacAddressData(), MacAddress::MacAddressSize);

    nn::Result result;
    result = g_InfraManager->SetTcpSessionInformation(sfdlMac, srcIp, dstIp,
            srcPort, dstPort, ackNum, windowSize);
    return result;
}

nn::Result RemoveTcpSessionInformation() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);

    nn::Result result;
    result = g_InfraManager->RemoveTcpSessionInformation();

    return result;
}

nn::Result GetWakeupReasonRaw(uint32_t* pOutReason) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pOutReason == NULL )
    {
        return nn::wlan::ResultInvalidArgument();
    }

    // 出力用変数
    uint32_t out = 0;
    auto result = g_InfraManager->GetWakeupReasonRaw(&out);
    if( result.IsSuccess() )
    {
        *pOutReason = out;
    }
    else
    {
        *pOutReason = 0;
    }
    return result;
}

nn::Result SetWakeupReasonRaw(uint32_t reason) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);

    auto result = g_InfraManager->SetWakeupReasonRaw(reason);
    return result;
}

nn::Result EnableWowlFeatures(uint32_t features) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);

    auto result = g_InfraManager->EnableWowlFeatures(features);
    return result;
}

nn::Result GetWowlStats(WowlWakeCount* pCounts, WowlSleepStats* pStats) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    if( pCounts == NULL )
    {
        NN_SDK_REQUIRES(false, "pCounts must not be null.\n");
        return nn::wlan::ResultInvalidArgument();
    }
    if( pStats == NULL )
    {
        NN_SDK_REQUIRES(false, "pStats must not be null.\n");
        return nn::wlan::ResultInvalidArgument();
    }

    // 出力用変数
    WowlWakeCount counts;
    WowlSleepStats stats;
    auto result = g_InfraManager->GetWowlStats(&counts, &stats);
    if( result.IsSuccess() )
    {
        std::memcpy(pCounts, &counts, sizeof(WowlWakeCount));
        std::memcpy(pStats, &stats, sizeof(WowlSleepStats));
    }
    return result;
}

nn::Result ClearWowlStats(bool counts, bool stats) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->ClearWowlStats(counts, stats);
    return result;
}

nn::Result InitializeWlanOnSkipBoot() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->InitializeWlanOnSkipBoot();
    return result;
}

nn::Result EmulateDriverInitFail() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InfraManager);
    auto result = g_InfraManager->EmulateDriverInitFail();
    return result;
}

}
}
}

