﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/wlan_DetectApi.h>
#include <nn/wlan/detail/wlan_InternalTypes.h>
#include <nn/sf/sf_Types.h>
#include "wlan_CreateWlanManagers.h"
#include "../../Processes/wlan/wlan_SignalStrength.h"

namespace nn {
namespace wlan {

namespace {

nn::sf::SharedPointer<detail::IDetectManager> g_DetectManager;
nn::sf::HipcSimpleClientSessionManager g_DetectSessionManager;
nn::sf::SharedPointer<detail::IDetectManager> g_DetectGetActionFrameManager;
nn::sf::HipcSimpleClientSessionManager g_DetectGetActionFrameSessionManager;

// Initialize の参照カウント
int g_DetectInitializeCount = 0;
}

nn::Result InitializeDetectManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_DetectManager);
    NN_SDK_REQUIRES(!g_DetectGetActionFrameManager);
    if( g_DetectInitializeCount == 0 )
    {
        g_DetectManager = nn::wlan::CreateDetectManagerByHipc(&g_DetectSessionManager);
        g_DetectGetActionFrameManager = nn::wlan::CreateDetectManagerByHipc(&g_DetectGetActionFrameSessionManager);
        g_DetectInitializeCount++;
    }
    NN_RESULT_SUCCESS;
}

nn::Result FinalizeDetectManager() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    NN_SDK_REQUIRES(g_DetectGetActionFrameManager);
    if( g_DetectInitializeCount > 0 )
    {
        g_DetectManager = nullptr;
        g_DetectSessionManager.Finalize();
        g_DetectGetActionFrameManager = nullptr;
        g_DetectGetActionFrameSessionManager.Finalize();
        g_DetectInitializeCount--;
    }
    NN_RESULT_SUCCESS;
}

namespace Detect {

nn::Result OpenMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->OpenMode();
}


nn::Result OpenMode(uint16_t channel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);

    if( channel < WirelessChannel_1ch || channel > WirelessChannel_165ch )
    {
        NN_SDK_REQUIRES(false, "%s: channel is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    return g_DetectManager->OpenModeWithChannel(channel);
}


nn::Result CloseMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->CloseMode();
}


nn::Result StartCommunication() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->StartCommunication();
}


nn::Result StopCommunication() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->StopCommunication();
}

nn::Result GetState(WlanState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pOutState == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutState must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t state;  // 出力用変数
    auto result = g_DetectManager->GetState(&state);
    if( result.IsFailure() )
    {
        return result;
    }

    *pOutState = static_cast<WlanState>(state);

    NN_RESULT_SUCCESS;
}

nn::Result GetAllowedChannels(int16_t (&outChannels)[WirelessChannelsCountMax], uint32_t* pOutCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pOutCount == NULL )
    {
        NN_SDK_REQUIRES(false, "pOutCount must not be null");
        return nn::wlan::ResultInvalidArgument();
    }

    return g_DetectManager->GetAllowedChannels(nn::sf::OutArray<int16_t>(outChannels, WirelessChannelsCountMax), pOutCount);
}

nn::Result GetMacAddress(MacAddress* pOutMac) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pOutMac == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutMac must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::wlan::detail::SfdlMacAddress pMac;
    auto result = g_DetectManager->GetMacAddress(nn::sf::Out<nn::wlan::detail::SfdlMacAddress>(&pMac));
    if( result.IsFailure() )
    {
        return result;
    }

    pOutMac->Set(pMac.data);

    return result;
}


LinkLevel ConvertRssiToLinkLevel(int32_t rssi) NN_NOEXCEPT
{
    return SignalStrength(rssi).GetLinkLevelLocal();
}


nn::Result PutActionFrameOneShot(const MacAddress& dstMac, ActionFrameType subtype, DetectHash hash,
        const uint8_t* pData, size_t size, uint32_t dwellTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( dstMac == MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "%s: dstMac must not be Zero mac address\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( subtype != ActionFrameType_Detect )
    {
        NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pData must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size > DetectActionFramePayloadMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( dwellTime > 300 )
    {
        NN_SDK_REQUIRES(false, "%s: dwellTime must be less than 301\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pData), size);
    nn::wlan::detail::SfdlMacAddress mac;
    std::memcpy(&mac.data[0], dstMac.GetMacAddressData(), MacAddress::MacAddressSize);

    auto result = g_DetectManager->PutActionFrameOneShot(mac, static_cast<uint8_t>(subtype), hash, pInBuffer, dwellTime);
    return result;
}


nn::Result StartPeriodicActionFrame(ActionFrameType subtype, DetectHash hash,
        const uint8_t* pData, size_t size, uint32_t interval) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( subtype != ActionFrameType_Detect )
    {
        NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pData must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size > DetectActionFramePayloadMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( interval > PeriodicActionFrameIntervalMax )
    {
        NN_SDK_REQUIRES(false, "%s: interval is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pData), size);

    auto result = g_DetectManager->StartPeriodicActionFrame(static_cast<uint8_t>(subtype), hash, pInBuffer, interval);
    return result;
}


nn::Result CancelPeriodicActionFrame() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    auto result = g_DetectManager->CancelPeriodicActionFrame();
    return result;
}


nn::Result GetActionFrame(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size,
        size_t* pOutSize, uint32_t rxId, uint16_t* pOutChannel, int16_t* pOutRssi, nn::os::Tick* pOutTick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectGetActionFrameManager);
    if( pOutSrcMac == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutSrcMac must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutBuf == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutBuf must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutSize == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutSize must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutChannel == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutChannel must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutRssi == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRssi must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pOutTick == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutTick must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pOutBuf), size);
    uint32_t outSize = 0;  // 受信データサイズが32bit以上になることはないので、uint32_tで十分
    nn::wlan::detail::SfdlMacAddress mac;  // 出力用
    uint16_t outChannel;
    int16_t outRssi;
    int64_t outTick;
    auto result = g_DetectGetActionFrameManager->GetActionFrame(&mac, pOutBuffer, &outSize, rxId, &outChannel, &outRssi, &outTick);

    pOutSrcMac->Set(mac.data);
    *pOutSize = static_cast<size_t>(outSize);
    *pOutChannel = outChannel;
    *pOutRssi = outRssi;
    nn::os::Tick retTick(outTick);
    *pOutTick = retTick;

    return result;
}


nn::Result CancelGetActionFrame(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    auto result = g_DetectManager->CancelGetActionFrame(rxId);
    return result;
}


nn::Result CreateRxEntryForActionFrame(uint32_t* pOutRxId, const uint16_t pAfTypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pAfTypes == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pAfTypes must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( count > ActionFrameTypeCountMax )
    {
        NN_SDK_REQUIRES(false, "%s: The number of ActionFrameType must be less than %d\n", __FUNCTION__, ActionFrameTypeCountMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }
    if( capacity > ActionFrameRxCapacityMax )
    {
        NN_SDK_REQUIRES(false, "%s: Capacity of RxEntry must be less than %d\n", __FUNCTION__, ActionFrameRxCapacityMax + 1);
        return nn::wlan::ResultInvalidArgument();
    }
    for( int i = 0; i < count; i++ )
    {
        if( pAfTypes[i] < ActionFrameType_Beacon || pAfTypes[i] >= ActionFrameType_End )
        {
            NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
            return nn::wlan::ResultInvalidArgument();
        }
    }

    nn::sf::InArray<uint16_t> InArray(pAfTypes, count);
    uint32_t rxId = 0;
    auto result = g_DetectManager->CreateRxEntryForActionFrame(&rxId, InArray, capacity);
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}


nn::Result DeleteRxEntryForActionFrame(uint32_t rxId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);

    auto result = g_DetectManager->DeleteRxEntryForActionFrame(rxId);

    return result;
}


nn::Result AddSubtypeToRxEntryForActionFrame(uint32_t rxId, ActionFrameType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
    {
        NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    auto result = g_DetectManager->AddSubtypeToRxEntryForActionFrame(rxId, static_cast<uint32_t>(type));

    return result;
}


nn::Result DeleteSubtypeFromRxEntryForActionFrame(uint32_t* pOutRxId, ActionFrameType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pOutRxId == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pOutRxId must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( type < ActionFrameType_Beacon || type >= ActionFrameType_End )
    {
        NN_SDK_REQUIRES(false, "%s: Action frame type is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    uint32_t rxId = 0;
    auto result = g_DetectManager->DeleteSubtypeFromRxEntryForActionFrame(&rxId, static_cast<uint32_t>(type));
    if( result.IsSuccess() )
    {
        *pOutRxId = rxId;
    }

    return result;
}


nn::Result ReserveDetectSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->ReserveDetectSleep();
}


nn::Result CancelDetectSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->CancelDetectSleep();
}


nn::Result RequestSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->RequestSleep();
}


nn::Result RequestWakeUp() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->RequestWakeUp();
}


nn::Result SetActionFrameForSleep(ActionFrameType subtype, DetectHash hash,
        const uint8_t* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( subtype != ActionFrameType_Detect )
    {
        NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pData must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size > DetectActionFramePayloadMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pData), size);

    return g_DetectManager->SetActionFrameForSleep(static_cast<uint8_t>(subtype), hash, pInBuffer);
}


nn::Result SetHashList(const uint64_t* pHashList, uint32_t numOfHash) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pHashList == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pHashList must not be NULL\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( numOfHash > DetectSleepFilterMax )
    {
        NN_SDK_REQUIRES(false, "%s: numOfHash is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pHashList), numOfHash * sizeof(uint64_t));

    return g_DetectManager->SetHashList(pInBuffer);
}


nn::Result SetPeriodicActionFrameCycle(DetectPeriodicAfCycle pattern, DetectPeriodicAfCycleTarget target) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);

    // TODO パラメータチェック

    return g_DetectManager->SetPeriodicActionFrameCycle(pattern, static_cast<uint8_t>(target));
}

nn::Result PutActionFrameOneShotEx(const MacAddress& dstMac, const MacAddress& bssid, ActionFrameType subtype,
        DetectHeader dhp, const uint8_t* pData, size_t size, uint32_t dwellTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( dstMac == MacAddress::CreateZeroMacAddress() )
    {
        NN_SDK_REQUIRES(false, "%s: dstMac must not be Zero mac address\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( subtype != ActionFrameType_Detect )
    {
        NN_SDK_REQUIRES(false, "%s: subtype must be ActionFrameType_Detect\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( pData == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pData must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( size > DetectActionFramePayloadMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    if( dwellTime > 300 )
    {
        NN_SDK_REQUIRES(false, "%s: dwellTime must be less than 301\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pData), size);
    nn::wlan::detail::SfdlMacAddress mac;
    std::memcpy(&mac.data[0], dstMac.GetMacAddressData(), MacAddress::MacAddressSize);
    nn::wlan::detail::SfdlMacAddress sfdlBssid;
    std::memcpy(&sfdlBssid.data[0], bssid.GetMacAddressData(), MacAddress::MacAddressSize);

    auto result = g_DetectManager->PutActionFrameOneShotEx(mac, sfdlBssid, static_cast<uint8_t>(subtype), dhp, pInBuffer, dwellTime);
    return result;
}

nn::Result SetMacAddress(const MacAddress& mac) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    nn::wlan::detail::SfdlMacAddress sfdlMac;
    std::memcpy(&sfdlMac.data[0], mac.GetMacAddressData(), MacAddress::MacAddressSize);
    return g_DetectManager->SetMacAddress(sfdlMac);
}

nn::Result GetTotalRecvCountInSleep(uint64_t* pCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    if( pCount == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pCount must not be null\n", __FUNCTION__);
        return nn::wlan::ResultInvalidArgument();
    }
    uint64_t count = 0;
    nn::Result result = g_DetectManager->GetTotalRecvCountInSleep(&count);
    *pCount = count;

    return result;
}

nn::Result ClearTotalRecvCountInSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DetectManager);
    return g_DetectManager->ClearTotalRecvCountInSleep();
}

}
}
}

