﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_LocalIndirectLayer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/vi/vi_Result.h>
#include "../visrv_Macro.h"
#include "../client/visrv_ClientObject.h"
#include "../native/visrv_ParcelIo.h"
#include "../native/visrv_BinderFunctions.h"
#include "../native/detail/visrv_TransactionIGraphicBufferProducer.h"
#include "visrv_LocalServerObject.h"

#include <utils/Timers.h>

namespace nn{ namespace visrv{ namespace local{

    LocalIndirectLayer::LocalIndirectLayer() NN_NOEXCEPT
        : m_pClientObject(nullptr)
        , m_ProducerHandle(0)
    {
    }

    namespace {

        struct GetBinderHandleArgument
        {
            int count;
            uint32_t handle;
        };

        void GetBinderHandle(flat_binder_object* pObject, void* userPtr) NN_NOEXCEPT
        {
            auto pArgument = reinterpret_cast<GetBinderHandleArgument*>(userPtr);

            pArgument->count++;
            pArgument->handle = pObject->handle;
        }

    }

    nn::Result LocalIndirectLayer::Initialize(
        client::ClientObject* pClient,
        nn::vi::IndirectProducerHandleType handle,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pClient);

        NN_VISRV_PROCESS_START();

        size_t dataSize = 0;
        NN_RESULT_DO(pClient->BindIndirectProducerEndPoint(
            &dataSize,
            g_ParcelWorkspace,
            sizeof(g_ParcelWorkspace),
            handle,
            aruid
        ));
        NN_VISRV_PROCESS_ROLLBACK(pClient->UnbindIndirectProducerEndPoint(handle));

        android::Parcel parcel;
        size_t dataPos = 0;
        NN_RESULT_THROW_UNLESS(
            native::ParcelIo::OpenParcel(&parcel, &dataPos, g_ParcelWorkspace, dataSize) == android::NO_ERROR,
            nn::vi::ResultOperationFailed()
        );
        NN_UTIL_SCOPE_EXIT{ native::ParcelIo::CloseParcel(&parcel); };

        uint32_t binderHandle = 0;
        {
            GetBinderHandleArgument arg = {};
            parcel.foreachObject(GetBinderHandle, &arg);
            NN_SDK_ASSERT_EQUAL(arg.count, 1);
            binderHandle = arg.handle;
        }

        // GraphicBufferProducer の RefCount を上げる
        pClient->IndirectDisplayAdjustRefcount(binderHandle, +1, 1);

        NN_VISRV_PROCESS_SUCCESS();
        m_pClientObject  = pClient;
        m_ProducerHandle = handle;
        m_BinderHandle   = binderHandle;
        NN_RESULT_SUCCESS;
    }

    void LocalIndirectLayer::Finalize() NN_NOEXCEPT
    {
        // GraphicBufferProducer の RefCount を下げる
        m_pClientObject->IndirectDisplayAdjustRefcount(m_BinderHandle, -1, true);

        // 先に IndirectProducer が破棄されている場合には何も起こらない
        m_pClientObject->UnbindIndirectProducerEndPoint(m_ProducerHandle);

        m_pClientObject  = nullptr;
        m_ProducerHandle = 0;
        m_BinderHandle   = 0;
    }

    bool LocalIndirectLayer::IsInitialized() const NN_NOEXCEPT
    {
        return m_pClientObject != nullptr;
    }

    nn::vi::IndirectProducerHandleType LocalIndirectLayer::GetProducerHandle() const NN_NOEXCEPT
    {
        return m_ProducerHandle;
    }

    uint64_t LocalIndirectLayer::GetLayerResourceId() const NN_NOEXCEPT
    {
        return m_ProducerHandle;
    }

    nn::Result LocalIndirectLayer::SetBufferCount(int count) NN_NOEXCEPT
    {
        //see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            // bufferCount
            request.writeInt32(static_cast<int32_t>(count));
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_SetBufferCount,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        native::detail::TransactionIGraphicBufferProducer::SetBufferCountReply reply = {};
        NN_RESULT_DO(native::detail::TransactionIGraphicBufferProducer::DecodeSetBufferCountReply(&reply, pReplyBuffer, bufferSize));
        NN_RESULT_THROW_UNLESS(reply.result == android::NO_ERROR, nn::vi::ResultOperationFailed());

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::Connect() NN_NOEXCEPT
    {
        // see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            // listener == 0
            request.writeInt32(0);
            // connect with cpu
            request.writeInt32(NATIVE_WINDOW_API_CPU);
            // NOT controlled by app (see SurfaceControl::getSurface())
            request.writeInt32(0);
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_Connect,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        int32_t result = 0;
        {
            android::Parcel reply;
            size_t replyPosition = 0;
            NN_RESULT_THROW_UNLESS(android::NO_ERROR == native::ParcelIo::OpenParcel(&reply, &replyPosition, pReplyBuffer, bufferSize), nn::vi::ResultOperationFailed());
            NN_UTIL_SCOPE_EXIT{ native::ParcelIo::CloseParcel(&reply); };

            reply.readInplace(sizeof(android::IGraphicBufferProducer::QueueBufferOutput)); // QueueBufferOutput を読み飛ばす
            result = reply.readInt32();
        }

        NN_RESULT_THROW_UNLESS(result == android::NO_ERROR, nn::vi::ResultOperationFailed());
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::Disconnect() NN_NOEXCEPT
    {
        //see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            // disconnect cpu
            request.writeInt32(NATIVE_WINDOW_API_CPU);
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_Disconnect,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        int32_t result = 0;
        {
            android::Parcel reply;
            size_t replyPosition = 0;
            NN_RESULT_THROW_UNLESS(android::NO_ERROR == native::ParcelIo::OpenParcel(&reply, &replyPosition, pReplyBuffer, bufferSize), nn::vi::ResultOperationFailed());
            NN_UTIL_SCOPE_EXIT{ native::ParcelIo::CloseParcel(&reply); };

            result = reply.readInt32();
        }
        NN_RESULT_THROW_UNLESS(result == android::NO_ERROR, nn::vi::ResultOperationFailed());

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::SetBuffer(int index, const android::sp<android::GraphicBuffer>& pBuffer) NN_NOEXCEPT
    {
        // pBuffer == nullptr の場合、バッファのクリアが行われる。

        //see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            // slot number
            request.writeInt32(static_cast<int32_t>(index));
            if(pBuffer != nullptr)
            {
                // buffer exists
                request.writeInt32(1);
                // export buffer
                request.write(*pBuffer);
            }
            else
            {
                // buffer not exists
                request.writeInt32(0);
            }
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_SetPreallocatedBuffer,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        int32_t result = 0;
        {
            android::Parcel reply;
            size_t replyPosition = 0;
            NN_RESULT_THROW_UNLESS(android::NO_ERROR == native::ParcelIo::OpenParcel(&reply, &replyPosition, pReplyBuffer, bufferSize), nn::vi::ResultOperationFailed());
            NN_UTIL_SCOPE_EXIT{ native::ParcelIo::CloseParcel(&reply); };

            result = reply.readInt32();
        }
        NN_RESULT_THROW_UNLESS(result == android::NO_ERROR, nn::vi::ResultOperationFailed());

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::RequestBuffer(int index) NN_NOEXCEPT
    {
        //see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            // slot number
            request.writeInt32(static_cast<int32_t>(index));
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_RequestBuffer,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        native::detail::TransactionIGraphicBufferProducer::RequestBufferReply reply = {};
        NN_RESULT_DO(native::detail::TransactionIGraphicBufferProducer::DecodeRequestBufferReply(&reply, pReplyBuffer, bufferSize));
        NN_RESULT_THROW_UNLESS(reply.result == android::NO_ERROR, nn::vi::ResultOperationFailed());

        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::Present(int index, const android::sp<android::Fence>& fence, const nn::vi::CropRegion& crop, int scalingMode, nn::vi::ImageTransformType transform, int presentInterval) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(fence);

        int64_t timestamp = systemTime(SYSTEM_TIME_MONOTONIC);
        bool isAutoTimestamp = true;
        android::Rect nativeRect;
        nativeRect.left   = crop.x;
        nativeRect.top    = crop.y;
        nativeRect.right  = crop.x + crop.width;
        nativeRect.bottom = crop.y + crop.height;
        android::IGraphicBufferProducer::QueueBufferInput input(
            timestamp,
            isAutoTimestamp,
            nativeRect,
            scalingMode,
            transform,
            false /* async */,
            presentInterval,
            fence,
            0 /* flags */
        );


        //see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            // write present index
            request.writeInt32(static_cast<int32_t>(index));
            // write input
            request.write(input);
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_QueueBuffer,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        native::detail::TransactionIGraphicBufferProducer::QueueBufferReply reply = {};
        NN_RESULT_DO(native::detail::TransactionIGraphicBufferProducer::DecodeQueueBufferReply(&reply, pReplyBuffer, bufferSize));

        NN_RESULT_THROW_UNLESS(reply.result == android::NO_ERROR, nn::vi::ResultOperationFailed());
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::Acquire(int* pOutIndex, android::sp<android::Fence>* pOutFence, int expectedWidth, int expectedHeight) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutIndex);
        NN_SDK_REQUIRES_NOT_NULL(pOutFence);

        int async = 1; // IsSytemEventAttached
        static const int usage = GRALLOC_USAGE_HW_RENDER | GRALLOC_USAGE_HW_TEXTURE;

        //see IGraphicBufferProducer.cpp
        android::Parcel request;
        {
            // write interface token
            request.writeInterfaceToken(native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor());
            request.writeInt32(async);
            request.writeInt32(expectedWidth);
            request.writeInt32(expectedHeight);
            request.writeInt32(1 /*DisplayFormat*/);
            request.writeInt32(usage);
        }

        NN_STATIC_ASSERT(sizeof(g_ParcelWorkspace) % 2 == 0);
        size_t bufferSize = sizeof(g_ParcelWorkspace) / 2;
        void* pRequestBuffer = g_ParcelWorkspace;
        void* pReplyBuffer   = g_ParcelWorkspace + bufferSize;

        size_t requestSize = 0;
        native::ParcelIo::WriteParcel(&requestSize, pRequestBuffer, bufferSize, &request);

        NN_RESULT_THROW_UNLESS(requestSize > 0, nn::vi::ResultOperationFailed());
        m_pClientObject->IndirectDisplayTransactParcel(
            m_BinderHandle,
            native::IGraphicBufferProducerFunctionCode_DequeueBuffer,
            g_ParcelWorkspace,
            requestSize,
            pReplyBuffer,
            bufferSize,
            0 /*flags*/
        );

        int index = -1;
        android::sp<android::Fence> fence = new android::Fence();
        int32_t result = 0;
        {
            android::Parcel reply;
            size_t replyPosition = 0;
            NN_RESULT_THROW_UNLESS(android::NO_ERROR == native::ParcelIo::OpenParcel(&reply, &replyPosition, pReplyBuffer, bufferSize), nn::vi::ResultOperationFailed());
            NN_UTIL_SCOPE_EXIT{ native::ParcelIo::CloseParcel(&reply); };

            index = reply.readInt32();
            int hasFence = reply.readInt32();
            if(hasFence)
            {
                reply.read(*fence);
            }
            result = reply.readInt32();
        }
        NN_RESULT_THROW_UNLESS(result == android::NO_ERROR, nn::vi::ResultOperationFailed());

        *pOutIndex = index;
        *pOutFence = fence;
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::GetAcquirableEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

        nn::sf::NativeHandle h;
        nn::sf::Out<nn::sf::NativeHandle> outHandle(&h);
        m_pClientObject->IndirectDisplayGetNativeHandle(m_BinderHandle, native::IGraphicBufferProducerFunctionCode_GetBufferReleasedEvent, outHandle);
        NN_RESULT_THROW_UNLESS(h.GetOsHandle() != nn::os::InvalidNativeHandle, nn::vi::ResultOperationFailed());

        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::SetLayerStackFlags(nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        NN_UNUSED(stacks);
        NN_RESULT_SUCCESS;
    }

    nn::Result LocalIndirectLayer::GetLayerStackFlags(nn::vi::LayerStackFlagType* pOutStacks) NN_NOEXCEPT
    {
        *pOutStacks = 0;
        NN_RESULT_SUCCESS;
    }

}}}

//namespace android{
//    size_t IGraphicBufferProducer::QueueBufferInput::getFlattenedSize() const {
//        return sizeof(timestamp)
//             + sizeof(isAutoTimestamp)
//             + sizeof(crop)
//             + sizeof(scalingMode)
//             + sizeof(transform)
//             + sizeof(flags)
//             + sizeof(async)
//             + sizeof(interval)
//             + fence->getFlattenedSize();
//    }
//
//    size_t IGraphicBufferProducer::QueueBufferInput::getFdCount() const {
//        return fence->getFdCount();
//    }
//
//    status_t IGraphicBufferProducer::QueueBufferInput::flatten(
//            void*& buffer, size_t& size, int*& fds, size_t& count) const
//    {
//        if (size < getFlattenedSize()) {
//            return NO_MEMORY;
//        }
//        FlattenableUtils::write(buffer, size, timestamp);
//        FlattenableUtils::write(buffer, size, isAutoTimestamp);
//        FlattenableUtils::write(buffer, size, crop);
//        FlattenableUtils::write(buffer, size, scalingMode);
//        FlattenableUtils::write(buffer, size, transform);
//        FlattenableUtils::write(buffer, size, flags);
//        FlattenableUtils::write(buffer, size, async);
//        FlattenableUtils::write(buffer, size, interval);
//        return fence->flatten(buffer, size, fds, count);
//    }
//
//    status_t IGraphicBufferProducer::QueueBufferInput::unflatten(
//            void const*& buffer, size_t& size, int const*& fds, size_t& count)
//    {
//        size_t minNeeded =
//                  sizeof(timestamp)
//                + sizeof(isAutoTimestamp)
//                + sizeof(crop)
//                + sizeof(scalingMode)
//                + sizeof(transform)
//                + sizeof(flags)
//                + sizeof(async)
//                + sizeof(interval);
//
//        if (size < minNeeded) {
//            return NO_MEMORY;
//        }
//
//        FlattenableUtils::read(buffer, size, timestamp);
//        FlattenableUtils::read(buffer, size, isAutoTimestamp);
//        FlattenableUtils::read(buffer, size, crop);
//        FlattenableUtils::read(buffer, size, scalingMode);
//        FlattenableUtils::read(buffer, size, transform);
//        FlattenableUtils::read(buffer, size, flags);
//        FlattenableUtils::read(buffer, size, async);
//        FlattenableUtils::read(buffer, size, interval);
//
//        fence = new Fence();
//        return fence->unflatten(buffer, size, fds, count);
//    }
//}
