﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "visrv_IndirectLayer.h"

namespace nn{ namespace visrv{ namespace indirect{

    // pLayer を初期化して pClient に登録します。
    //   - pLayer を pClient を Owner として初期化
    //   - pClinet に pLayer を登録
    nn::Result CreateIndirectLayerImpl(
        nn::vi::IndirectLayerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        client::ClientObject* pClient
    ) NN_NOEXCEPT;

    // pLayer を pClient から登録解除して破棄します。
    // ProducerEndPoint, ConsumerEndPoint が作成済の場合破棄されます。
    //   - pLayer の ProducerEndPoint を破棄
    //   - pLayer の ConsumerEndPoint を破棄
    //   - pClinet から pLayer を登録解除
    //   - pLayer を破棄
    void DestroyIndirectLayerImpl(IndirectLayer* pLayer) NN_NOEXCEPT;

    // pLayer の ProducerEndPoint を作成します。
    //   - pLayer の ProducerEndPoint に UserAruid を登録
    nn::Result CreateIndirectProducerEndPointImpl(
        nn::vi::IndirectProducerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT;

    // pLayer の ProducerEndPoint を破棄します。
    // ProducerEndPoint がバインド済の場合、バインド解除します。
    //   - pLayer の ProducerEndPoint のバインド解除
    //   - pLayer の ProducerEndPoint の UserAruid 登録解除
    void DestroyIndirectProducerEndPointImpl(IndirectLayer* pLayer) NN_NOEXCEPT;

    // pLayer の ConsumerEndPoint を作成します。
    //   - pLayer の ConsumerEndPoint に UserAruid を登録
    nn::Result CreateIndirectConsumerEndPointImpl(
        nn::vi::IndirectConsumerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT;

    // pLayer の ConsumerEndPoint を破棄します。
    // ConsumerEndPoint がバインド済の場合、バインド解除します。
    //   - pLayer の ConsumerEndPoint のバインド解除
    //   - pLayer の ConsumerEndPoint の UserAruid 登録解除
    void DestroyIndirectConsumerEndPointImpl(IndirectLayer* pLayer) NN_NOEXCEPT;

    // pClinet を pLayer の ProducerEndPoint にバインドします。
    //   - pLayer の ProducerEndPoint に pClient を登録
    //   - pClient のバインド済 ProducerEndPoint に pLayer を登録
    //   - pClinet の BinderTable に pLayer を登録
    nn::Result BindIndirectProducerEndPointImpl(
        int* pOutBinderEntryIndex,
        IndirectLayer* pLayer,
        client::ClientObject* pClient,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT;

    // pLayer の ProducerEndPoint と pClient のバインドを解除します。
    //   - pLayer の ProducerEndPoint を切断
    //   - pClient の BinderTable から pLayer を登録解除
    //   - pClient のバインド済 ProducerEndPoint から pLayer を登録解除
    //   - pLayer の ProducerEndPoint から pClient を登録解除
    void UnbindIndirectProducerEndPointImpl(IndirectLayer* pLayer) NN_NOEXCEPT;

    // pClient を pLayer の ConsumerEndPoint にバインドします（ MapDefer 用）。
    //   - pLayer の ConsumerEndPoint に pClinet を登録
    //   - pClinet のバインド済 ConsumerEndPoint に pLayer を登録
    nn::Result BindIndirectConsumerEndPointMapDeferImpl(
        IndirectLayer* pLayer,
        client::ClientObject* pClientObject,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT;

    // pClient を pLayer の ConsumerEndPoint にバインドします（ Transfer 用）。
    //   - pLayer の ConsumerEndPoint に pClinet を登録
    //   - pClinet のバインド済 ConsumerEndPoint に pLayer を登録
    nn::Result BindIndirectConsumerEndPointTransferImpl(
        nn::os::NativeHandle* pOutIsBufferReadyEventHandle,
        bool* pOutIsBufferReadyEventHandleManaged,
        IndirectLayer* pLayer,
        client::ClientObject* pClientObject,
        nn::applet::AppletResourceUserId aruid,
        nn::os::NativeHandle bufferTransferMemoryHandle,
        bool* pIsBufferTransferMemoryHandleManaged,
        size_t bufferTransferMemorySize,
        int width,
        int height
    ) NN_NOEXCEPT;

    // pLayer の ConsumerEndPoint と pClient のバインドを解除します。
    //   - pLayer の ConsumerEndPoint を切断
    //   - pClinet のバインド済 ConsumerEndPoint から pLayer を登録解除
    //   - pLayer の ConsumerEndPoint から pClient を登録解除
    void UnbindIndirectConsumerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT;

    void DisconnectIndirectProducerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT;
    //void DisconnectIndirectConsumerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT;

    nn::Result FlipIndirectProducerImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT;
    nn::Result KickIndirectCopyImageTransferImpl(size_t* pOutSize, size_t* pOutStride, indirect::IndirectLayer* pLaeyr, float sourceRectX, float sourceRectY, float sourceRectWidth, float sourceRectHeight) NN_NOEXCEPT;
    nn::Result SetIndirectProducerFlipOffset(indirect::IndirectLayer* pLayer, nn::TimeSpan offset) NN_NOEXCEPT;


}}}
