﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectLayerOperationImpl.h"

#include <nn/vi/vi_Result.h>
#include "../visrv_Config.h"
#include "../visrv_Log.h"
#include "../client/visrv_ClientObject.h"

#define NN_VISRV_LOG_INDIRECT_DO(messageFormat,...)                             \
    NN_VISRV_LOG_INDIRECT(messageFormat "\n", __VA_ARGS__);                     \
    NN_UTIL_SCOPE_EXIT{                                                         \
            NN_VISRV_LOG_INDIRECT(messageFormat " -> DONE\n", __VA_ARGS__);     \
    };

#define NN_VISRV_LOG_INDIRECT_RESULT(var,messageFormat,resultFormat,...)                    \
    bool var = false;                                                                       \
    NN_VISRV_LOG_INDIRECT(messageFormat "\n", __VA_ARGS__);                                 \
    NN_UTIL_SCOPE_EXIT{                                                                     \
        if(var)                                                                             \
        {                                                                                   \
            NN_VISRV_LOG_INDIRECT(messageFormat " -> " resultFormat "\n", __VA_ARGS__);     \
        }                                                                                   \
        else                                                                                \
        {                                                                                   \
            NN_VISRV_LOG_INDIRECT(messageFormat " -> FAILED\n", __VA_ARGS__);               \
        }                                                                                   \
    };

#define NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(var,messageFormat,...) \
    NN_VISRV_LOG_INDIRECT_RESULT(var, messageFormat, "%s", __VA_ARGS__, (var ? "SUCCESS" : "FAILED"))


namespace nn{ namespace visrv{ namespace indirect{

    nn::Result CreateIndirectLayerImpl(
        nn::vi::IndirectLayerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        client::ClientObject* pClient
    ) NN_NOEXCEPT
    {
        nn::vi::IndirectLayerHandleType h = 0;
        NN_VISRV_LOG_INDIRECT_RESULT(isSuccess, "Creating IndirectLayer", "#%lld", h);

        // pLayer を初期化
        NN_RESULT_DO(pLayer->Initialize(&h, pClient));

        // pClient に登録
        auto pList = client::ClientObjectAccessor::GetOwnIndirectLayerList(pClient);
        pList->push_back(*pLayer);

        *pOutHandle = h;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void DestroyIndirectLayerImpl(IndirectLayer* pLayer) NN_NOEXCEPT
    {
        auto h = pLayer->GetHandle();
        NN_VISRV_LOG_INDIRECT_DO("Destroying IndirectLayer(#%llu)", h);
        NN_UNUSED(h);

        // 先に ProducerEndPoint と ConsumerEndPoint を破棄する
        if(pLayer->GetProducerState() != indirect::IndirectProducerEndPointState_Invalid)
        {
            DestroyIndirectProducerEndPointImpl(pLayer);
        }
        if(pLayer->GetConsumerState() != indirect::IndirectConsumerEndPointState_Invalid)
        {
            DestroyIndirectConsumerEndPointImpl(pLayer);
        }

        // ClientObject から登録解除
        auto pList = client::ClientObjectAccessor::GetOwnIndirectLayerList(pLayer->GetOwnerClient());
        pList->erase(pList->iterator_to(*pLayer));

        // Layer を破棄
        pLayer->Finalize();
    }

    nn::Result CreateIndirectProducerEndPointImpl(
        nn::vi::IndirectProducerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        nn::vi::IndirectProducerHandleType h = 0;
        NN_VISRV_LOG_INDIRECT_RESULT(isSuccess, "Creating IndirectProducerEndPoint(#%llu,%08llX)", "#%llu", pLayer->GetHandle(), aruid, h);

        NN_RESULT_THROW_UNLESS(
            pLayer->GetProducerState() == IndirectProducerEndPointState_Invalid,
            nn::vi::ResultAlreadyOpened()
        );

        NN_RESULT_DO(pLayer->RegisterProducerAruid(&h, aruid));

        *pOutHandle = h;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void DestroyIndirectProducerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Destroying IndirectProducerEndPoint(#%llu)", pLayer->GetProducerEndPointHandle());

        if(pLayer->GetProducerState() != indirect::IndirectProducerEndPointState_Unbound)
        {
            UnbindIndirectProducerEndPointImpl(pLayer);
        }

        pLayer->UnregisterProducerAruid();
    }

    nn::Result CreateIndirectConsumerEndPointImpl(
        nn::vi::IndirectConsumerHandleType* pOutHandle,
        IndirectLayer* pLayer,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        nn::vi::IndirectConsumerHandleType h = 0;
        NN_VISRV_LOG_INDIRECT_RESULT(isSuccess, "Creating IndirectConsumerEndPoint(#%llu,%08llX)", "#%llu", pLayer->GetHandle(), aruid, h);

        NN_RESULT_THROW_UNLESS(
            pLayer->GetConsumerState() == IndirectConsumerEndPointState_Invalid,
            nn::vi::ResultAlreadyOpened()
        );

        NN_RESULT_DO(pLayer->RegisterConsumerAruid(&h, aruid));

        *pOutHandle = h;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void DestroyIndirectConsumerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Destroying IndirectConsumerEndPoint(#%llu)", pLayer->GetConsumerEndPointHandle());

        if(pLayer->GetConsumerState() != indirect::IndirectConsumerEndPointState_Unbound)
        {
            UnbindIndirectConsumerEndPointImpl(pLayer);
        }

        pLayer->UnregisterConsumerAruid();
    }

    nn::Result BindIndirectProducerEndPointImpl(
        int* pOutBinderEntryIndex,
        IndirectLayer* pLayer,
        client::ClientObject* pClient,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        int entryIndex = 0;
        NN_VISRV_LOG_INDIRECT_RESULT(isSuccess, "Binding IndirectProducerEndPoint(#%llu,%08llX)", "<%d>", pLayer->GetProducerEndPointHandle(), aruid, entryIndex);

        NN_RESULT_THROW_UNLESS(
            pLayer->GetProducerState() == IndirectProducerEndPointState_Unbound,
            nn::vi::ResultAlreadyOpened()
        );

        NN_RESULT_DO(pLayer->BindProducer(pClient, aruid));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pLayer->UnbindProducer();
            }
        };

        // ClientObject に登録
        auto pList = client::ClientObjectAccessor::GetProducerBoundIndirectLayerList(pClient);
        pList->push_back(*pLayer);
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pList->erase(pList->iterator_to(*pLayer));
            }
        };

        // BinderTable に登録
        auto pTable = client::ClientObjectAccessor::GetIndirectProducerBinderTable(pClient);
        pTable->Register(&entryIndex, pLayer);
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pTable->Unregister(entryIndex);
            }
        };

        *pOutBinderEntryIndex = entryIndex;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void UnbindIndirectProducerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Unbinding IndirectProducerEndPoint(#%llu)", pLayer->GetProducerEndPointHandle());

        if(pLayer->GetProducerState() != indirect::IndirectProducerEndPointState_Bound)
        {
            DisconnectIndirectProducerEndPointImpl(pLayer);
        }

        // BinderTable 登録解除
        auto pTable = client::ClientObjectAccessor::GetIndirectProducerBinderTable(pLayer->GetProducerClient());
        int entryIndex = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pTable->Find(&entryIndex, pLayer));
        pTable->Unregister(entryIndex);

        // ClientObject から登録解除
        auto pList = client::ClientObjectAccessor::GetProducerBoundIndirectLayerList(pLayer->GetProducerClient());
        pList->erase(pList->iterator_to(*pLayer));

        pLayer->UnbindProducer();
    }

    nn::Result BindIndirectConsumerEndPointMapDeferImpl(
        IndirectLayer* pLayer,
        client::ClientObject* pClientObject,
        nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(isSuccess, "Binding IndirectConsumerEntPoint/MapDefer(#%llu,%08llX)", pLayer->GetConsumerEndPointHandle(), aruid);

        NN_RESULT_THROW_UNLESS(
            pLayer->GetConsumerState() == indirect::IndirectConsumerEndPointState_Unbound,
            nn::vi::ResultAlreadyOpened()
        );

        // Layer に登録
        NN_RESULT_DO(pLayer->BindConsumerMapDefer(pClientObject, aruid));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pLayer->UnbindConsumer();
            }
        };

        // ClientObject に登録
        auto pList = client::ClientObjectAccessor::GetConsumerBoundIndirectLayerList(pClientObject);
        pList->push_back(*pLayer);
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pList->erase(pList->iterator_to(*pLayer));
            }
        };


        isSuccess = true;
        NN_RESULT_SUCCESS;
    }


    nn::Result BindIndirectConsumerEndPointTransferImpl(
        nn::os::NativeHandle* pOutIsBufferReadyEventHandle,
        bool* pOutIsBufferReadyEventHandleManaged,
        IndirectLayer* pLayer,
        client::ClientObject* pClientObject,
        nn::applet::AppletResourceUserId aruid,
        nn::os::NativeHandle bufferTransferMemoryHandle,
        bool* pIsBufferTransferMemoryHandleManaged,
        size_t bufferTransferMemorySize,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(isSuccess, "Binding IndirectConsumerEntPoint/Transfer(#%llu,%08llX)", pLayer->GetConsumerEndPointHandle(), aruid);

        NN_RESULT_THROW_UNLESS(
            pLayer->GetConsumerState() == indirect::IndirectConsumerEndPointState_Unbound,
            nn::vi::ResultAlreadyOpened()
        );

        // Layer に登録
        NN_RESULT_DO(pLayer->BindConsumerTransfer(
            pOutIsBufferReadyEventHandle,
            pOutIsBufferReadyEventHandleManaged,
            pClientObject,
            aruid,
            bufferTransferMemoryHandle,
            pIsBufferTransferMemoryHandleManaged,
            bufferTransferMemorySize,
            width,
            height
        ));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pLayer->UnbindConsumer();
            }
        };

        // ClientObject に登録
        auto pList = client::ClientObjectAccessor::GetConsumerBoundIndirectLayerList(pClientObject);
        pList->push_back(*pLayer);
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                pList->erase(pList->iterator_to(*pLayer));
            }
        };

        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void UnbindIndirectConsumerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Unbinding IndirectConsumerEndPoint(#%llu)", pLayer->GetConsumerEndPointHandle());

        // Disconnect
        //if(pLayer->GetConsumerState() != indirect::IndirectConsumerEndPointState_Bound)
        //{
        //    DisconnectIndirectConsumerEndPointImpl(pLayer);
        //}

        // ClientObject から登録解除
        auto pList = client::ClientObjectAccessor::GetConsumerBoundIndirectLayerList(pLayer->GetConsumerClient());
        pList->erase(pList->iterator_to(*pLayer));

        pLayer->UnbindConsumer();
    }

    void DisconnectIndirectProducerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Disconnecting IndirectProducerEndPoint(#%llu)", pLayer->GetProducerEndPointHandle());

        pLayer->DisconnectProducer();
    }

    //void DisconnectIndirectConsumerEndPointImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    //{
    //    NN_VISRV_LOG_INDIRECT_DO("Disconnecting IndirectConsumerEndPoint(#%llu)", pLayer->GetConsumerEndPointHandle());

    //    pLayer->DisconnectConsumer();
    //}

    nn::Result SetIndirectProducerFlipOffset(indirect::IndirectLayer* pLayer, nn::TimeSpan offset) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Set IndirectProducerFlipOffset(#%llu,#%llu) %lldus\n", pLayer->GetHandle(), pLayer->GetProducerEndPointHandle(), offset.GetMicroSeconds());
        NN_RESULT_THROW_UNLESS(
            pLayer->GetProducerState() != indirect::IndirectProducerEndPointState_Invalid,
            nn::vi::ResultDenied()
        );
        NN_RESULT_THROW_UNLESS(offset >= 0, nn::vi::ResultInvalidRange());

        pLayer->SetProducerFlipOffset(offset);
        NN_RESULT_SUCCESS;
    }

    nn::Result FlipIndirectProducerImpl(indirect::IndirectLayer* pLayer) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_FLIP("Flip IndirectProducerEndPoint(#%llu/#%llu)\n", pLayer->GetHandle(), pLayer->GetProducerEndPointHandle());
        NN_RESULT_THROW_UNLESS(
            pLayer->GetProducerState() == indirect::IndirectProducerEndPointState_Connected,
            nn::vi::ResultDenied()
        );
        NN_RESULT_DO(pLayer->FlipStagedBuffer());
        NN_RESULT_SUCCESS;
    }

    nn::Result KickIndirectCopyImageTransferImpl(size_t* pOutSize, size_t* pOutStride, indirect::IndirectLayer* pLayer, float sourceRectX, float sourceRectY, float sourceRectWidth, float sourceRectHeight) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_DO("Kicking IndirectCopyImage/Transfer(#%llu,#%llu)", pLayer->GetHandle(), pLayer->GetConsumerEndPointHandle());
        NN_RESULT_THROW_UNLESS(
            pLayer->GetConsumerState() == indirect::IndirectConsumerEndPointState_Bound,
            nn::vi::ResultDenied()
        );
        NN_RESULT_THROW_UNLESS(
            pLayer->GetProducerState() == indirect::IndirectProducerEndPointState_Connected,
            nn::vi::ResultNotReady()
        );
        NN_RESULT_DO(pLayer->KickCopyImageTransfer(pOutSize, pOutStride, sourceRectX, sourceRectY, sourceRectWidth, sourceRectHeight));
        NN_RESULT_SUCCESS;
    }

}}}
