﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectCopyImageTransfer.h"
#include "visrv_IndirectLayer.h"

#include <nn/os/os_TransferMemoryApi.h>
#include <nn/vi/vi_Result.h>
#include <nvrm_channel.h>
#include <nvgr.h>

#include "../visrv_Log.h"
#include "../vic/visrv_VicMemoryPool.h"
#include "../vic/visrv_VicOperation.h"
#include "../native/visrv_SyncpointWaiter.h"

namespace nn{ namespace visrv{ namespace indirect{

    IndirectConsumerBindExStateTransfer::IndirectConsumerBindExStateTransfer() NN_NOEXCEPT
        : m_pBufferMemory(nullptr)
        , m_BufferMemorySize(0)
        , m_MemoryPool()
        , m_ImageBuffer()
    {
        std::memset(&m_IsBufferReadySystemEvent, 0, sizeof(m_IsBufferReadySystemEvent));
        std::memset(&m_BufferTransferMemory, 0, sizeof(m_BufferTransferMemory));
        m_Parameter.SetDefault();
    }

    nn::Result IndirectConsumerBindExStateTransfer::Initialize(
        nn::os::NativeHandle* pOutIsBufferReadyEventHandle,
        bool* pOutIsBufferReadyEventHandleManaged,
        nn::os::NativeHandle bufferTransferMemoryHandle,
        bool* pIsBufferTransferMemoryHandleManaged,
        size_t bufferTransferMemorySize,
        int width,
        int height
        ) NN_NOEXCEPT
    {
        bool isSuccess = false;

        NN_RESULT_TRY(nn::os::CreateSystemEvent(&m_IsBufferReadySystemEvent, nn::os::EventClearMode_ManualClear, true))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::vi::ResultOperationFailed());
            }
        NN_RESULT_END_TRY;
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                nn::os::DestroySystemEvent(&m_IsBufferReadySystemEvent);
            }
        };

        nn::os::AttachTransferMemory(&m_BufferTransferMemory, bufferTransferMemorySize, bufferTransferMemoryHandle, *pIsBufferTransferMemoryHandleManaged);
        *pIsBufferTransferMemoryHandleManaged = false;
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                nn::os::DestroyTransferMemory(&m_BufferTransferMemory);
            }
        };

        NN_RESULT_TRY(nn::os::MapTransferMemory(&m_pBufferMemory, &m_BufferTransferMemory, nn::os::MemoryPermission_ReadOnly))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::vi::ResultOperationFailed());
            }
        NN_RESULT_END_TRY;
        m_BufferMemorySize = bufferTransferMemorySize;
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                nn::os::UnmapTransferMemory(&m_BufferTransferMemory);
                m_pBufferMemory = nullptr;
                m_BufferMemorySize = 0;
            }
        };

        NN_RESULT_DO(m_MemoryPool.Initialize(&vic::g_VicModule, m_pBufferMemory, m_BufferMemorySize));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_MemoryPool.Finalize();
            }
        };

        vic::VicImageBufferInfo imgInfo = {};
        imgInfo.width  = width;
        imgInfo.height = height;
        imgInfo.format = vic::VicImageFormat_Rgba8;
        NN_RESULT_DO(m_ImageBuffer.Initialize(&vic::g_VicModule, imgInfo, &m_MemoryPool, 0, m_BufferMemorySize));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_ImageBuffer.Finalize();
            }
        };


        // 最初はシグナル状態
        nn::os::SignalSystemEvent(&m_IsBufferReadySystemEvent);
        *pOutIsBufferReadyEventHandle = nn::os::GetReadableHandleOfSystemEvent(&m_IsBufferReadySystemEvent);
        *pOutIsBufferReadyEventHandleManaged = false;
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void IndirectConsumerBindExStateTransfer::Finalize() NN_NOEXCEPT
    {
        m_ImageBuffer.Finalize();
        m_MemoryPool.Finalize();
        nn::os::DestroyTransferMemory(&m_BufferTransferMemory);
        m_pBufferMemory = nullptr;
        m_BufferMemorySize = 0;

        nn::os::DestroySystemEvent(&m_IsBufferReadySystemEvent);
    }

    void IndirectConsumerBindExStateTransfer::SetParameter(const Parameter& param) NN_NOEXCEPT
    {
        m_Parameter = param;
    }

    void IndirectConsumerBindExStateTransfer::SignalIsBufferReadySystemEvent() NN_NOEXCEPT
    {
        nn::os::SignalSystemEvent(&m_IsBufferReadySystemEvent);
    }

    void IndirectConsumerBindExStateTransfer::ClearIsBufferReadySystemEvent() NN_NOEXCEPT
    {
        nn::os::ClearSystemEvent(&m_IsBufferReadySystemEvent);
    }

    size_t IndirectConsumerBindExStateTransfer::GetImageSize() const NN_NOEXCEPT
    {
        return m_ImageBuffer.GetStride() * m_ImageBuffer.GetHeight();
    }

    size_t IndirectConsumerBindExStateTransfer::GetImageStride() const NN_NOEXCEPT
    {
        return m_ImageBuffer.GetStride();
    }

    void IndirectConsumerBindExStateTransfer::Execute() NN_NOEXCEPT
    {
        NvRmFence fence { NVRM_INVALID_SYNCPOINT_ID, 0 };

        {
            vic::VicCopyOption option = {};
            option.backgroundColor = nn::util::Color4f(0, 0, 0, 0);
            option.filter = vic::VicCopyFilter_Bilinear;
            option.alphaMode = vic::VicCopyAlphaMode_Source;


            vic::VicCopySingleDestinationInfo dst = {};
            vic::VicCopySingleSourceInfo src = {};

            // setup dst
            {
                auto pBuffer = &m_ImageBuffer;
                NN_SDK_ASSERT(pBuffer->IsInitialized());
                dst.pSurface      = pBuffer->GetSurface();
                dst.region.x      = 0;
                dst.region.y      = 0;
                dst.region.width  = pBuffer->GetWidth();
                dst.region.height = pBuffer->GetHeight();
            }

            // setup src
            {
                auto pSrcNativeBuffer = m_Parameter.pSourceGraphicBuffer;
                const NvRmSurface* pSrcSurfaceList = nullptr;
                size_t srcSurfaceCount = 0;

                if(pSrcNativeBuffer == nullptr)
                {
                    NN_VISRV_LOG_INDIRECT_ERR("CopyVicTask: source buffer has no buffer\n");
                    goto KICKERROR;
                }

                nvgr_get_surfaces(pSrcNativeBuffer->handle, &pSrcSurfaceList, &srcSurfaceCount);
                if(srcSurfaceCount != 1)
                {
                    NN_VISRV_LOG_INDIRECT_ERR("CopyVicTask: source buffer has multiple surfaces\n");
                    goto KICKERROR;
                }

                src.pSurface      = &pSrcSurfaceList[0];
                src.region.x      = static_cast<int>(pSrcNativeBuffer->width  * m_Parameter.sourceRectX);
                src.region.y      = static_cast<int>(pSrcNativeBuffer->height * m_Parameter.sourceRectY);
                src.region.width  = static_cast<int>(pSrcNativeBuffer->width  * m_Parameter.sourceRectWidth);
                src.region.height = static_cast<int>(pSrcNativeBuffer->height * m_Parameter.sourceRectHeight);
            }

            // kick
            {
                auto result = vic::VicOperation::CopySingle(&fence, dst, &vic::g_VicModule, src, option);
                if(result.IsFailure())
                {
                    NN_VISRV_LOG_INDIRECT_ERR("CopyVicTask: kick vic failed (%d,%d)\n", result.GetModule(), result.GetDescription());
                    goto KICKERROR;
                }
            }
        }
        // ReleaseFence を更新
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Parameter.pSourceBufferStage->UpdateVicReleaseFence(fence));
            m_Parameter.pSourceGraphicBuffer.clear();
            m_Parameter.pSourceBufferLockCounter->ReleaseLock();
            m_Parameter.pSourceBufferStage       = nullptr;
            m_Parameter.pSourceBufferLockCounter = nullptr;
        }

        // VIC が終わるまで待つ
        {
            native::SyncpointEntry e(fence);
            native::g_SyncpointWaiter.Enqueue(&e);
            e.Wait();
        }

        // Consumer プロセスに完了を通知
        nn::os::SignalSystemEvent(&m_IsBufferReadySystemEvent);

        // コピー先をアンロック
        {
            m_Parameter.pDestinationBoundLockCounter->ReleaseLock();
            m_Parameter.pDestinationBoundLockCounter = nullptr;
        }
        return;

    KICKERROR:
        // 失敗したものは仕方ないので正常終了したことにする
        {
            //NN_ABORT_UNLESS_RESULT_SUCCESS(m_Parameter.pSourceBufferStage->UpdateVicReleaseFence(fence));
            m_Parameter.pSourceGraphicBuffer.clear();
            m_Parameter.pSourceBufferLockCounter->ReleaseLock();
            m_Parameter.pSourceBufferStage       = nullptr;
            m_Parameter.pSourceBufferLockCounter = nullptr;
        }
        nn::os::SignalSystemEvent(&m_IsBufferReadySystemEvent);
        {
            m_Parameter.pDestinationBoundLockCounter->ReleaseLock();
            m_Parameter.pDestinationBoundLockCounter = nullptr;
        }
        return;
    }

}}}
