﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_IndirectConsumerBindState.h"

#include <new>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../visrv_ResourceIdManagement.h"

namespace nn{ namespace visrv{ namespace indirect{

    bool IndirectConsumerBoundId::operator==(const IndirectConsumerBoundId& v) const NN_NOEXCEPT
    {
        return this->value == v.value;
    }

    IndirectConsumerBindState::IndirectConsumerBindState() NN_NOEXCEPT
        : m_BoundMode(IndirectConsumerBoundMode_None)
        , m_BoundId({0})
        , m_LockCounter()
        , m_pExStateTransfer(nullptr)
    {
    }

    void IndirectConsumerBindState::InitializeMapDefer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_BoundMode, IndirectConsumerBoundMode_None);
        m_BoundMode = IndirectConsumerBoundMode_MapDefer;
        m_BoundId   = { AcquireResourceId() };
        m_LockCounter.Initialize(nn::os::GetThreadId(nn::os::GetCurrentThread()));
    }

    nn::Result IndirectConsumerBindState::InitializeTransfer(
        nn::os::NativeHandle* pOutIsBufferReadyEventHandle,
        bool* pOutIsBufferReadyEventHandleManaged,
        nn::os::NativeHandle bufferTransferMemoryHandle,
        bool* pIsBufferTransferMemoryHandleManaged,
        size_t bufferTransferMemorySize,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_BoundMode, IndirectConsumerBoundMode_None);
        NN_STATIC_ASSERT(sizeof(m_ExStorage) >= sizeof(IndirectConsumerBindExStateTransfer));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_ExStorage) >= NN_ALIGNOF(IndirectConsumerBindExStateTransfer));

        bool isSuccess = false;
        m_pExStateTransfer = new(&m_ExStorage) IndirectConsumerBindExStateTransfer();
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_pExStateTransfer->~IndirectConsumerBindExStateTransfer();
                m_pExStateTransfer = nullptr;
            }
        };

        NN_RESULT_DO(m_pExStateTransfer->Initialize(
            pOutIsBufferReadyEventHandle,
            pOutIsBufferReadyEventHandleManaged,
            bufferTransferMemoryHandle,
            pIsBufferTransferMemoryHandleManaged,
            bufferTransferMemorySize,
            width,
            height
        ));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                m_pExStateTransfer->Finalize();
            }
        };

        m_BoundMode = IndirectConsumerBoundMode_Transfer;
        m_BoundId   = { AcquireResourceId() };
        m_LockCounter.Initialize(nn::os::GetThreadId(nn::os::GetCurrentThread()));
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }


    void IndirectConsumerBindState::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_BoundMode, IndirectConsumerBoundMode_None);

        // BoundMode ごとの処理はココ
        if(m_BoundMode == IndirectConsumerBoundMode_MapDefer)
        {
            // なし
        }
        else if(m_BoundMode == IndirectConsumerBoundMode_Transfer)
        {
            m_pExStateTransfer->Finalize();
            m_pExStateTransfer->~IndirectConsumerBindExStateTransfer();
            m_pExStateTransfer = nullptr;
        }

        ReleaseResourceId(m_BoundId.value);
        m_BoundMode = IndirectConsumerBoundMode_None;
        m_BoundId   = { 0 };
        m_LockCounter.Finalize();
    }

    IndirectConsumerBoundMode IndirectConsumerBindState::GetBoundMode() const NN_NOEXCEPT
    {
        return m_BoundMode;
    }

    IndirectConsumerBoundId IndirectConsumerBindState::GetBoundId() const NN_NOEXCEPT
    {
        return m_BoundId;
    }

    IndirectLockCounter* IndirectConsumerBindState::GetLockCounter() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_BoundMode, IndirectConsumerBoundMode_None);
        return &m_LockCounter;
    }

    IndirectConsumerBindExStateTransfer* IndirectConsumerBindState::GetExStateTransfer() NN_NOEXCEPT
    {
        return m_pExStateTransfer;
    }

}}}
