﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Color.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/vi_CropRegion.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include "../../visrv_ResourceIdManagement.h"
#include "../../native/visrv_NativeType.h"

#include <nvrm_surface.h>
#include <nvnflinger_service.h>

namespace nn{ namespace visrv{ namespace fbshare{
    // @brief SMMU_ADDRESS_ALIGNMENT と同じ
    static const size_t SharedBufferMemoryUnitSize = 128 * 1024;
}}}

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{

    struct SharedFrameBufferContentParameter
    {
    public:
        static SharedFrameBufferContentParameter GetDefaultValue() NN_NOEXCEPT;

    public:
        nn::vi::LayerStackFlagType layerStacks;
        nn::vi::CropRegion cropRegion;
        int scalingMode;
        uint32_t transform;
        int presentInterval;
    };

    class SharedFrameBuffer
    {
    public:
        SharedFrameBuffer() NN_NOEXCEPT;

        nn::Result Initialize(
            NvRmDeviceHandle hDevice,
            NvRmMemHandle hMem,
            ptrdiff_t offset,
            size_t size,
            int width,
            int height,
            alloc_device_t* pGrAlloc
        ) NN_NOEXCEPT;

        void Finalize(
            NvRmDeviceHandle hDevice,
            alloc_device_t* pGrAlloc
        ) NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;
        bool IsAttachedToClientLayer() const NN_NOEXCEPT;

        int GetWidth() const NN_NOEXCEPT;
        int GetHeight() const NN_NOEXCEPT;

        NvRmSurface& GetSurface() NN_NOEXCEPT;
        const android::sp<android::GraphicBuffer>& GetGraphicBuffer() NN_NOEXCEPT;

        // @brief このバッファの画像データのパラメータを設定する。
        // @details
        // 画像の内容が書き換わったタイミングで設定される。
        // - Client からの Present 時
        // - クリア API の呼出時
        // など。
        void SetContentParameter(const SharedFrameBufferContentParameter& param) NN_NOEXCEPT;
        // @brief このバッファの画像データのパラメータを取得する。
        SharedFrameBufferContentParameter GetContentParameter() const NN_NOEXCEPT;

        // @brief このバッファを表示するために低レベルレイヤが待つべきフェンスを設定する。
        // @details
        // Client が Present した際に SharedClientLayer により設定される。
        // 低レベルレイヤから Dequeue した際に SharedClientLayer により無効化される。
        void SetAcquireFence(const android::sp<android::Fence>& fence) NN_NOEXCEPT;
        // @brief このバッファを表示するために低レベルレイヤが待つべきフェンスを取得する。
        android::sp<android::Fence> GetAcquireFence() const NN_NOEXCEPT;

        // @brief このバッファに書き込むために Client が待つべきフェンスを設定する。
        // @details
        // 低レベルレイヤから Dequeue した際に SharedLowLevelLayer により設定される。
        // 低レベルレイヤが Release した際に SharedLowLevelLayer により無効化される。
        void SetReleaseFence(const android::sp<android::Fence>& fence) NN_NOEXCEPT;
        // @brief このバッファに書き込むために Client が待つべきフェンスを取得する。
        android::sp<android::Fence> GetReleaseFence() const NN_NOEXCEPT;

        // @brief このバッファが現在提出されている低レベルレイヤの ID を設定する。
        // @details
        // 低レベルレイヤに Present する際に SharedLowLevelLayer により設定される。
        // 低レベルレイヤが Release した際に SharedLowLevelLayer により無効化される。
        // 値の設定は SharedLowLevelLayer が行う。
        void SetPresentedLowLevelLayerId(ResourceId layerId) NN_NOEXCEPT;
        // @brief このバッファが現在提出されている低レベルレイヤの ID を取得する。バッファが Idle の場合、 0 が返る。
        ResourceId GetPresentedLowLevelLayerId() const NN_NOEXCEPT;

        // @brief このバッファがアタッチされている SharedClientLayer のハンドルを設定する。
        // @details
        // アタッチ時にアタッチする SharedClientLayer の ID を設定する。
        // デタッチ時に無効値を設定する。
        // 値の設定は SharedClientLayer が行う。
        void SetAttachedClientLayerHandle(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT;
        // @brief このバッファがアタッチされている SharedClientLayer のハンドルを取得する。アタッチされていない場合、 InvalidHandle が返る。
        nn::vi::fbshare::SharedLayerHandle GetAttachedClientLayerHandle() const NN_NOEXCEPT;

    private:
        static nn::Result InitializeSurface(
            NvRmSurface* pSurface,
            NvRmDeviceHandle hDevice,
            NvRmMemHandle hMem,
            ptrdiff_t offset,
            size_t size,
            int width,
            int height
        ) NN_NOEXCEPT;

        static void FinalizeSurface(NvRmSurface* pSurface, NvRmDeviceHandle hDevice) NN_NOEXCEPT;

        static nn::Result InitializeGraphicBuffer(
            buffer_handle_t* pBufferHandle,
            android::sp<android::GraphicBuffer>* pGraphicBuffer,
            alloc_device_t* pGrAlloc,
            NvRmSurface& surface,
            int width,
            int height
        ) NN_NOEXCEPT;

        static void FinalizeGraphicBuffer(
            buffer_handle_t* bufferHandle,
            android::sp<android::GraphicBuffer>* pGraphicBuffer,
            alloc_device_t* pGrAlloc
        ) NN_NOEXCEPT;

    private:
        NvRmSurface                         m_RmSurface;
        buffer_handle_t                     m_BufferHandle;
        android::sp<android::GraphicBuffer> m_pGraphicBuffer;
        SharedFrameBufferContentParameter   m_ContentParameter;

        android::sp<android::Fence> m_AcquireFence;
        android::sp<android::Fence> m_ReleaseFence;

        ResourceId                  m_PresentedLowLevelLayerId;
        nn::vi::fbshare::SharedLayerHandle   m_AttachedClientLayerHandle;
    };

}}}}
