﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharedClientLayerManager.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include "../../visrv_Macro.h"

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{
    static const int GlobalMultiWaitIndexCount = SharedClientLayerManager::MultiWaitTagMax - SharedClientLayerManager::MultiWaitTagMin + 1;
    static const int RequiredMultiWaitIndexCount = SharedClientLayerManager::Size * SharedClientLayer::MultiWaitTagCount;

    NN_STATIC_ASSERT(GlobalMultiWaitIndexCount >= RequiredMultiWaitIndexCount);

    //------------------------------------------

    nn::Result SharedClientLayerManager::SharedClientLayerPool::Find(SharedClientLayer** pOutValue, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(FindImpl(&pLayer, [&](const SharedClientLayer& e){ return e.GetHandle() == h; }));
        NN_SDK_ASSERT_NOT_NULL(pLayer);
        *pOutValue = pLayer;
        NN_RESULT_SUCCESS;
    }

    //------------------------------------------

    const int SharedClientLayerManager::Size;
    const uintptr_t SharedClientLayerManager::MultiWaitTagMin;
    const uintptr_t SharedClientLayerManager::MultiWaitTagMax;

    SharedClientLayerManager::SharedClientLayerManager() NN_NOEXCEPT
    {
    }

    void SharedClientLayerManager::Initialize(ServerManager* pServerManager) NN_NOEXCEPT
    {
        m_LayerPool.Initialize();
        m_pServerManager = pServerManager;
    }

    void SharedClientLayerManager::Finalize() NN_NOEXCEPT
    {
        m_LayerPool.Finalize();
        m_pServerManager = nullptr;
    }

    nn::Result SharedClientLayerManager::Create(
        SharedClientLayer** pOutLayer,
        client::ClientObject* pOwnerClient,
        nn::applet::AppletResourceUserId userAruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutLayer);
        NN_SDK_REQUIRES_NOT_NULL(pOwnerClient);

        NN_VISRV_PROCESS_START();

        SharedClientLayer* pLayer = nullptr;
        NN_RESULT_DO(m_LayerPool.Allocate(&pLayer));
        NN_VISRV_PROCESS_ROLLBACK(m_LayerPool.Free(pLayer));

        nn::vi::fbshare::SharedLayerHandle h = {};
        NN_RESULT_DO(pLayer->Initialize(&h, pOwnerClient, userAruid));
        NN_VISRV_PROCESS_ROLLBACK(pLayer->Finalize());
        NN_SDK_ASSERT(!h.IsInvalid());
        int index = m_LayerPool.GetIndex(pLayer);
        pLayer->SetupMultiWaitAdaptor(MultiWaitTagMin + SharedClientLayer::MultiWaitTagCount * index);

        NN_VISRV_PROCESS_SUCCESS();
        *pOutLayer = pLayer;
        NN_RESULT_SUCCESS;
    }

    void SharedClientLayerManager::Destroy(SharedClientLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pLayer);

        pLayer->Finalize();
        m_LayerPool.Free(pLayer);
    }

    nn::Result SharedClientLayerManager::Find(SharedClientLayer** pOutValue, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        return m_LayerPool.Find(pOutValue, h);
    }

    void SharedClientLayerManager::HandleMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        NN_SDK_ASSERT_MINMAX(pHolder->userData, MultiWaitTagMin, MultiWaitTagMax);
        int index = (pHolder->userData - MultiWaitTagMin) / SharedClientLayer::MultiWaitTagCount;
        NN_SDK_ASSERT_RANGE(index, 0, Size);

        auto pLayer = m_LayerPool.GetAt(index);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        pLayer->DispatchMultiWait(pHolder);
    }

}}}}

