﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ClientObject.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ContextControl.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>

#include "visrv_ClientUtility.h"
#include "../visrv_Log.h"
#include "../indirect/visrv_IndirectDisplay.h"
#include "../indirect/visrv_IndirectNative.h"
#include "../indirect/visrv_IndirectLayerOperationImpl.h"
#include "../indirect/visrv_IndirectCopyImageMapDefer.h"
#include "../vic/visrv_VicModule.h"
#include "../vic/visrv_VicMemoryPool.h"
#include "../vic/visrv_VicImageBuffer.h"
#include "../vic/visrv_VicOperation.h"
#include "../vic/visrv_VicTaskWorker.h"

#include "../native/visrv_BinderFunctions.h"
#include "../native/visrv_SyncpointWaiter.h"

#define NN_VISRV_LOG_INDIRECT_DO(messageFormat,...)                             \
    NN_VISRV_LOG_INDIRECT(messageFormat "\n", __VA_ARGS__);                     \
    NN_UTIL_SCOPE_EXIT{                                                         \
            NN_VISRV_LOG_INDIRECT(messageFormat " -> DONE\n", __VA_ARGS__);     \
    };

#define NN_VISRV_LOG_INDIRECT_RESULT(var,messageFormat,resultFormat,...)                    \
    bool var = false;                                                                       \
    NN_VISRV_LOG_INDIRECT(messageFormat "\n", __VA_ARGS__);                                 \
    NN_UTIL_SCOPE_EXIT{                                                                     \
        if(var)                                                                             \
        {                                                                                   \
            NN_VISRV_LOG_INDIRECT(messageFormat " -> " resultFormat "\n", __VA_ARGS__);     \
        }                                                                                   \
        else                                                                                \
        {                                                                                   \
            NN_VISRV_LOG_INDIRECT(messageFormat " -> FAILED\n", __VA_ARGS__);               \
        }                                                                                   \
    };

#define NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(var,messageFormat,...) \
    NN_VISRV_LOG_INDIRECT_RESULT(var, messageFormat, "%s", __VA_ARGS__, (var ? "SUCCESS" : "FAILED"))


namespace nn{ namespace visrv{ namespace client{

    namespace {
        // nn::Result (*function)(indirect::IndirectLayer* pLayer)
        //
        // layerHandle に該当する pLayer に対して function を呼ぶ。
        // layerHandle に該当するレイヤが存在しない場合、 nn::vi::ResultNotFound が返される。
        // 見つかった場合、 function(pLayer) の値が返される。
        template<typename F>
        nn::Result ApplyToOwnLayer(
            ClientObject* pClient,
            nn::vi::IndirectLayerHandleType layerHandle,
            F function
            ) NN_NOEXCEPT
        {
            indirect::IndirectLayer* pLayer = nullptr;

            auto pList = ClientObjectAccessor::GetOwnIndirectLayerList(pClient);
            for(auto it = pList->begin(); it != pList->end(); ++it)
            {
                NN_SDK_ASSERT(it->IsInitialized());
                NN_SDK_ASSERT_EQUAL(it->GetOwnerClient(), pClient);
                if(it->GetHandle() == layerHandle)
                {
                    pLayer = &*it;
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(pLayer != nullptr, nn::vi::ResultNotFound());
            NN_RESULT_DO(function(pLayer));
            NN_RESULT_SUCCESS;
        }

        // nn::Result (*function)(indirect::IndirectLayer* pLayer)
        //
        // consumerHandle に該当する pLayer に対して function を呼ぶ。
        // consumerHandle に該当するレイヤが存在しない場合、 nn::vi::ResultNotFound が返される。
        // 見つかった場合、 function(pLayer) の値が返される。
        template<typename F>
        nn::Result ApplyToConsumerBoundLayer(
            ClientObject* pClient,
            nn::vi::IndirectConsumerHandleType consumerHandle,
            F function
            ) NN_NOEXCEPT
        {
            indirect::IndirectLayer* pLayer = nullptr;
            auto pList = ClientObjectAccessor::GetConsumerBoundIndirectLayerList(pClient);
            for(auto it = pList->begin(); it != pList->end(); ++it)
            {
                NN_SDK_ASSERT(it->IsInitialized());
                NN_SDK_ASSERT_EQUAL(it->GetConsumerClient(), pClient);
                if(it->GetConsumerEndPointHandle() == consumerHandle)
                {
                    pLayer = &*it;
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(pLayer != nullptr, nn::vi::ResultNotFound());
            NN_RESULT_DO(function(pLayer));
            NN_RESULT_SUCCESS;
        }

        // nn::Result (*function)(indirect::IndirectLayer* pLayer)
        //
        // producerHandle に該当する pLayer に対して function を呼ぶ。
        // producerHandle に該当するレイヤが存在しない場合、 nn::vi::ResultNotFound が返される。
        // 見つかった場合、 function(pLayer) の値が返される。
        template<typename F>
        nn::Result ApplyToProducerBoundLayer(
            ClientObject* pClient,
            nn::vi::IndirectProducerHandleType producerHandle,
            F function
            ) NN_NOEXCEPT
        {
            indirect::IndirectLayer* pLayer = nullptr;
            auto pList = ClientObjectAccessor::GetProducerBoundIndirectLayerList(pClient);
            for(auto it = pList->begin(); it != pList->end(); ++it)
            {
                NN_SDK_ASSERT(it->IsInitialized());
                NN_SDK_ASSERT_EQUAL(it->GetProducerClient(), pClient);
                if(it->GetProducerEndPointHandle() == producerHandle)
                {
                    pLayer = &*it;
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(pLayer != nullptr, nn::vi::ResultNotFound());
            NN_RESULT_DO(function(pLayer));
            NN_RESULT_SUCCESS;
        }


        // bool (*cond)(indirect::IndirectLayer* pLayer)
        // nn::Result (*function)(indirect::IndirectLayer* pLayer)
        //
        // cond(pLayer) が true となる pLayer に対して function(pLayer) を呼ぶ。
        // cond(pLayer) が true となる pLayer が存在しない場合、 nn::vi::ResultNotFound が返される。
        // 見つかった場合、 function(pLayer) の値が返される。
        template<typename C, typename F>
        nn::Result ApplyToPoolLayer(
            C cond,
            F function
            ) NN_NOEXCEPT
        {
            indirect::IndirectLayer* pLayer = nullptr;
            for(int i = 0; i < indirect::IndirectDisplay::LayerCountMax; i++)
            {
                auto p = indirect::g_IndirectDisplay.GetLayer(i);
                if(!p->IsInitialized())
                {
                    continue;
                }
                if(cond(p))
                {
                    pLayer = p;
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(pLayer != nullptr, nn::vi::ResultNotFound());
            NN_RESULT_DO(function(pLayer));
            NN_RESULT_SUCCESS;
        }
    }


    nn::Result ClientObject::CreateIndirectLayer(nn::vi::IndirectLayerHandleType* pOutLayerHandle) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutLayerHandle);
        NN_VISRV_CHECK_ALIVE();
        auto pDisplay = &indirect::g_IndirectDisplay;

        // 未初期化のレイヤを探す
        indirect::IndirectLayer* pLayer = nullptr;
        for(indirect::IndirectLayerIndex index = 0; index < indirect::IndirectDisplay::LayerCountMax; index++)
        {
            auto p = pDisplay->GetLayer(index);
            if(!p->IsInitialized())
            {
                pLayer = p;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(pLayer != nullptr, nn::vi::ResultResourceLimit());

        // レイヤの作成と登録
        nn::vi::IndirectLayerHandleType layerHandle = 0;
        NN_RESULT_DO(indirect::CreateIndirectLayerImpl(&layerHandle, pLayer, this));

        *pOutLayerHandle = layerHandle;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::DestroyIndirectLayer(nn::vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        NN_RESULT_DO(ApplyToOwnLayer(
            this,
            layerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                // レイヤの登録解除と破棄
                indirect::DestroyIndirectLayerImpl(pLayer);
                NN_RESULT_SUCCESS;
            }
        ));

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::CreateIndirectProducerEndPoint(
        nn::vi::IndirectProducerHandleType* pOutProducerHandle,
        nn::vi::IndirectLayerHandleType layerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutProducerHandle);
        NN_VISRV_CHECK_ALIVE();

        nn::vi::IndirectProducerHandleType handle = 0;
        NN_RESULT_DO(ApplyToOwnLayer(
            this,
            layerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_DO(
                    indirect::CreateIndirectProducerEndPointImpl(&handle, pLayer, aruid)
                );
                NN_RESULT_SUCCESS;
            }
        ));

        *pOutProducerHandle = handle;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::DestroyIndirectProducerEndPoint(
        nn::vi::IndirectLayerHandleType layerHandle
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        NN_RESULT_DO(ApplyToOwnLayer(
            this,
            layerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
               indirect::DestroyIndirectProducerEndPointImpl(pLayer);
                NN_RESULT_SUCCESS;
            }
        ));

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::CreateIndirectConsumerEndPoint(
        nn::vi::IndirectConsumerHandleType* pOutConsumerHandle,
        nn::vi::IndirectLayerHandleType layerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutConsumerHandle);
        NN_VISRV_CHECK_ALIVE();

        nn::vi::IndirectConsumerHandleType handle = 0;
        NN_RESULT_DO(ApplyToOwnLayer(
            this,
            layerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_DO(
                    indirect::CreateIndirectConsumerEndPointImpl(&handle, pLayer, aruid)
                );
                NN_RESULT_SUCCESS;
            }
        ));

        *pOutConsumerHandle = handle;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::DestroyIndirectConsumerEndPoint(
        nn::vi::IndirectLayerHandleType layerHandle
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();

        NN_RESULT_DO(ApplyToOwnLayer(
            this,
            layerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                indirect::DestroyIndirectConsumerEndPointImpl(pLayer);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::BindIndirectProducerEndPoint(
        size_t* pOutNativeWindowDataSize,
        void* pNativeWindowDataBuffer,
        size_t nativeWindowDataBufferSize,
        nn::vi::IndirectProducerHandleType producerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutNativeWindowDataSize);
        NN_VISRV_CHECK_NOT_NULL(pNativeWindowDataBuffer);
        NN_VISRV_CHECK_ALIVE();

        bool isSuccess = false;
        int entryIndex = 0;
        indirect::IndirectLayer* pBoundLayer = nullptr;
        NN_RESULT_DO(ApplyToPoolLayer(
            [&](indirect::IndirectLayer* pLayer) -> bool
            {
                return pLayer->GetProducerEndPointHandle() == producerHandle;
            },
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_DO(indirect::BindIndirectProducerEndPointImpl(&entryIndex, pLayer, this, aruid));
                pBoundLayer = pLayer;
                NN_RESULT_SUCCESS;
            }
        ));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                indirect::UnbindIndirectProducerEndPointImpl(pBoundLayer);
            }
        };

        NN_RESULT_DO(indirect::SerializeIndirectProducer(
            pOutNativeWindowDataSize,
            pNativeWindowDataBuffer,
            nativeWindowDataBufferSize,
            pBoundLayer->GetGraphicBufferProducer()->asBinder(),
            NN_VISRV_INDIRECT_DEFAULT_TRANSACTION_SERVICE_NAME,
            entryIndex
        ));

        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::UnbindIndirectProducerEndPoint(
        nn::vi::IndirectProducerHandleType producerHandle
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_DO(ApplyToProducerBoundLayer(
            this,
            producerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                UnbindIndirectProducerEndPointImpl(pLayer);
                NN_RESULT_SUCCESS;
            }
       ));
       NN_RESULT_SUCCESS;
    }

    //--------------------------------------------------------------------------------------

    nn::Result ClientObject::FlipIndirectProducerBuffer(
        nn::vi::IndirectProducerHandleType producerHandle
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_THROW(nn::vi::ResultNotReady());
        NN_UNUSED(producerHandle);
        //NN_RESULT_DO(ApplyToProducerBoundLayer(
        //    this,
        //    producerHandle,
        //    [&](indirect::IndirectLayer* pLayer) -> nn::Result
        //    {
        //        NN_RESULT_DO(indirect::FlipIndirectProducerImpl(pLayer));
        //        NN_RESULT_SUCCESS;
        //    }
        //));
        //NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetIndirectProducerFlipOffset(
        nn::vi::IndirectLayerHandleType layerHandle,
        nn::vi::IndirectProducerHandleType producerHandle,
        nn::TimeSpan offset
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_DO(ApplyToPoolLayer(
            [&](indirect::IndirectLayer* pLayer) -> bool
            {
                return pLayer->GetHandle() == layerHandle;
            },
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_THROW_UNLESS(pLayer->GetProducerState() != indirect::IndirectProducerEndPointState_Invalid, nn::vi::ResultNotFound());
                NN_RESULT_THROW_UNLESS(pLayer->GetProducerEndPointHandle() == producerHandle, nn::vi::ResultNotFound());
                NN_RESULT_DO(indirect::SetIndirectProducerFlipOffset(pLayer, offset));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;

    }


    //--------------------------------------------------------------------------------------
    nn::Result ClientObject::GetIndirectLayerImageRequiredMemoryInfo(
        size_t* pOutSize,
        size_t* pOutAlignment,
        int width,
        int height
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutSize);
        NN_VISRV_CHECK_NOT_NULL(pOutAlignment);
        NN_VISRV_CHECK_MINMAX(width, 0, std::numeric_limits<int>::max());
        NN_VISRV_CHECK_MINMAX(height, 0, std::numeric_limits<int>::max());
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_LOG_INDIRECT_DO("GetIndirectLayerImageRequiredMemoryInfo", 0);

        vic::VicImageBuffer::InfoType info = {};
        info.width = width;
        info.height = height;
        info.format = vic::VicImageFormat_Rgba8;

        size_t imgSize = vic::VicImageBuffer::GetRequiredMemorySize(info);
        //size_t imgAlignment = vic::VicImageBuffer::GetRequiredMemoryAlignment(&vic::g_VicModule, info);

        size_t poolUnit = vic::VicMemoryPool::GetRequiredUnitSize();
        //size_t poolAlignment = vic::VicMemoryPool::GetRequiredAlignment(&vic::g_VicModule);

        *pOutSize = ((imgSize + poolUnit - 1) / poolUnit) * poolUnit;
        *pOutAlignment = nn::os::MemoryPageSize;
        NN_RESULT_SUCCESS;
    }

    // MapDefer 版
    nn::Result ClientObject::GetIndirectLayerImageMap(
        size_t* pOutSize,
        size_t* pOutStride,
        void* pBuffer,
        size_t bufferSize,
        int width,
        int height,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        nn::Result result = {};

        auto pContext = indirect::g_IndirectMapCopyImageContextTable.GetCurrentContext();
        if(pContext == nullptr)
        {
            NN_VISRV_CHECK_NOT_NULL(pOutSize);
            NN_VISRV_CHECK_NOT_NULL(pOutStride);
            NN_VISRV_CHECK_NOT_NULL(pBuffer);
            NN_VISRV_CHECK_MINMAX(width, 0, std::numeric_limits<int>::max());
            NN_VISRV_CHECK_MINMAX(height, 0, std::numeric_limits<int>::max());
            NN_VISRV_CHECK_MINMAX(sourceRectX     , 0, 1);
            NN_VISRV_CHECK_MINMAX(sourceRectY     , 0, 1);
            NN_VISRV_CHECK_MINMAX(sourceRectWidth , 0, 1);
            NN_VISRV_CHECK_MINMAX(sourceRectHeight, 0, 1);
            NN_VISRV_CHECK_MINMAX(sourceRectX + sourceRectWidth , 0, 1);
            NN_VISRV_CHECK_MINMAX(sourceRectY + sourceRectHeight, 0, 1);
            NN_VISRV_CHECK_ALIVE();
            NN_RESULT_DO(indirect::g_IndirectMapCopyImageContextTable.Acquire(&pContext));
            result = GetIndirectLayerImageMap_I(pContext, pBuffer, bufferSize, width, height, sourceRectX, sourceRectY, sourceRectWidth, sourceRectHeight, consumerHandle, aruid);
        }
        else
        {
            result = GetIndirectLayerImageMap_II(pOutSize, pOutStride, pContext, consumerHandle);
        }

        if(!nn::sf::ResultProcessDeferred::Includes(result))
        {
            if(pContext != nullptr)
            {
                if(pContext->IsInitialized())
                {
                    pContext->Finalize();
                }
                indirect::g_IndirectMapCopyImageContextTable.Release(pContext);
            }
        }

        return result;
    }

    nn::Result ClientObject::GetIndirectLayerImageMap_I(
        indirect::IndirectMapCopyImageContext* pContext,
        void* pBuffer,
        size_t bufferSize,
        int width,
        int height,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(isSuccess, "Getting IndirectLayerImage/KickOperation(#%llu,%08llX)", consumerHandle, aruid);
        // レイヤを検索
        // MapDefer 用にバインド
        indirect::IndirectLayer* pBoundLayer = nullptr;
        NN_RESULT_DO(ApplyToPoolLayer(
            [&](indirect::IndirectLayer* pLayer) -> bool
            {
                return pLayer->GetConsumerEndPointHandle() == consumerHandle;
            },
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_THROW_UNLESS(aruid != nn::applet::AppletResourceUserId::GetInvalidId(), nn::vi::ResultDenied());
                NN_RESULT_THROW_UNLESS(aruid == pLayer->GetConsumerAruid(), nn::vi::ResultDenied());

                NN_RESULT_THROW_UNLESS(pLayer->GetProducerState() == indirect::IndirectProducerEndPointState_Connected, nn::vi::ResultNotReady());
                NN_RESULT_THROW_UNLESS(pLayer->GetConsumerState() == indirect::IndirectConsumerEndPointState_Unbound, nn::vi::ResultAlreadyOpened());

                NN_RESULT_DO(indirect::BindIndirectConsumerEndPointMapDeferImpl(pLayer, this, aruid));
                pBoundLayer = pLayer;
                NN_RESULT_SUCCESS;
            }
        ));
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                indirect::UnbindIndirectConsumerEndPointImpl(pBoundLayer);
            }
        };

        auto pSourceStage       = pBoundLayer->GetBufferStage();
        auto pSourceLockCounter = pBoundLayer->GetBufferLockCounter();
        NN_RESULT_THROW_UNLESS(pSourceStage->HasReadyBuffer(), nn::vi::ResultNotReady());
        auto pSourceBuffer      = pSourceStage->GetReadyBuffer()->GetGraphicBuffer();

        auto pDestinationLockCounter = pBoundLayer->GetConsumerBoundLockCounter();

        pSourceLockCounter->AcquireLock();
        pDestinationLockCounter->AcquireLock();

        indirect::IndirectMapCopyImageContextParameter param = {};
        param.pBuffer    = pBuffer;
        param.bufferSize = bufferSize;
        param.width      = width;
        param.height     = height;
        param.sourceRectX      = sourceRectX;
        param.sourceRectY      = sourceRectY;
        param.sourceRectWidth  = sourceRectWidth;
        param.sourceRectHeight = sourceRectHeight;
        param.pSourceBufferStage       = pSourceStage;
        param.pSourceBufferLockCounter = pSourceLockCounter;
        param.pSourceGraphicBuffer     = pSourceBuffer;
        param.pDestinationBoundLockCounter = pDestinationLockCounter;
        param.destinationBoundId           = pBoundLayer->GetConsumerBoundId();

        NN_ABORT_UNLESS_RESULT_SUCCESS(pContext->Initialize(param));

        vic::g_VicTaskWorker.SubmitTask(pContext);
        indirect::g_IndirectMapCopyImageContextTable.DeferExecution(pContext);
        isSuccess = true;
        return nn::sf::DeferProcess();
    }

    nn::Result ClientObject::GetIndirectLayerImageMap_II(
        size_t* pOutSize,
        size_t* pOutStride,
        indirect::IndirectMapCopyImageContext* pContext,
        nn::vi::IndirectConsumerHandleType consumerHandle
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(isSuccess, "Getting IndirectLayerImage/FetchResult(#%llu,%08llX)", consumerHandle);
        NN_VISRV_CHECK_ALIVE();

        // レイヤを unbind
        NN_RESULT_TRY(ApplyToPoolLayer(
            [&](indirect::IndirectLayer* pLayer) -> bool
            {
                return pLayer->GetConsumerEndPointHandle() == consumerHandle;
            },
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_THROW_UNLESS(pLayer->GetConsumerBoundId() == pContext->GetParameter().destinationBoundId, nn::vi::ResultDenied());
                indirect::UnbindIndirectConsumerEndPointImpl(pLayer);
                NN_RESULT_SUCCESS;
            }
        ))
            NN_RESULT_CATCH_ALL
            {
                NN_VISRV_LOG_INDIRECT_WARN("Consumer already unbound\n");
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(pContext->GetOutResult());
        *pOutSize   = pContext->GetOutImageSize();
        *pOutStride = pContext->GetOutImageStride();
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }


    // TransferMemory 版
    nn::Result ClientObject::BindIndirectConsumerEndPointTransfer(
        nn::os::NativeHandle* pOutIsImageReadyEventHandle,
        bool* pOutIsImageReadyEventHandleManaged,
        nn::os::NativeHandle bufferHandle,
        bool* pIsBufferHandleManaged,
        size_t bufferSize,
        int width,
        int height,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutIsImageReadyEventHandle);
        NN_VISRV_CHECK_NOT_NULL(pOutIsImageReadyEventHandleManaged);
        NN_VISRV_CHECK_MINMAX(width, 0, std::numeric_limits<int>::max());
        NN_VISRV_CHECK_MINMAX(height, 0, std::numeric_limits<int>::max());
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_THROW_UNLESS(IsIndirectLayerSupportTransferMemory, nn::vi::ResultNotSupported());

        NN_RESULT_DO(ApplyToPoolLayer(
            [&](indirect::IndirectLayer* pLayer) -> bool
            {
                return pLayer->GetConsumerEndPointHandle() == consumerHandle;
            },
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_DO(indirect::BindIndirectConsumerEndPointTransferImpl(
                    pOutIsImageReadyEventHandle,
                    pOutIsImageReadyEventHandleManaged,
                    pLayer,
                    this,
                    aruid,
                    bufferHandle,
                    pIsBufferHandleManaged,
                    bufferSize,
                    width,
                    height
                ));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::KickIndirectCopyImageTransfer(
        size_t* pOutSize,
        size_t* pOutStride,
        nn::vi::IndirectConsumerHandleType consumerHandle,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_MINMAX(sourceRectX     , 0, 1);
        NN_VISRV_CHECK_MINMAX(sourceRectY     , 0, 1);
        NN_VISRV_CHECK_MINMAX(sourceRectWidth , 0, 1);
        NN_VISRV_CHECK_MINMAX(sourceRectHeight, 0, 1);
        NN_VISRV_CHECK_MINMAX(sourceRectX + sourceRectWidth , 0, 1);
        NN_VISRV_CHECK_MINMAX(sourceRectY + sourceRectHeight, 0, 1);
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_THROW_UNLESS(IsIndirectLayerSupportTransferMemory, nn::vi::ResultNotSupported());

        NN_RESULT_DO(ApplyToConsumerBoundLayer(
            this,
            consumerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                NN_RESULT_DO(indirect::KickIndirectCopyImageTransferImpl(pOutSize, pOutStride, pLayer, sourceRectX, sourceRectY, sourceRectWidth, sourceRectHeight));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    // 共通 Unbind
    nn::Result ClientObject::UnbindIndirectConsumerEndPoint(nn::vi::IndirectConsumerHandleType consumerHandle) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_DO(ApplyToConsumerBoundLayer(
            this,
            consumerHandle,
            [&](indirect::IndirectLayer* pLayer) -> nn::Result
            {
                indirect::UnbindIndirectConsumerEndPointImpl(pLayer);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    //--------------------------------------------------------------------------------------
    // transaction
    //--------------------------------------------------------------------------------------


    namespace {
        android::sp<android::IBinder> GetProducerBinder(int32_t binderHandle, ClientObject* pClient) NN_NOEXCEPT
        {
            if(binderHandle < 0 || binderHandle >= indirect::IndirectProducerBinderTable::Size)
            {
                return nullptr;
            }

            auto pTable = ClientObjectAccessor::GetIndirectProducerBinderTable(pClient);
            auto& entry = pTable->Get(binderHandle);
            if(entry.pLayer == nullptr)
            {
                return nullptr;
            }

            auto pProducer = entry.pLayer->GetGraphicBufferProducer();
            if(pProducer == nullptr)
            {
                return nullptr;
            }

            return pProducer->asBinder();
        }

        indirect::IndirectLayer* GetIndirectLayer(int32_t binderHandle, ClientObject* pClient) NN_NOEXCEPT
        {
            if(binderHandle < 0 || binderHandle >= indirect::IndirectProducerBinderTable::Size)
            {
                return nullptr;
            }

            auto pTable = ClientObjectAccessor::GetIndirectProducerBinderTable(pClient);
            auto& entry = pTable->Get(binderHandle);
            return entry.pLayer;
        }

        void FreeParcelBuffer(
            android::Parcel* parcel,
            const uint8_t* data,
            size_t dataSize,
            const binder_size_t* objects,
            size_t objectsSize,
            void* cookie
            ) NN_NOEXCEPT
        {
            NN_UNUSED(parcel);
            NN_UNUSED(data);
            NN_UNUSED(dataSize);
            NN_UNUSED(objects);
            NN_UNUSED(objectsSize);
            NN_UNUSED(cookie);
            // do nothing
        }

        bool ReadParcel(android::Parcel &parcel, uint8_t *pSrcBuffer, size_t srcSize) NN_NOEXCEPT
        {
            using namespace android;

            uint8_t* buffer = reinterpret_cast<uint8_t*>(pSrcBuffer);
            size_t bufferSize = srcSize;
            if(bufferSize < sizeof(hosbinder_transaction_t))
            {
                return false;
            }

            hosbinder_transaction_t *tr = (hosbinder_transaction_t *) buffer;
            size_t requiredSize = sizeof(*tr) + tr->data_size + tr->obj_size;
            if (bufferSize < requiredSize)
            {
                return false;
            }

            size_t num_obj = tr->obj_size / sizeof(binder_size_t);
            if(num_obj > 0)
            {
                NN_VISRV_LOG_INDIRECT_ERR("transaction request contains object\n");
                return false;
            }
            //binder_size_t *objs = (binder_size_t *) (buffer + tr->obj_offset);
            //for (size_t i=0; i<num_obj; ++i) {
            //    flat_binder_object *flat;
            //    flat = (flat_binder_object *) (buffer + tr->data_offset + objs[i]);
            //    if (flat->type == BINDER_TYPE_HANDLE) {
            //        sp<IBinder> binder = FindObject(flat->handle);
            //        flat->type = BINDER_TYPE_BINDER;
            //        flat->handle = 0;
            //        if (binder != NULL) {
            //            flat->binder = reinterpret_cast<uintptr_t>(binder->getWeakRefs());
            //            flat->cookie = reinterpret_cast<uintptr_t>(binder.get());
            //        } else {
            //            flat->binder = 0;
            //            flat->cookie = 0;
            //        }
            //    } else {
            //        ALOGE("Unexpected Parcel object references in request");
            //        assert(0);
            //    }
            //}

            parcel.ipcSetDataReference(
                buffer + tr->data_offset,
                tr->data_size,
                (binder_size_t *) (buffer + tr->obj_offset),
                tr->obj_size / sizeof(binder_size_t),
                FreeParcelBuffer,
                nullptr
            );

            return true;
        }

        size_t WriteParcel(
            const android::Parcel& parcel,
            uint8_t *dstBuffer,
            size_t dstSize
            ) NN_NOEXCEPT
        {
            using namespace android;

            if(parcel.errorCheck() != NO_ERROR)
            {
                return 0;
            }

            size_t dataSize = parcel.ipcDataSize();
            size_t objSize  = parcel.ipcObjectsCount() * sizeof(binder_size_t);
            size_t requiredSize = sizeof(hosbinder_transaction_t) + dataSize + objSize;

            if (dstSize < requiredSize)
            {
                return 0;
            }

            auto tr = reinterpret_cast<hosbinder_transaction_t*>(dstBuffer);

            memset(tr, 0, sizeof(*tr));
            tr->data_size   = dataSize;
            tr->data_offset = sizeof(*tr);
            tr->obj_size    = objSize;
            tr->obj_offset  = tr->data_offset + tr->data_size;
            memcpy(dstBuffer + tr->data_offset, reinterpret_cast<void*>(parcel.ipcData())   , tr->data_size);
            memcpy(dstBuffer + tr->obj_offset,  reinterpret_cast<void*>(parcel.ipcObjects()), tr->obj_size);

            // Before sending the reply, need to convert the object references into
            // client handles
            size_t num_obj = tr->obj_size / sizeof(binder_size_t);
            if(num_obj > 0)
            {
                NN_VISRV_LOG_INDIRECT_ERR("transaction response contains object\n");
                return false;
            }
            //binder_size_t *objs = (binder_size_t *) (dst + tr->obj_offset);
            //for (size_t i=0; i<num_obj; ++i) {
            //    flat_binder_object *flat;
            //    flat = (flat_binder_object *) (dst + tr->data_offset + objs[i]);
            //    if (flat->type == BINDER_TYPE_BINDER) {
            //        sp<IBinder> binder = (IBinder *) (uintptr_t) flat->cookie;
            //        flat->type = BINDER_TYPE_HANDLE;
            //        flat->binder = 0;
            //        flat->handle = GetObjectHandle(binder);
            //        flat->cookie = 0;
            //        memset(flat->service_name, 0, sizeof(flat->service_name));
            //        strncpy(flat->service_name,
            //                m_manager.GetServiceName(),
            //                sizeof(flat->service_name) - 1);
            //    } else if (flat->type == BINDER_TYPE_WEAK_BINDER) {
            //        ALOGE("Unexpected Parcel weak object references in reply");
            //        assert(0);
            //    }
            //}

            return requiredSize;
        }

        bool CheckInterface(android::Parcel& data, const android::String16& interfaceDesc) NN_NOEXCEPT
        {
            return data.enforceInterface(interfaceDesc);
        }

        char g_TransactionBuffer[4096];
    }

    void ClientObject::IndirectDisplayTransactParcel(
        std::int32_t handle,
        std::uint32_t code,
        const void* requestBuffer,
        size_t requestBufferSize,
        void* replyBuffer,
        size_t replyBufferSize,
        std::uint32_t flags
        ) NN_NOEXCEPT
    {
        {
            if(!IsAlive())
            {
                goto FAIL;
            }

            if(requestBufferSize > sizeof(g_TransactionBuffer))
            {
                goto FAIL;
            }

            memcpy(g_TransactionBuffer, requestBuffer, requestBufferSize);
            android::Parcel request;
            if(!ReadParcel(request, reinterpret_cast<uint8_t*>(g_TransactionBuffer), requestBufferSize))
            {
                goto FAIL;
            }

            android::Parcel response;
            android::Parcel* pResponse = nullptr;
            if(replyBuffer != nullptr && replyBufferSize > 0)
            {
                pResponse = &response;
            }

            android::status_t err = android::NO_ERROR;
            switch(code)
            {
            case native::IGraphicBufferProducerFunctionCode_Connect:
                {
                    NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(isSuccess, "Connecting IndirectProducerEndPoint<%d>", static_cast<int>(handle));
                    auto p = GetIndirectLayer(handle, this);
                    if(p == nullptr)
                    {
                        goto FAIL;
                    }
                    if(!CheckInterface(request, native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor()))
                    {
                        NN_VISRV_LOG_INDIRECT_ERR("checking interface failed\n");
                        goto FAIL;
                    }
                    int listener = request.readInt32();
                    if(listener != 0)
                    {
                        NN_VISRV_LOG_INDIRECT_ERR("connect request contains listener(%d)\n", listener);
                        goto FAIL;
                    }
                    int api = request.readInt32();
                    bool controlledByApp = static_cast<bool>(request.readInt32());
                    auto pOutput = reinterpret_cast<android::IGraphicBufferProducer::QueueBufferOutput*>(response.writeInplace(sizeof(android::IGraphicBufferProducer::QueueBufferOutput)));
                    memset(pOutput, 0, sizeof(android::IGraphicBufferProducer::QueueBufferOutput));
                    if(p->ConnectProducer(pOutput, api, controlledByApp).IsFailure())
                    {
                        goto FAIL;
                    }
                    response.writeInt32(android::NO_ERROR);
                    err = android::NO_ERROR;
                    isSuccess = true;
                    break;
                }
            case native::IGraphicBufferProducerFunctionCode_Disconnect:
                {
                    NN_VISRV_LOG_INDIRECT_SUCCESS_FAILED(isSuccess, "Disconnecting IndirectProducerEndPoint<%d>", handle);
                    auto p = GetIndirectLayer(handle, this);
                    if(p == nullptr)
                    {
                        goto FAIL;
                    }
                    if(!CheckInterface(request, native::InterfaceDescriptor::GetIGraphicBufferProducerDescriptor()))
                    {
                        NN_VISRV_LOG_INDIRECT_ERR("checking interface failed\n");
                        goto FAIL;
                    }
                    int api = request.readInt32();
                    NN_UNUSED(api);
                    p->DisconnectProducer();
                    response.writeInt32(android::NO_ERROR);
                    err = android::NO_ERROR;
                    isSuccess = true;
                    break;
                }
            default:
                {
                    auto p = GetProducerBinder(handle, this);
                    if(p == nullptr)
                    {
                        goto FAIL;
                    }
                    err = p->transact(code, request, pResponse, flags);
                    break;
                }
            }

            if(err == android::NO_ERROR && pResponse != nullptr)
            {
                size_t replySize = WriteParcel(*pResponse, reinterpret_cast<uint8_t*>(replyBuffer), replyBufferSize);
                NN_UNUSED(replySize);
            }
        }
        return;

    FAIL:
        NN_VISRV_LOG_INDIRECT_ERR("IndirectTransactParcel failed\n");
        return;
    }

    void ClientObject::IndirectDisplayAdjustRefcount(std::int32_t handle, std::int32_t diff, std::int32_t isStrong) NN_NOEXCEPT
    {
        NN_UNUSED(handle);
        NN_UNUSED(diff);
        NN_UNUSED(isStrong);
        // nothing to do
    }

    void ClientObject::IndirectDisplayGetNativeHandle(std::int32_t handle, std::uint32_t code, nn::sf::Out<nn::sf::NativeHandle>& result) NN_NOEXCEPT
    {
        {
            if(!IsAlive())
            {
                goto FAIL;
            }

            auto pBinder = GetProducerBinder(handle, this);
            if(pBinder == nullptr)
            {
                goto FAIL;
            }

            nn::os::NativeHandle h;
            bool isManaged;
            pBinder->getNativeHandle(code, h, isManaged);
            result.Set(nn::sf::NativeHandle(h, isManaged));
        }
        return;

    FAIL:
        (*result).Reset();
        return;
    }

}}}
