﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_HotplugState.h>
#include <nn/vi/vi_PowerState.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/vi_LayerSettings.h>
#include "detail/vi_PlatformDisplayInfo.h"
#include "vi_Layer.h"
#include "detail/vi_PoolAllocator.h"

namespace nn { namespace vi {

    namespace detail
    {
        class IModeFilter;
    }

    /**
     * @brief Platform-agnostic implementation for display logic.
     */
    class Display
    {
        NN_DISALLOW_COPY(Display);
    public:
        /**
         * @brief Type for intrusive list of layers.
         */
        typedef nn::util::IntrusiveList<Layer, nn::util::IntrusiveListMemberNodeTraits<Layer, &Layer::m_ListNode>> LayerList;

        /**
         * @brief Allocator type used for layers.
         */
        typedef detail::PoolAllocator<Layer> Allocator;

        /**
         * @brief Creates a new Display object for clients.
         *
         * @param[in] info       Information about the display.
         * @param[in] pAllocator Allocator to use for creating layers.
         *
         * @pre  @a pAllocator != nullptr
         * @post An unopened display object is created.
         *
         * @details This object is reference counted.  Note that the design must enforce that
         *          only one display with a particular name exists.  That constraint is not
         *          enforced here.
         *
         *          Note a copy is made of @a info.  On PC, it will be possible to enumerate
         *          more than just the named displays.  The information will not exist in the
         *          display info segment, so a copy must be made to support both cases.
         */
        Display(const detail::PlatformDisplayInfo& info, Allocator* pAllocator, const detail::IModeFilter* pFilter) NN_NOEXCEPT;

        /**
         * @brief Releases all resources used by this display.
         */
        virtual ~Display() NN_NOEXCEPT;

        /**
         * @brief Queries whether the display is open based on its reference count.
         *
         * @return       Whether this object is still open.
         * @retval true  Display is open.
         * @retval false No references to this display exist within the process.
         */
        bool IsOpen() const NN_NOEXCEPT;

        /**
         * @brief Opens the display.
         *
         * @return                       Whether the display was opened successfully.
         * @retval ResultSuccess         Display was opened successfully.
         * @retval ResultOperationFailed An error occurred while opening the display.
         *
         * @pre  None.
         * @post The display is opened.
         */
        nn::Result Open() NN_NOEXCEPT;

        /**
         * @brief Closes the display.
         *
         * @pre  @a m_OpenCount > 0
         * @post Display may be closed.
         *
         * @details The display may remain open if the reference count indicates that
         *          the process is still referencing the display.  When finally closed,
         *          all layers will be destroyed as well.
         */
        void Close() NN_NOEXCEPT;

        /**
         * @brief Retrieve information about the display.
         *
         * @return The display's information.
         */
        const detail::PlatformDisplayInfo& GetInfo() const NN_NOEXCEPT;

        /**
         * @brief Creates a new layer on a particular display.
         *
         * @param[out] pOutLayer Layer handle.
         * @param[in]  width     The initial width.
         * @param[in]  height    The initial height.
         * @param[in]  format    The pixel format.
         * @param[in]  settings  Initial settings for the layer.
         *
         * @return                          Whether the layer was created.
         * @retval ResultSuccess            Layer created successfully.
         * @retval ResultDenied             The process has exceeded the maximum number of layers allowed
         *                                  on @a pDisplay.
         * @retval ResultOperationFailed    An unknown error occurred.
         *
         * @pre  @a pOutLayer != nullptr @n
         *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
         * @post A valid handle is written to @a pOutLayer.
         */
        nn::Result CreateLayer(Layer** pOutLayer, int width, int height, PixelFormat format, LayerSettings settings) NN_NOEXCEPT;

        /**
         * @brief Destroys the layer and disables content.
         *
         * @param[in] pLayer Layer handle.
         *
         * @pre  @a pLayer != nullptr and @a pLayer is a valid handle.
         * @post @a pLayer invalidated.
         */
        void DestroyLayer(Layer* pLayer) NN_NOEXCEPT;

        /**
         * @brief Setting to enable or disable all layers for the process for a particular display.
         *
         * @param[in] isEnabled When @a isEnabled is true, all layers are enabled.  Otherwise, no content
         *                      from the process is displayed.
         *
         * @return                       Whether the display is showing layers from the process.
         * @retval ResultSuccess         The new setting has taken effect.
         * @retval ResultOperationFailed An unknown error occurred.
         *
         * @pre  @a pDisplay != nullptr and @a pDisplay is a valid handle.
         * @post All layers from the process on @a pDisplay match that of @a isEnabled.
         */
        nn::Result SetEnabled(bool isEnabled) NN_NOEXCEPT;

        int ListModes(DisplayModeInfo* pOutModes, int modeCountMax) const NN_NOEXCEPT;
        nn::Result SetMode(const DisplayModeInfo* pMode) NN_NOEXCEPT;

        virtual nn::Result GetHotplugState(HotplugState* pOutState) const NN_NOEXCEPT = 0;
        virtual nn::Result GetMode(DisplayModeInfo* pOutMode) const NN_NOEXCEPT = 0;

        virtual int ListRgbRanges(RgbRange* pOutRanges, int rgbRangeCountMax) const NN_NOEXCEPT = 0;
        virtual nn::Result GetRgbRange(RgbRange* pOutRange) const NN_NOEXCEPT = 0;
        virtual nn::Result SetRgbRange(RgbRange range) NN_NOEXCEPT = 0;

        virtual nn::Result SetUnderscan(int underscan) NN_NOEXCEPT = 0;
        virtual nn::Result GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT = 0;

        virtual nn::Result SetAlpha(float alpha) NN_NOEXCEPT = 0;

        virtual nn::Result SetPowerState(PowerState state) NN_NOEXCEPT = 0;

        virtual nn::Result SetLayerStack(LayerStack id) NN_NOEXCEPT = 0;

        virtual nn::Result SetCmuLuma(float luma) NN_NOEXCEPT = 0;
        virtual nn::Result GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT = 0;

        virtual nn::Result GetHotplugEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT = 0;
        virtual nn::Result GetVsyncEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT = 0;

        nn::util::IntrusiveListNode m_ListNode; //!< Necessary for storing in an intrusive list.

        nn::os::Mutex& GetLayerListLock() const NN_NOEXCEPT;

        const LayerList& GetLayerList() const NN_NOEXCEPT;
    private:
        detail::PlatformDisplayInfo m_Info; //!< Various information about the display.

        LayerList m_Layers; //!< Intrusive list of layers.

        Allocator* m_pAllocator; //!< Layer allocator.
        const detail::IModeFilter* m_pFilter;
        mutable nn::os::Mutex m_Lock; //!< Lock for protecting layer access.

        int m_OpenCount; //!< Reference count for this display.

        virtual nn::Result Initialize() = 0;
        virtual int ListModes(DisplayModeInfo* pOutModes, int modeCountMax, const detail::IModeFilter* pFilter) const NN_NOEXCEPT = 0;
        virtual nn::Result SetMode(const DisplayModeInfo* pMode, const detail::IModeFilter* pFilter) NN_NOEXCEPT = 0;

        void ForceClose() NN_NOEXCEPT; //!< Releases all resources for this display.
    };

}}
