﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma comment(lib, "dxgi.lib")
#pragma comment(lib, "Netapi32.lib")

#include <nn/util/util_ScopeExit.h>
#include "vi_Windows.h"
#include "vi_VsyncManager.h"

namespace
{
    // 60 Hz
    const std::int64_t FallbackTiming = -166667LL;
}

nn::vi::detail::VsyncManager::DxgiPtr<IDXGIOutput> nn::vi::detail::VsyncManager::GetPrimaryOutput(DxgiPtr<IDXGIFactory>& pFactory) NN_NOEXCEPT
{
    // Only getting vsync from primary monitor
    HMONITOR primary = MonitorFromPoint({0, 0}, MONITOR_DEFAULTTONEAREST);

    IDXGIAdapter* pAdapter = nullptr;
    IDXGIOutput* pOutput = nullptr;
    for( int i = 0; pFactory->EnumAdapters(i, &pAdapter) == S_OK; ++i )
    {
        NN_UTIL_SCOPE_EXIT
        {
            pAdapter->Release();
        };

        for( int j = 0; pAdapter->EnumOutputs(j, &pOutput) == S_OK; ++j )
        {
            DXGI_OUTPUT_DESC info;
            if( pOutput->GetDesc(&info) == S_OK && primary == info.Monitor )
            {
                return DxgiPtr<IDXGIOutput>(pOutput);
            }

            pOutput->Release();
        }
    }

    return nullptr;
}

void nn::vi::detail::VsyncManager::FakeVsync(nn::os::SystemEvent* pVsync, const std::atomic<bool>& isRunning, DxgiPtr<IDXGIOutput>& pOutput) NN_NOEXCEPT
{
    LARGE_INTEGER waitTime;
    waitTime.QuadPart = FallbackTiming;

    if( pOutput != nullptr )
    {
        pOutput->WaitForVBlank();

        // There's no reliable way to query the current refresh rate.
        // Old Win32 API calls return an integer for Hz, so this can fail in 59.97 cases...
        // DXGI only exposes ways to find the closest mode, but there's no guarantee we can
        // accurately get the current mode given only width/height.

        // Train the timer over a period of a few vsyncs.
        const int VsyncTrainingCount = 4;
        waitTime.QuadPart = 0;
        for( int i = 0; i < VsyncTrainingCount && isRunning.load(std::memory_order_acquire); ++i )
        {
            nn::os::Tick start = nn::os::GetSystemTick();
            pOutput->WaitForVBlank();
            nn::os::Tick end = nn::os::GetSystemTick();

            waitTime.QuadPart += (end - start).ToTimeSpan().GetMicroSeconds();
        }

        waitTime.QuadPart /= VsyncTrainingCount;
        // Negative indicates relative time in the future.
        // Timer is specified in number of 100 ns intervals.
        waitTime.QuadPart *= -10;
    }

    HANDLE vsyncTimer = CreateWaitableTimer(nullptr, false, nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        CloseHandle(vsyncTimer);
    };

    while( isRunning.load(std::memory_order_acquire) )
    {
        pVsync->Signal();
        SetWaitableTimer(vsyncTimer, &waitTime, 0, nullptr, nullptr, true);
        WaitForSingleObject(vsyncTimer, INFINITE);
    }
}

void nn::vi::detail::VsyncManager::RealVsync(nn::os::SystemEvent* pVsync, const std::atomic<bool>& isRunning, DxgiPtr<IDXGIOutput>& pOutput) NN_NOEXCEPT
{
    while( isRunning.load(std::memory_order_acquire) )
    {
        pOutput->WaitForVBlank();
        pVsync->Signal();
    }
}

nn::vi::detail::VsyncManager::VsyncThreadArgs::VsyncThreadArgs() NN_NOEXCEPT
    : vsyncEvent(nn::os::EventClearMode_AutoClear, true)
    , isRunning(true)
{
}

nn::vi::detail::VsyncManager::VsyncManager() NN_NOEXCEPT
    : m_VsyncThread(nullptr)
    , m_IsAvailable(true)
{
    IDXGIFactory* pFactory;
    NN_ABORT_UNLESS(SUCCEEDED(CreateDXGIFactory(__uuidof(IDXGIFactory), reinterpret_cast<void**>(&pFactory))));
    m_Factory.reset(pFactory);

    m_Args.pOutput = GetPrimaryOutput(m_Factory);

    m_VsyncThread = CreateThread(nullptr, 4096, VsyncThread, &m_Args, 0, nullptr);
    NN_ABORT_UNLESS_NOT_NULL(m_VsyncThread);
}

nn::vi::detail::VsyncManager::~VsyncManager() NN_NOEXCEPT
{
    m_Args.isRunning.store(false, std::memory_order_release);
    WaitForSingleObject(m_VsyncThread, INFINITE);
}

nn::os::NativeHandle nn::vi::detail::VsyncManager::GetClientReadableHandle() NN_NOEXCEPT
{
    if( m_IsAvailable )
    {
        m_IsAvailable = false;
        return m_Args.vsyncEvent.DetachReadableHandle();
    }

    return nullptr;
}

DWORD WINAPI nn::vi::detail::VsyncManager::VsyncThread(void* ptr) NN_NOEXCEPT
{
    // SIGLO-81356: Only use real vsync from DXGI on Windows 10
    BYTE* pBuffer = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        if( pBuffer != nullptr )
        {
            NetApiBufferFree(pBuffer);
        }
    };

    auto pArgs = static_cast<VsyncThreadArgs*>(ptr);

    if( pArgs->pOutput == nullptr || NetWkstaGetInfo(nullptr, 100, reinterpret_cast<BYTE**>(&pBuffer)) != NERR_Success || reinterpret_cast<WKSTA_INFO_100*>(pBuffer)->wki100_ver_major < 10 )
    {
        FakeVsync(&pArgs->vsyncEvent, pArgs->isRunning, pArgs->pOutput);
    }
    else
    {
        RealVsync(&pArgs->vsyncEvent, pArgs->isRunning, pArgs->pOutput);
    }

    return 0;
}
