﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_IndirectConsumer.h>

#include <nn/nn_SdkAssert.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/am/am_Shim.h>
#include <nn/vi/vi_Result.h>

#include "vi_CommonUtility.h"

namespace nn{ namespace vi{

    nn::Result GetIndirectImageMemoryRequirement(
        size_t* pOutRequiredSize,
        size_t* pOutRequiredAlignment,
        int width,
        int height
        ) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pOutRequiredSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutRequiredAlignment);
        NN_SDK_REQUIRES_GREATER(width, 0);
        NN_SDK_REQUIRES_GREATER(height, 0);

        int64_t outSize = 0;
        int64_t outAlginemnt = 0;
        NN_RESULT_DO(GetService()->GetIndirectLayerImageRequiredMemoryInfo(
            &outSize,
            &outAlginemnt,
            static_cast<int64_t>(width),
            static_cast<int64_t>(height)
        ));

        *pOutRequiredSize = static_cast<size_t>(outSize);
        *pOutRequiredAlignment = static_cast<size_t>(outAlginemnt);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetIndirectImage(
        size_t* pOutSize,
        size_t* pOutStride,
        void* pBuffer,
        size_t bufferSize,
        IndirectConsumerHandleType handle,
        int width,
        int height
        ) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutStride);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
        NN_SDK_REQUIRES_GREATER(handle, 0u);
        NN_SDK_REQUIRES_GREATER(width, 0);
        NN_SDK_REQUIRES_GREATER(height, 0);

        //// use blocking session
        //NN_RESULT_TRY(CreateBlockingService())
        //    NN_RESULT_CATCH(ResultAlreadyOpened) { /* ok */ }
        //NN_RESULT_END_TRY;
        //auto pService = GetBlockingService();
        //NN_ABORT_UNLESS_NOT_NULL(pService);

        auto pService = GetService();

        int64_t outSize = 0;
        int64_t outStride = 0;
        NN_RESULT_DO(pService->GetIndirectLayerImageMap(
            &outSize,
            &outStride,
            nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize),
            static_cast<int64_t>(width),
            static_cast<int64_t>(height),
            handle,
            nn::applet::GetAppletResourceUserId()
        ));

        *pOutSize = static_cast<size_t>(outSize);
        *pOutStride = static_cast<size_t>(outStride);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetIndirectImage(
        size_t* pOutSize,
        size_t* pOutStride,
        void* pBuffer,
        size_t bufferSize,
        IndirectConsumerHandleType handle,
        int width,
        int height,
        float sourceRectX,
        float sourceRectY,
        float sourceRectWidth,
        float sourceRectHeight
        ) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutStride);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
        NN_SDK_REQUIRES_GREATER(handle, 0u);
        NN_SDK_REQUIRES_GREATER(width, 0);
        NN_SDK_REQUIRES_GREATER(height, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(sourceRectX, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(sourceRectY, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(sourceRectWidth, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(sourceRectHeight, 0);
        NN_SDK_REQUIRES_LESS_EQUAL(sourceRectX + sourceRectWidth, 1);
        NN_SDK_REQUIRES_LESS_EQUAL(sourceRectY + sourceRectHeight, 1);

        //// use blocking session
        //NN_RESULT_TRY(CreateBlockingService())
        //    NN_RESULT_CATCH(ResultAlreadyOpened) { /* ok */ }
        //NN_RESULT_END_TRY;
        //auto pService = GetBlockingService();
        //NN_ABORT_UNLESS_NOT_NULL(pService);

        auto pService = GetService();

        int64_t outSize = 0;
        int64_t outStride = 0;
        NN_RESULT_DO(pService->GetIndirectLayerImageCropMap(
            &outSize,
            &outStride,
            nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize),
            static_cast<int64_t>(width),
            static_cast<int64_t>(height),
            sourceRectX,
            sourceRectY,
            sourceRectWidth,
            sourceRectHeight,
            handle,
            nn::applet::GetAppletResourceUserId()
        ));

        *pOutSize = static_cast<size_t>(outSize);
        *pOutStride = static_cast<size_t>(outStride);
        NN_RESULT_SUCCESS;
    }

}}
