﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedTextureMemoryPool.h>

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/vi/vi_Result.h>
#include <nn/vi/fbshare/vi_SharedTexturePool.h>
#include "../vi_Log.h"
#include "../vi_Macro.h"
#include "../vi_Config.h"

#include <nvn/nvn_FuncPtrInline.h>
#include <nvrm_memmgr.h>

namespace nn{ namespace vi{ namespace fbshare{
    typedef void (*PFNNVNMEMORYPOOLBUILDERSETNATIVEHANDLENVXPROC)(NVNmemoryPoolBuilder*, uint64_t);

    namespace {

        bool ExtractCompressibleOption(SharedTextureMemoryPoolOption option) NN_NOEXCEPT
        {
            return option & SharedTextureMemoryPoolOption_Compressible;
        }

        uint64_t CastRmMemHandleToUint64(NvRmMemHandle h) NN_NOEXCEPT
        {
            uint64_t v = 0;
            std::memcpy(&v, &h, std::min(sizeof(NvRmMemHandle), sizeof(uint64_t)));
            return v;
        }

        NvRmMemHandle CastUint64ToRmMemHandle(uint64_t v) NN_NOEXCEPT
        {
            NvRmMemHandle h = 0;
            std::memcpy(&h, &v, std::min(sizeof(NvRmMemHandle), sizeof(uint64_t)));
            return h;
        }

    }

    SharedTextureMemoryPool::SharedTextureMemoryPool() NN_NOEXCEPT
        : m_State(SharedTextureMemoryPoolState_Invalid)
        , m_MemoryHandle()
        , m_MemorySize()
        , m_ImportedMemoryPool()
    {
    }

    bool SharedTextureMemoryPool::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != SharedTextureMemoryPoolState_Invalid;
    }

    bool SharedTextureMemoryPool::IsCompressible() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return ExtractCompressibleOption(m_Option);
    }
    size_t SharedTextureMemoryPool::GetSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_MemorySize;
    }

    nn::Result SharedTextureMemoryPool::Initialize(
        NVNdevice* pDevice,
        nn::vi::native::NativeMemoryHandleId memId,
        size_t size,
        SharedTextureMemoryPoolOption option
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pDevice);

        NN_VI_PROCESS_START();

        NN_VI_LOG_DEV("[share][mempool]Loading Proc\n");
        PFNNVNMEMORYPOOLBUILDERSETNATIVEHANDLENVXPROC nvnMemoryPoolBuilderSetNativeHandleNVX = nullptr;
        nvnMemoryPoolBuilderSetNativeHandleNVX = (PFNNVNMEMORYPOOLBUILDERSETNATIVEHANDLENVXPROC)nvnDeviceGetProcAddress(pDevice, "nvnMemoryPoolBuilderSetNativeHandleNVX");
        NN_RESULT_THROW_UNLESS(nvnMemoryPoolBuilderSetNativeHandleNVX != nullptr, nn::vi::ResultNotSupported());

        NN_VI_LOG_DEV("[share][mempool]Importing shared buffer\n");
        NvRmMemHandle hMem = {};
        NN_RESULT_THROW_UNLESS(NvRmMemHandleFromFd(static_cast<int>(memId._data), &hMem) == NvSuccess, nn::vi::ResultOperationFailed());
        NN_VI_PROCESS_ROLLBACK(NvRmMemHandleFree(hMem));

        // check size
        size_t memSize = NvRmMemGetSize(hMem);
        NN_RESULT_THROW_UNLESS(memSize >= size, nn::vi::ResultOperationFailed());

        NN_VI_LOG_DEV("[share][mempool]Creating imported physical pool\n");
        NVNmemoryPool& importedPool = m_ImportedMemoryPool;
        {
            int flags =
                NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT |
                NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT |
                (ExtractCompressibleOption(option) ? NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT : 0);
            NVNmemoryPoolBuilder builder;
            nvnMemoryPoolBuilderSetDevice(&builder, pDevice);
            nvnMemoryPoolBuilderSetDefaults(&builder);
            nvnMemoryPoolBuilderSetFlags(&builder, flags);
            nvnMemoryPoolBuilderSetNativeHandleNVX(&builder, CastRmMemHandleToUint64(hMem));
            NN_RESULT_THROW_UNLESS(nvnMemoryPoolInitialize(&importedPool, &builder), nn::vi::ResultOperationFailed());
        }
        NN_VI_PROCESS_ROLLBACK(nvnMemoryPoolFinalize(&importedPool));

        NN_VI_PROCESS_SUCCESS();
        m_Option = option;
        m_MemoryHandle = CastRmMemHandleToUint64(hMem);
        m_MemorySize   = size;
        m_State = SharedTextureMemoryPoolState_Initialized;
        NN_RESULT_SUCCESS;
    }

    void SharedTextureMemoryPool::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        nvnMemoryPoolFinalize(&m_ImportedMemoryPool);
        NvRmMemHandleFree(CastUint64ToRmMemHandle(m_MemoryHandle));
        m_MemorySize = 0;
        m_MemoryHandle = 0;
        m_Option = SharedTextureMemoryPoolOption_None;
        m_State = SharedTextureMemoryPoolState_Invalid;
    }

    NVNmemoryPool* SharedTextureMemoryPool::GetNvnImportedMemoryPool() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return &m_ImportedMemoryPool;
    }

}}}
