﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/usb/pd/driver/usb_Pd.h>

#include "../detail/usb_IPdCradleManager.h"
#include "../detail/usb_IPdCradleSession.h"
#include "../detail/usb_PdServiceName.h"

#include "usb_PdCradleManagerImpl.h"

namespace nn { namespace usb { namespace pd { namespace server {


// IPdCradleSession 実装クラス
class PdCradleManagerImpl::PdCradleSessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、セッションに必要な情報を渡す
    PdCradleSessionImpl(PdCradleManagerImpl* pParent, nn::usb::pd::driver::CradleSession internalSession) NN_NOEXCEPT
        : m_Parent(pParent, true)
        , m_InternalSession(internalSession)
    {}

    // Session のデストラクタで Session のクローズを行う
    ~PdCradleSessionImpl()
    {
        nn::usb::pd::driver::CloseCradleSession(&m_InternalSession);
    }

    // USB::PD ドライバライブラリの API のうち SystemEvent を使わないもの
    nn::Result SetCradleVdo(std::uint32_t value, std::uint32_t command) NN_NOEXCEPT;
    nn::Result GetCradleVdo(nn::sf::Out<std::uint32_t> outValue, std::uint32_t command) NN_NOEXCEPT;
    nn::Result ResetCradleUsbHub() NN_NOEXCEPT;
    nn::Result GetHostPdcFirmwareType(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT;
    nn::Result GetHostPdcFirmwareRevision(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT;
    nn::Result GetHostPdcManufactureId(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT;
    nn::Result GetHostPdcDeviceId(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT;
    nn::Result EnableCradleRecovery(nn::sf::Out<bool> isSuspended) NN_NOEXCEPT;
    nn::Result DisableCradleRecovery(nn::sf::Out<bool> isSuspended) NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<PdCradleManagerImpl> m_Parent;

    // USB::PD ライブラリが持つ Session 構造体のメンバ変数
    nn::usb::pd::driver::CradleSession m_InternalSession;
};

// ここから各 USB::PD API の内部実装
nn::Result PdCradleManagerImpl::PdCradleSessionImpl::SetCradleVdo(std::uint32_t value, std::uint32_t command) NN_NOEXCEPT
{
    Result result;
    result = nn::usb::pd::driver::SetCradleVdo( m_InternalSession, value, command );

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::GetCradleVdo(nn::sf::Out<std::uint32_t> outValue, std::uint32_t command) NN_NOEXCEPT
{
    Result result;
    uint32_t tmp;
    result = nn::usb::pd::driver::GetCradleVdo( &tmp, m_InternalSession, command );
    *outValue = tmp;

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::ResetCradleUsbHub() NN_NOEXCEPT
{
    Result result;
    result = nn::usb::pd::driver::ResetCradleUsbHub( m_InternalSession );

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::GetHostPdcFirmwareType(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT
{
    Result result;
    uint16_t tmp;
    result = nn::usb::pd::driver::GetHostPdcFirmwareType( &tmp, m_InternalSession );
    *outValue = tmp;

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::GetHostPdcFirmwareRevision(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT
{
    Result result;
    uint16_t tmp;
    result = nn::usb::pd::driver::GetHostPdcFirmwareRevision( &tmp, m_InternalSession );
    *outValue = tmp;

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::GetHostPdcManufactureId(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT
{
    Result result;
    uint16_t tmp;
    result = nn::usb::pd::driver::GetHostPdcManufactureId( &tmp, m_InternalSession );
    *outValue = tmp;

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::GetHostPdcDeviceId(nn::sf::Out<std::uint16_t> outValue) NN_NOEXCEPT
{
    Result result;
    uint16_t tmp;
    result = nn::usb::pd::driver::GetHostPdcDeviceId( &tmp, m_InternalSession );
    *outValue = tmp;

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::EnableCradleRecovery(nn::sf::Out<bool> isSuspended) NN_NOEXCEPT
{
    Result result;
    bool tmp;
    result = nn::usb::pd::driver::EnableCradleRecovery( &tmp, m_InternalSession );
    *isSuspended = tmp;

    return result;
}

nn::Result PdCradleManagerImpl::PdCradleSessionImpl::DisableCradleRecovery(nn::sf::Out<bool> isSuspended) NN_NOEXCEPT
{
    Result result;
    bool tmp;
    result = nn::usb::pd::driver::DisableCradleRecovery( &tmp, m_InternalSession );
    *isSuspended = tmp;

    return result;
}

// マネージャーの実装
PdCradleManagerImpl::PdCradleManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

PdCradleManagerImpl::~PdCradleManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result PdCradleManagerImpl::OpenCradleSession(nn::sf::Out<nn::sf::SharedPointer<nn::usb::pd::detail::IPdCradleSession>> outSession) NN_NOEXCEPT
{
    // USB::PD ドライバライブラリに渡す Session 構造体
    nn::usb::pd::driver::CradleSession internalSession;

    nn::usb::pd::driver::OpenCradleSession(&internalSession);

    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IPdCradleSession, PdCradleManagerImpl::PdCradleSessionImpl>(&m_Allocator, this, internalSession);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}}}
