﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/usb/pd/driver/usb_Pd.h>

#include "usb_PdEventHolder.h"
#include "usb_PdLog.h"

namespace nn {
namespace usb {
namespace pd {
namespace driver {
namespace detail {

// 論理的な通信セッションの最大値
static const int MaxSessions = 7;               // psm、apm、omm、usb、テストx2、予備
static const int MaxCradleSessions = 4;         // omm、テストx2、製造工程

class Session
{
    NN_DISALLOW_COPY(Session);
    NN_DISALLOW_MOVE(Session);

    typedef nn::util::IntrusiveList<EventHolder, nn::util::IntrusiveListMemberNodeTraits<EventHolder, &EventHolder::m_Node>>  BoundEventHolderList;
    typedef ::nn::util::TypedStorage<EventHolder, sizeof(EventHolder), NN_ALIGNOF(EventHolder)> EventHolderStorage;

public:
    Session() NN_NOEXCEPT :
        m_InitializeCount(0),
        m_InitializeCountMutex(false),
        m_SessionMutex(true)
        {
            for ( int i = 0; i < MaxSessions; i++ )
            {
                m_IsSessionOpen[i] = false;
            }
            for ( int i = 0; i < MaxCradleSessions; i++ )
            {
                m_IsCradleSessionOpen[i] = false;
            }
            memset( &m_EventHolder, 0, sizeof(m_EventHolder) );
        }

    void     Initialize() NN_NOEXCEPT;
    void     Finalize() NN_NOEXCEPT;

    void     OpenSession( driver::Session* pOutSession ) NN_NOEXCEPT;
    void     CloseSession( const driver::Session& session ) NN_NOEXCEPT;
    bool     IsOpen( int sessionId ) const NN_NOEXCEPT;
    void     OpenCradleSession( driver::CradleSession* pOutSession ) NN_NOEXCEPT;
    void     CloseCradleSession( const driver::CradleSession& session ) NN_NOEXCEPT;
    bool     IsCradleOpen( int sessionId ) const NN_NOEXCEPT;

    Result   BindNoticeEvent( nn::os::SystemEventType* pEvent, const driver::Session& session ) NN_NOEXCEPT;
    Result   UnbindNoticeEvent( const driver::Session& session ) NN_NOEXCEPT;

    void     NoticeEventCallback( int sessionId ) NN_NOEXCEPT;
    void     NoticeEventCallback() NN_NOEXCEPT;

private:
    int      FindFreeSessionId( bool* isOpenTable, int maxSessions ) const NN_NOEXCEPT;

private:
    int           m_InitializeCount;
    bool          m_IsSessionOpen[MaxSessions];
    bool          m_IsCradleSessionOpen[MaxCradleSessions];
    nn::os::Mutex m_InitializeCountMutex;  // ロック対象は m_InitializeCount
    nn::os::Mutex m_SessionMutex;          // ロック対象はセッション処理

    // 紐付済み Session を管理するリスト
    BoundEventHolderList m_NoticeEventList;
    EventHolderStorage m_EventHolder[MaxSessions];
};

} // detail
} // driver
} // pd
} // usb
} // nn
