﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/util/util_BitPack.h>

#include "usb_PdDriver.h"

namespace nn{
namespace usb{
namespace pd{
namespace driver {
namespace detail{

void Driver::StartRecoveringOverVoltage() NN_NOEXCEPT
{
    // USB 端子の高電圧通知
    NotifyOverVoltage( true );
    // バッテリが無い場合はリセットできないため OVP 復旧は非サポート
    if( !IsBatterySupported() )
    {
        USBPD_ABORT("Over Voltage Protection is prohibited on Copper!");
    }
    // HW リセット
    HardwareReset( true );
    StartPdcResetToAcOkTimer( HardResetToAcOkTimeSpan );
    // PDC リセット後の再設定
    SetupPdc();
    // PDC の SpdSrc 再設定要求
    m_PowerRequestEnableEvent.Signal();
    // 高電圧回復検出タイマー起動
    m_OvpRecoverTimerEvent.Stop();
    m_OvpRecoverTimerEvent.Clear();
    m_OvpRecoverTimerEvent.StartOneShot( OvpRecoverTimeSpan );
}

void Driver::StopRecoveringOverVoltage() NN_NOEXCEPT
{
    NotifyOverVoltage( false );
    m_OvpRecoverTimerEvent.Stop();
    m_OvpRecoverTimerEvent.Clear();
}

void Driver::NotifyOverVoltage( bool isOverVoltage ) NN_NOEXCEPT
{
    Result result;

    if ( isOverVoltage )
    {
        // HW リセットのリトライ回数制限
        if ( m_HwResetRetryCount == HwResetRetryMax ||
             ++m_HwResetRetryCount < HwResetRetryMax
           )
        {
            return;
        }
        USBPD_WARNING("Notify over voltage!\n");
    }
    else
    {
        int hwResetRetryCount = m_HwResetRetryCount;
        m_HwResetRetryCount = 0;
        if ( hwResetRetryCount < HwResetRetryMax )
        {
            return;
        }
        USBPD_DBG_LOG("Recover from over voltage!\n");
    }

    // USB 端子の高電圧（or 解除）通知
    m_IpcError = StatusError_None;
    if ( isOverVoltage )
    {
        m_IpcError = StatusError_OverVoltage;
    }
    UpdateStatus( false, Notice::ErrorNotice::Mask );
}

bool Driver::RecoverHardReset() NN_NOEXCEPT
{
    Result result;

    // Plug Remove 前に Hard Reset が発生する場合の誤検出防止
    if ( m_pAlertEvent[AlertStatus::PlugInsertRemove::Pos]->TimedWait( PdDebounceTimeSpan ) )
    {
        return true;
    }
    // Plug Insert 維持のチェック
    result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    UpdateAlert();
    if ( !m_pAlertEvent[AlertStatus::PlugInsertRemove::Pos]->TryWait() && m_Status1.Get<Status1::PlugInsertion>() )
    {
        StartPdcResetToAcOkTimer( HardResetToAcOkTimeSpan );
        if ( !m_CradleRecoveryEnabled )
        {
            USBPD_WARNING("PDC recovery is suspended!\n");

            // クレードルの復旧処理で再接続が発生するのを上位層から拒否されている時に Hard Reset が発生
            m_CradleAwakeSuspended = true;
            return false;
        }
        m_CradleAwakeSuspended = false;

#ifndef ENABLE_PDC_HARD_RESET_SLEEP_TEST
        if ( m_Status1.Get<Status1::DataRole>() == Status1DataRole_Ufp )
#endif
        {
            USBPD_WARNING("Recover from PDC hard reset!\n");

            ClearStatusBase();
            // 非アクティブ状態をできるだけ早く通知
            UpdateStatus( false );
            // HW リセット
            HardwareReset( false );
            // OVP エラーアラートのクリア
            m_pAlertEvent[AlertStatus::Error::Pos]->Clear();
            // Hard Reset 発生時に UFP かつ Plug Insert 維持であれば強制切断
            ForceToRemoveDevice();
            StartPdcResetToAcOkTimer( SystemResetToAcOkTimeSpan );
            // PDC リセット後の再設定
            SetupPdc();
            // PDC の SpdSrc 再設定要求
            m_PowerRequestEnableEvent.Signal();
        }
    }

    return true;
}

void Driver::StartPdcResetToAcOkTimer( nn::TimeSpan timeSpan ) NN_NOEXCEPT
{
    m_IsPdcResetToAcOkTimerEnabled = true;
    m_PdcResetToAcOkTimerEvent.Stop();
    m_PdcResetToAcOkTimerEvent.Clear();
    m_PdcResetToAcOkTimerEvent.StartOneShot( timeSpan );
}

void Driver::WaitForAcOkFromPdcReset() NN_NOEXCEPT
{
    if ( m_IsPdcResetToAcOkTimerEnabled )
    {
        USBPD_WARNING("Waiting for /ACOK from PDC hard reset!\n");
        m_PdcResetToAcOkTimerEvent.Wait();
        USBPD_WARNING("Waited for /ACOK from PDC hard reset!\n");
        m_IsPdcResetToAcOkTimerEnabled = false;
    }
    m_PdcResetToAcOkTimerEvent.Stop();
    m_PdcResetToAcOkTimerEvent.Clear();
}

Result Driver::EnableCradleRecovery( bool* pIsSuspended, int sessionId ) NN_NOEXCEPT
{
    USBPD_DBG_LOG("EnableCradleRecovery\n");
    return EnableCradleRecoveryCore( pIsSuspended, sessionId, true );
}

Result Driver::DisableCradleRecovery( bool* pIsSuspended, int sessionId ) NN_NOEXCEPT
{
#ifdef DISABLE_TO_DISABLE_CRADLE_RECOVERY
    return ResultSuccess();
#else
    USBPD_DBG_LOG("DisableCradleRecovery\n");
    return EnableCradleRecoveryCore( pIsSuspended, sessionId, false );
#endif
}

Result Driver::EnableCradleRecoveryCore( bool* pIsSuspended, int sessionId, bool enable ) NN_NOEXCEPT
{
    if ( !m_CradleRecoveryInitialized )
    {
        m_CradleRecoveryInitialized = true;
        m_CradleRecoverySessionId = sessionId;
    }
    else if ( sessionId != m_CradleRecoverySessionId )
    {
        return ResultInvalidSession();
    }
    m_CradleRecoveryEnabled = enable;
    *pIsSuspended = m_CradleAwakeSuspended;
    if ( enable && m_CradleAwakeSuspended )
    {
        m_pAlertEvent[AlertStatus::HardReset::Pos]->Signal();
    }
    return ResultSuccess();
}

} // detail
} // driver
} // pd
} // usb
} // nn
