﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/gpio/gpio.h>
#include <nn/i2c/i2c.h>
#include <nn/psc.h>

#include <nn/usb/pd/usb_PdType.h>
#include <nn/usb/pd/usb_PdCradleVdm.h>
#include "usb_PdLog.h"
#include "usb_PdResult.h"
#include "usb_PdRegister.h"
#include "usb_PdSession.h"


//#define ENABLE_PDC_FW_BOOT_TIMEOUT_TEST
//#define ENABLE_PDC_I2C_NACK_TEST
//#define ENABLE_PDC_HARD_RESET_IN_SLEEP_TEST
//#define ENABLE_PDC_HARD_RESET_INTO_SLEEP_READY_TEST
//#define ENABLE_PDC_HARD_RESET_INTO_MINIMUM_AWAKE_TEST
//#define ENABLE_PDC_HARD_RESET_OUT_OF_MINIMUM_AWAKE_TEST
//#define ENABLE_PDC_HARD_RESET_INTO_FULL_AWAKE_TEST
//#define ENABLE_PDC_COMMAND1_SIZE_WORKAROUND_TEST
//#define ENABLE_ACOK_WAIT_AFTER_SYSTEM_RESET_TEST


#if defined(ENABLE_ACOK_WAIT_AFTER_SYSTEM_RESET_TEST)
#if !defined(ENABLE_PDC_HARD_RESET_INTO_SLEEP_READY_TEST)
#define ENABLE_PDC_HARD_RESET_INTO_SLEEP_READY_TEST
#endif
#define DISABLE_TO_DISABLE_CRADLE_RECOVERY
#endif
#if defined(ENABLE_PDC_HARD_RESET_IN_SLEEP_TEST) || \
    defined(ENABLE_PDC_HARD_RESET_INTO_SLEEP_READY_TEST) || \
    defined(ENABLE_PDC_HARD_RESET_INTO_MINIMUM_AWAKE_TEST) || \
    defined(ENABLE_PDC_HARD_RESET_OUT_OF_MINIMUM_AWAKE_TEST) || \
    defined(ENABLE_PDC_HARD_RESET_INTO_FULL_AWAKE_TEST)
#define ENABLE_PDC_HARD_RESET_SLEEP_TEST
#endif


namespace nn {
namespace usb {
namespace pd {
namespace driver {
namespace detail {

/**
 * @brief   コールバック関数の型宣言
 */
typedef void (*CallbackType)(void* arg);

/**
 * @brief   PD コンのドライバークラス。
 * @detail  実質的なドライバのコア機能の実装部分。
 *          コピー・代入・移動禁止。
 */
class Driver
{
    NN_DISALLOW_COPY(Driver);
    NN_DISALLOW_MOVE(Driver);

public:
    Driver() NN_NOEXCEPT
    {
    }

    void Initialize( nn::i2c::SpeedMode i2cSpeed ) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void AttachSessionObject( Session* pSession ) NN_NOEXCEPT;
    void DetachSessionObject() NN_NOEXCEPT;

    bool IsActive() const NN_NOEXCEPT { return m_IsActive; };
    Result GetStatus( Status* status, int sessionId ) NN_NOEXCEPT;
    Result GetStatusCore( Status* pStatus ) NN_NOEXCEPT;
    Result GetNotice( Notice* pNotice, int sessionId ) NN_NOEXCEPT;
    Result EnablePowerRequestNotice( int sessionId ) NN_NOEXCEPT;
    Result DisablePowerRequestNotice( int sessionId ) NN_NOEXCEPT;
    Result ReplyPowerRequest( int sessionId, bool isSuccess ) NN_NOEXCEPT;
    Result EnableCradleRecovery( bool* pIsSuspended, int sessionId ) NN_NOEXCEPT;
    Result DisableCradleRecovery( bool* pIsSuspended, int sessionId ) NN_NOEXCEPT;

    FirmwareType GetHostPdcFirmwareType() const NN_NOEXCEPT { return m_HostPdcFwType; };
    FirmwareRevision GetHostPdcFirmwareRevision() const NN_NOEXCEPT { return m_HostPdcFwRevision; };
    ManufactureId GetHostPdcManufactureId() const NN_NOEXCEPT { return m_ManufactureId; };
    DeviceId GetHostPdcDeviceId() const NN_NOEXCEPT { return m_DeviceId; };

    bool IsSleeping() const NN_NOEXCEPT { return m_IsSleeping; };
    Result SleepRequest() NN_NOEXCEPT;
    Result MinimumAwakeRequest() NN_NOEXCEPT;
    Result FullAwakeRequest() NN_NOEXCEPT;

    bool IsCradleFamily( StatusDeviceType deviceType ) NN_NOEXCEPT;
    Result ResetCradleUsbHub() NN_NOEXCEPT;
    Result SendAndReceiveNxVdm( void* pDst, size_t dstSize, const void* pSrc, VdmCommand command ) NN_NOEXCEPT;
    Result SendAndReceiveNxVdmWithRequest( void* pDst, size_t dstSize, const void* pSrc, VdmSet rw, VdmCommand command ) NN_NOEXCEPT;

    Result SendCommand1( const void* pSrc, size_t size, L1Command command1 ) NN_NOEXCEPT;
    Result SendAndWaitCommand2( Status1* pStatus1, int alertMask, L2Command command2 ) NN_NOEXCEPT;
    Result ReceiveCommand1( void* pDst, size_t dstSize, L1Command command1, size_t* pReceivedSize = nullptr ) NN_NOEXCEPT;

private:
    void GetSettingsItemValue( void* buffer, size_t bufferSize, const char* name, const char* key ) NN_NOEXCEPT;
    bool IsBatterySupported() const NN_NOEXCEPT { return m_IsBatterySupported; };
    void InitializePdc() NN_NOEXCEPT;
    void FinalizePdc() NN_NOEXCEPT;
    void SetupPdc() NN_NOEXCEPT;
    void WaitDeviceId() NN_NOEXCEPT;
    void InitializeBuses( nn::i2c::SpeedMode i2cSpeed ) NN_NOEXCEPT;
    void FinalizeBuses() NN_NOEXCEPT;
    Result ForceToRemoveDevice() NN_NOEXCEPT;
    void HardwareReset( bool isPreWait ) NN_NOEXCEPT;

    // Base
    static void AlertThread(void* arg) NN_NOEXCEPT;
    static void ReceiveVdmThread(void* arg) NN_NOEXCEPT;
    static void ExecuteThread( void* arg ) NN_NOEXCEPT;
    VdmNoticeType SearchVdmNoticeType( VdmCommand c ) NN_NOEXCEPT;
    void StartAlertThread() NN_NOEXCEPT;
    void AlertThreadBoby() NN_NOEXCEPT;
    void SignalAlertEvents( AlertStatus alert ) NN_NOEXCEPT;
    void ReceiveVdmThreadBoby() NN_NOEXCEPT;
    void ExecuteThreadBody() NN_NOEXCEPT;
    void ExecuteAlertEvents( int alertIndex ) NN_NOEXCEPT;
    void SignalNoticeEvent( int sessionId ) NN_NOEXCEPT;
    void SignalNoticeEventAll() NN_NOEXCEPT;
    void GetInitialStatus() NN_NOEXCEPT;
    void PrintPdcBaseStatuses() NN_NOEXCEPT;
    void ClearStatusBase() NN_NOEXCEPT;
    void PrintPlugStatus() NN_NOEXCEPT;
    void DeactivateOnOtgRemove() NN_NOEXCEPT;
    void ActivateOnPlugInsert( bool* pIsInsertPreCheck, Bit16* pForceNotice ) NN_NOEXCEPT;
    void EnableIsActive() NN_NOEXCEPT;
    void SignalFullAwakeReadyEvent( int alertIndex ) NN_NOEXCEPT;
    bool IdentifyAcAdaptor( VdmProductId pid ) NN_NOEXCEPT;
    void UpdateStatus( bool isInsertCheck, Bit16 noticeMask = 0 ) NN_NOEXCEPT;
    bool GetStatusOnActive( Status* pStatus ) NN_NOEXCEPT;
    bool GetStatusOnFakeActive( Status* pStatus ) NN_NOEXCEPT;
    void UpdateAlert() NN_NOEXCEPT;
    Result SendCommand1Core( const void* pSrc, size_t size, L1Command command1 ) NN_NOEXCEPT;
    Result SendCommand2( L2Command command2 ) NN_NOEXCEPT;
    Result SendCommand2Core( L2Command command2 ) NN_NOEXCEPT;

    // 異常系
    void StartRecoveringOverVoltage() NN_NOEXCEPT;
    void StopRecoveringOverVoltage() NN_NOEXCEPT;
    void NotifyOverVoltage( bool isOverVoltage ) NN_NOEXCEPT;
    bool RecoverHardReset() NN_NOEXCEPT;
    void StartPdcResetToAcOkTimer( nn::TimeSpan timeSpan ) NN_NOEXCEPT;
    void WaitForAcOkFromPdcReset() NN_NOEXCEPT;

    // Cradle
    bool EnableCradle() NN_NOEXCEPT;
    bool ChangePower( bool* pIsNonPdDevice ) NN_NOEXCEPT;
    bool ChangePowerCore( bool* pIsNonPdDevice, int voltage ) NN_NOEXCEPT;
    bool WaitForPowerContract() NN_NOEXCEPT;
    void GetCurrentPower( int* pVoltage, int* pMaxAmperePdo, int* pMaxAmpereRdo ) NN_NOEXCEPT;
    bool SelectNewPower( bool* pIsNonPdDevice, int* pPdoPosition, int* pVolt, int* pAmp, int voltage ) NN_NOEXCEPT;
    bool RequestNewPower( int pdoPosition, int volt, int amp ) NN_NOEXCEPT;
    Result ChangeUfpIntoDfp() NN_NOEXCEPT;
    Result IdentifyCradle() NN_NOEXCEPT;
    bool IsDisplayPortPowerEnough() NN_NOEXCEPT;
    Result EnableDisplayPort() NN_NOEXCEPT;
    bool GetCradleVdmStatus() NN_NOEXCEPT;
    void NotifyCradleError() NN_NOEXCEPT;
    void StartCradleUsbHubTimer() NN_NOEXCEPT;
    void StopCradleUsbHubTimer() NN_NOEXCEPT;
    void DetectUsbHubInCradle() NN_NOEXCEPT;

    // OnTheGo
    void SetSpdSrc( ControllerConfiguration1SpdSrc spdSrc ) NN_NOEXCEPT;
    bool EnableOnTheGo() NN_NOEXCEPT;
    bool OutputVBusPower() NN_NOEXCEPT;
    bool InputVBusPower() NN_NOEXCEPT;
    Result EnablePowerRequestNoticeCore( int sessionId, bool enable ) NN_NOEXCEPT;
    void NotifyPowerRequest( StatusRequest request ) NN_NOEXCEPT;
    Result WaitPowerReply() NN_NOEXCEPT;
    Result EnableCradleRecoveryCore( bool* pIsSuspended, int sessionId, bool enable ) NN_NOEXCEPT;

    // Copper
    bool EnableOnCopper() NN_NOEXCEPT;
    bool GetPowerOnCopper() NN_NOEXCEPT;

    // Sleep
    Result SleepAndWaitForMinimumAwake() NN_NOEXCEPT;
    Result WaitForAcOkFromPdcResetBeforeSleep() NN_NOEXCEPT;
    Result Sleep() NN_NOEXCEPT;
    Result MinimumAwake() NN_NOEXCEPT;
    Result FullAwake() NN_NOEXCEPT;
    Result PmStateRequestCore( nn::psc::PmState pmState ) NN_NOEXCEPT;

    // VDM
    Result SendStandardVdm( Status1* pStatus1, VdmHeader* pHeader, VdmHeaderCommand command ) NN_NOEXCEPT;
    Result ReceiveStadardVdm( Status2* pStatus2, void* pDst, size_t size ) NN_NOEXCEPT;
    Result SendNxVdm( const void* pSrc, VdmCommand command ) NN_NOEXCEPT;
    Result SendNxVdmWithRequest( Status1* pStatus1, VdmHeader* pHeader, const void* pSrc, VdmSet rw, VdmCommand command ) NN_NOEXCEPT;
    Result ReceiveNxVdm( void* pDst, size_t size, const VdmHeader* pHeader, VdmCommand command ) NN_NOEXCEPT;
    Result ReceiveNxVdmCore( void* pDst, size_t size ) NN_NOEXCEPT;
    Result AcceptIncomingVdm( Status1* pStatus1 ) NN_NOEXCEPT;

public:

private:
    int m_InitializeCount{};
    nn::os::Event m_InitializePdcEvent{ nn::os::EventClearMode_ManualClear };
    bool m_IsFinalizing{ false };
    bool m_IsBatterySupported{ false };
    bool m_IsPdcBootFatalEnabled{ false };
    Result m_ResetResult{ ResultSuccess() };

    Status m_IpcStatus{};
    StatusRequest m_IpcRequest{};
    StatusError m_IpcError{};
    Notice m_IpcNotice[MaxSessions]{};
    nn::os::Mutex m_IpcStatusMutex{ true };
    Session* m_pSession{};

    bool m_IsActive{};
    bool m_IsDisplayPortAlternateMode{};
    bool m_IsPowerShortage{};
    DeviceId m_DeviceId{};
    ManufactureId m_ManufactureId{};
    FirmwareType m_HostPdcFwType{};
    FirmwareRevision m_HostPdcFwRevision{};
    VdmDeviceState m_CradleDeviceStatus{};
    VdmPdcHFwVersion m_CradlePdcHFwVersion{};
    VdmMcuFwVersion m_CradleMcuFwVersion{};
    DiscoverIdentityVdm m_IdentityVdm{};

    nn::os::ThreadType m_ExecuteThread{};
    nn::os::Event m_FinalizeExecuteEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_FinalizeExecuteEventHolder{};
    nn::os::MultiWaitHolderType m_CradleErrorEventHolder{};

    nn::os::ThreadType m_AlertThread{};
    nn::os::Event* m_pAlertEvent[AlertNum]{};
    uint64_t m_AlertEventBuffer[AlertNum][sizeof(nn::os::Event) / 8 + 1]{};
    nn::os::MultiWaitType m_AlertGpioWaiter{};
    nn::os::MultiWaitType m_ExecuteWaiter{};
    nn::os::MultiWaitHolderType m_AlertGpioEventHolder{};
    nn::os::MultiWaitHolderType m_AlertEventHolder[AlertNum]{};
    nn::os::Event m_FinalizeAlertEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_FinalizeAlertEventHolder{};
    nn::os::Mutex m_AlertMutex{ true };

    nn::os::MultiWaitType m_NewContractWaiter{};
    nn::os::Event m_NewContractEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_NewContractCancelEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_NewContractEventHolder{};
    nn::os::MultiWaitHolderType m_NewContractCancelEventHolder{};

    VdmCommonVdo1 m_CradleUsbHubState{};
    int m_CradleUsbHubRepeatCount{};
    nn::os::TimerEvent m_CradleUsbHubTimerEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_CradleUsbHubTimerEventHolder{};

    nn::os::ThreadType m_ReceiveVdmThread{};
    nn::os::Mutex m_ReceiveVdmMutex{ true };
    nn::os::Event m_StandardVdmEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Mutex m_ReplyCommonVdmMutex{ true };
    nn::os::Event m_ReplyCommonVdmEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event* m_pNoticeVdmEvent[VdmNoticeType_Num]{};
    bool m_IsVdmCanceled{ false };
    uint64_t m_NoticeEventBuffer[VdmNoticeType_Num][sizeof(nn::os::Event) / 8 + 1]{};
    VdmUnit m_StandardVdm[VdmCommandLength_Max]{};
    VdmUnit m_ReplyCommonVdm[VdmCommandLength_Max]{};
    VdmUnit m_NoticeVdm[VdmNoticeType_Num][VdmCommandLength_Max]{};

    uint8_t m_CommandBuf[L1CommandSize_Max + 2]{}; // ＋ 送信コマンド ＋ データカウント
    AlertStatus m_AlertStatus{};
    Status1 m_Status1{};
    Status2 m_Status2{};
    DisplayPortStatus m_DisplayPortStatus{};
    DisplayPortAlertEnable m_DisplayPortAlertEnable{};
    Pdo m_CurrentPdo{};
    Rdo m_CurrentRdo{};
    Pdo m_SelectedPdo{};  // new PDO リストから選択された PDO
    Rdo m_SetRdo{};       // SetRdo レジスタに格納した RDO
    bool m_IsSendRdoCompleted{ false };
    nn::os::Mutex m_L2CommandMutex{ true };

    // OTG
    bool m_IsChargerOtgMode{ false };
    nn::os::Event m_PowerRequestEnableEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_PowerRequestEnableEventHolder{};
    nn::os::Event m_PowerReplyEvent{ nn::os::EventClearMode_ManualClear };
    int m_PowerRequestNoticeSessionId{};
    bool m_PowerRequestNoticeInitialized{ false };
    bool m_IsPowerRequestEnabled{ false };
    bool m_IsPowerReplySuccess{ false };
    bool m_IsVBusOutputCanceled{ false };
    nn::os::TimerEvent m_VBusOutputTimerEvent{ nn::os::EventClearMode_ManualClear };

    // GPIO
    nn::gpio::GpioPadSession m_AlertSession{};
    nn::gpio::GpioPadSession m_PdResetSession{};
    nn::os::SystemEvent m_AlertGpioEvent{ nn::os::EventClearMode_ManualClear, true };

    // I2C
    nn::i2c::I2cSession m_I2cSession{};
    nn::os::Mutex m_I2cMutex{ true };

    // HW リセット
    int m_HwResetRetryCount{};
    nn::os::TimerEvent m_OvpRecoverTimerEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_OvpRecoverTimerEventHolder{};

    // スリープ / シャットダウン
    bool m_IsSleeping{ false };
    nn::psc::PmState m_PmState{ nn::psc::PmState_FullAwake };
    nn::os::MultiWaitHolderType m_SleepRequestEventHolder{};
    nn::os::MultiWaitHolderType m_FullAwakeRequestEventHolder{};
    nn::os::Event m_SleepRequestEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_SleepReplyEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_MinimumAwakeRequestEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_MinimumAwakeReplyEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_FullAwakeRequestEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_FullAwakeReplyEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::MultiWaitHolderType m_SleepAlertRequestEventHolder{};
    nn::os::Event m_SleepAlertRequestEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_SleepAlertReplyEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_AwakeAlertRequestEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Event m_AwakeAlertReplyEvent{ nn::os::EventClearMode_ManualClear };
    bool m_AwakeExecuteRequest{ false };
    nn::os::Event m_AwakeExecuteCompleteEvent{ nn::os::EventClearMode_ManualClear };
    nn::os::Mutex m_SleepRequestMutex{ true };
    int m_CradleRecoverySessionId{};
    bool m_CradleRecoveryInitialized{ false };
    bool m_CradleRecoveryEnabled{ true };
    bool m_CradleAwakeSuspended{ false };
    bool m_IsPdcResetToAcOkTimerEnabled{ false };
    nn::os::TimerEvent m_PdcResetToAcOkTimerEvent{ nn::os::EventClearMode_ManualClear };

private:
    // I2C 定数
    const nn::i2c::TransactionOption I2cSendConditions = static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition );
    const nn::i2c::TransactionOption I2cReceiveCommandConditions = nn::i2c::TransactionOption_StartCondition;
    const nn::i2c::TransactionOption I2cReceiveDataConditions = I2cSendConditions;

    // HW リセット最大リトライ回数（OVP 要因）
    const int HwResetRetryMax = 3;
    // クレードル USB HUB VBUS 検出リトライ回数
    const int CradleUsbHubRetryMax = 3;
    // コマンドリトライ回数
    const int CommnadRetryMax = 8;
    // Power Contract リトライ回数
    const int PowerContractRetryMax = 3;

    // HW リセット維持期間
    const nn::TimeSpan HwResetTimeSpan = nn::TimeSpan::FromMilliSeconds( 30 );
    // System Reset コマンド完了時間
    const nn::TimeSpan SystemResetTimeSpan = nn::TimeSpan::FromMilliSeconds( 10 );
    // FW ブートのタイムアウト時間
    const nn::TimeSpan FirmwareBootTimeoutSpan = nn::TimeSpan::FromMilliSeconds( 10 * 1000 );
    // FW ブート後 System Reset にすぐ応答できない期間
    const nn::TimeSpan FirmwareBootToSystemResetTimeSpan = nn::TimeSpan::FromMilliSeconds( 20 );
    // USB PD 規格 Debounce 時間 ＋ Switch マージン（IAAA-2513）
    const nn::TimeSpan PdDebounceTimeSpan = nn::TimeSpan::FromMilliSeconds( 23 );
    // VBUS 放電時間
    const nn::TimeSpan VBusDischargeTimeSpan = nn::TimeSpan::FromMilliSeconds( 100 );
    // VBUS 出力完了時間 (SIGLO-73178) (SIGLO-82568)
    // VBUS から 5V 出力にかかる最大時間(400 msec)
    const nn::TimeSpan VBusOutputCompleteTimeSpan = nn::TimeSpan::FromMilliSeconds( 400 );
    // OVP 回復判定時間
    const nn::TimeSpan OvpRecoverTimeSpan = nn::TimeSpan::FromMilliSeconds( 3 * 1000 );
    // Power Contract タイムアウト時間
    const nn::TimeSpan NewContractConsumerTimeoutSpan = nn::TimeSpan::FromMilliSeconds( 5 * 1000 );
    // Power Contract → PS_RDY 間のウェイト時間（EP1 以前向け）
    const nn::TimeSpan WaitForPsRdyTimeSpan = nn::TimeSpan::FromMilliSeconds( 550 );
    // Power Contract → Set SYS_RDY コマンド間のウェイト時間（EP2-1 以前向け）
    const nn::TimeSpan WaitForSetSysRdyTimeSpan = nn::TimeSpan::FromMilliSeconds( 550 );
    // VDM Reply タイムアウト時間
    const nn::TimeSpan VdmReplyTimeoutSpan = nn::TimeSpan::FromMilliSeconds( 5 * 1000 );
    // クレードル USB HUB VBUS 検出開始時間
    const nn::TimeSpan CradleUsbHubStartTimeSpan = nn::TimeSpan::FromMilliSeconds( 1300 );
    // クレードル USB HUB VBUS 検出リピート時間
    const nn::TimeSpan CradleUsbHubRepeatTimeSpan = nn::TimeSpan::FromMilliSeconds( 100 );
    // System Reset から PMIC の /ACOK までの期間（SIGLO-49238: 0.8秒 + マージン）
    const nn::TimeSpan SystemResetToAcOkTimeSpan = nn::TimeSpan::FromMilliSeconds( 1000 );
    // Hard Reset 発生から PMIC の /ACOK までの期間（SIGLO-49238: 1.3秒 + マージン）
    const nn::TimeSpan HardResetToAcOkTimeSpan = nn::TimeSpan::FromMilliSeconds( 1500 );

    // Command2 タイムアウト時間（最大 SendPowerRoleSwap 1.6秒）
    const nn::TimeSpan Command2TimeoutSpan = nn::TimeSpan::FromMilliSeconds( 1700 );
};

extern const VdmCommandType g_VdmCommandTypeList[VdmCommand_Max + 1];
extern const VdmCommandLength g_VdmCommandLength[VdmCommand_Max + 1];
extern const char* const g_pDataRoleName[];

} // detail
} // driver
} // pd
} // usb
} // nn
