﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief eXtensible Host Controller Interface (XHCI) Driver
 *
 * @details
 *
 *
 */

#pragma once

#define NN_USB_XHCI_LOG(format, ...)                            \
    nn::usb::detail::Log(nn::usb::LogLevel_Info,                \
                         nn::usb::LogModule_HsXhci,             \
                         NN_USB_LOG_TAG, format, ##__VA_ARGS__)

namespace nn { namespace usb { namespace hs {

class XhciDriver : public HostControllerDriver
{
public:
    static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;

    // Software controlled limitations
    static const int32_t  LimitMaxDeviceSlotCount = 128;
    static const int32_t  LimitMaxEventRingSegmentTableEntryCount = 1;

    // Defaults
    static const uint32_t RtIntReg0ImodDefault = 0x000001F4U; // 8000 IRQs/second
    static const int32_t  MaxEventTrbs = 16 * 13;
    static const int32_t  MaxCommandTrbs = 16 * 1;

    /*
     * Ref [xHCI r1.1] Table 143.
     */
    NN_STATIC_ASSERT(MaxEventTrbs >= 16 && MaxEventTrbs <= 4096);

    XhciDriver(Hs *pHs, StartupMode mode, PlatformController *pPlatformController,
               detail::RegisterBlock& capRegister)
        : HostControllerDriver(pHs, HostControllerDriver::ControllerType_Xhci, mode, pPlatformController)
        , CapRegister(capRegister)
        , m_PageSize(0)
        , m_MaxDeviceSlotCount(0)
        , m_ControllerVersion(0)
        , m_pDmaDeviceContextArray(nullptr)
        , m_ScratchPadBufferCount(0)
        , m_pDmaScratchPadBufferArray(nullptr)
        , m_EventRingSegmentTableEntryCount(0)
        , m_pDmaEventRingSegmentTable(nullptr)
        , m_pDmaEvents(nullptr)
        , m_DmaEventsIoVa(0)
        , m_EventCcs(1)
        , m_EventIndex(0)
        , m_CommandFreeIndex(0)
        , m_CommandPendIndex(0)
        , m_CommandPcs(1)
        , m_CommandPendCount(0)
        , m_pDmaCommandRing(nullptr)
        , m_DmaCommandRingIoVa(0)
        , m_IsContext64b(false)
        , m_IsControllerStall(false)
    {
        memset(m_CommandRing, 0, sizeof(m_CommandRing));
        memset(m_pDevices, 0, sizeof(m_pDevices));

    }
    virtual ~XhciDriver()
    {

    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;
    Result Initialize();
    Result Finalize();

private:
//////////////////////////////////////////////////////////////////////////
// XHCI Standard Constants
//////////////////////////////////////////////////////////////////////////
/**
 * @brief xHCI Host Controller Capability Registers
 */
NN_USB_DEFINE_REG_OFFSET(HcCapBase,  0x0);
NN_USB_DEFINE_FIELD32(HcCapBase,      Length,         0,  8);   // How long is the Capabilities register
NN_USB_DEFINE_FIELD32(HcCapBase,      Version,       16, 16);   // XHCI version
//
NN_USB_DEFINE_REG_OFFSET(HcsParams1, 0x4);
NN_USB_DEFINE_FIELD32(HcsParams1,     MaxSlots,       0,  8);   // Max Device Slots
NN_USB_DEFINE_FIELD32(HcsParams1,     MaxIntrs,       8, 11);   // Max Interrupters
NN_USB_DEFINE_FIELD32(HcsParams1,     MaxPorts,      24,  8);   // Max Ports - max value is 0x7F = 127 ports
//
NN_USB_DEFINE_REG_OFFSET(HcsParams2, 0x8);
NN_USB_DEFINE_FIELD32(HcsParams2,     Ist,           0,   4);   // Number of frames that software needs to queue transactions in advance
NN_USB_DEFINE_FIELD32(HcsParams2,     ErstMax,       4,   4);   // Max number of Event Ring segments
NN_USB_DEFINE_FIELD32(HcsParams2,     MaxSpbHi,      21,  5);   // Max scratchpad upper
NN_USB_DEFINE_FIELD32(HcsParams2,     Spr,           26,  1);   // Scratchpad restore
NN_USB_DEFINE_FIELD32(HcsParams2,     MaxSpbLo,      27,  5);   // Max scratchpad lower
//
NN_USB_DEFINE_REG_OFFSET(HcsParams3, 0xc);
NN_USB_DEFINE_FIELD32(HcsParams3,     U1Latency,      0,  8);   // Max U1 to U0 latency for the roothub ports
NN_USB_DEFINE_FIELD32(HcsParams3,     U2Latency,     16, 16);   // Max U2 to U0 latency for the roothub ports
//
NN_USB_DEFINE_REG_OFFSET(HccParams,  0x10);
NN_USB_DEFINE_FIELD32(HccParams,      64BitAddr,     0,   1);   // HC can use 64-bit address pointers
NN_USB_DEFINE_FIELD32(HccParams,      BwNeg,         1,   1);   // HC can do bandwidth negotiation
NN_USB_DEFINE_FIELD32(HccParams,      64ByteContext, 2,   1);   // HC uses 64-byte Device Context structures
NN_USB_DEFINE_FIELD32(HccParams,      Ppc,           3,   1);   // HC has port power switches
NN_USB_DEFINE_FIELD32(HccParams,      Indicator,     4,   1);   // HC has port indicators
NN_USB_DEFINE_FIELD32(HccParams,      LightReset,    5,   1);   // HC has Light HC Reset Capability
NN_USB_DEFINE_FIELD32(HccParams,      Ltc,           6,   1);   // HC supports latency tolerance messaging
NN_USB_DEFINE_FIELD32(HccParams,      Nss,           7,   1);   // No secondary Stream ID Support if true
NN_USB_DEFINE_FIELD32(HccParams,      Psa,           12,  4);   // Max size for Primary Stream Arrays - 2^(n+1)
NN_USB_DEFINE_FIELD32(HccParams,      ExtCaps,       16,  16);  // First extended capability pointer)
//
NN_USB_DEFINE_REG_OFFSET(HcCapDoorbellRegisters,      0x14);
NN_USB_DEFINE_REG_OFFSET(HcCapRuntimeRegisters,       0x18);

/**
 * @brief xHCI Host Controller Operational Registers
 */
NN_USB_DEFINE_REG_OFFSET(OpRegUsbCmd, 0x0);
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Run,       0,  1);  // Start/stop HC execution - do not write unless HC is halted
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Reset,     1,  1);  // Resets internal HC state machine and all registers
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Eie,       2,  1);  // Event Interrupt Enable - get irq when EINT bit is set in USBSTS register
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Hseie,     3,  1);  // Host System Error Interrupt Enable - get irq when HSEIE bit set in USBSTS
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Lreset,    7,  1);  // Light reset (port status stays unchanged) - reset completed when this is 0
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Css,       8,  1);  // Host controller save state
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Crs,       9,  1);  // Host controller restore state
NN_USB_DEFINE_FIELD32(OpRegUsbCmd,      Ewe,       10, 1);  // Enable Wrap Event - '1' means xHC generates an event when MFINDEX wraps
//
NN_USB_DEFINE_REG_OFFSET(OpRegStatus, 0x4);
NN_USB_DEFINE_FIELD32(OpRegStatus,      Halt,      0,  1);  // HC not running
NN_USB_DEFINE_FIELD32(OpRegStatus,      Fatal,     2,  1);  // Serious error, will clear the run bit
NN_USB_DEFINE_FIELD32(OpRegStatus,      Eint,      3,  1);  // Event interrupt - clear this prior to clearing any IP flags in IR set
NN_USB_DEFINE_FIELD32(OpRegStatus,      Port,      4,  1);  // Port change detect
NN_USB_DEFINE_FIELD32(OpRegStatus,      Save,      8,  1);  // Save state status - '1' means xHC is saving state
NN_USB_DEFINE_FIELD32(OpRegStatus,      Restore,   9,  1);  // Restore state status - '1' means xHC is restoring state
NN_USB_DEFINE_FIELD32(OpRegStatus,      Sre,       10, 1);  // Save or restore error
NN_USB_DEFINE_FIELD32(OpRegStatus,      Cnr,       11, 1);  // Controller Not Ready to accept doorbell or op reg writes after reset
NN_USB_DEFINE_FIELD32(OpRegStatus,      Hce,       12, 1);  // Internal Host Controller Error - SW needs to reset and reinitialize
//
NN_USB_DEFINE_REG_OFFSET(OpRegPagesize, 0x8);
NN_USB_DEFINE_FIELD32(OpRegPagesize,,             0, 16);  // This xHC supports a page size of 2^(n+12) if bit n is Set
//
NN_USB_DEFINE_REG_OFFSET(OpRegDnctrl,0x14);
NN_USB_DEFINE_REG_OFFSET(OpRegCrcrLo,0x18);
NN_USB_DEFINE_FIELD32(OpRegCrcrLo,      Rcs,      0,  1);  // CCS flag for the TRB referenced by the Command Ring Pointer
NN_USB_DEFINE_REG_OFFSET(OpRegCrcrHi, 0x1C);
NN_USB_DEFINE_REG_OFFSET(OpRegDcbaapLo, 0x30);
NN_USB_DEFINE_REG_OFFSET(OpRegDcbaapHi, 0x34);
//
NN_USB_DEFINE_REG_OFFSET(OpRegConfig, 0x38);
NN_USB_DEFINE_FIELD32(OpRegConfig,      MaxSlotsEn, 0,   8);  // Maximum number of enabled Device Slots
//
NN_USB_DEFINE_REG_OFFSET(OpRegPortSc, 0x400);
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Ccs,        0,   1);  // Current Connect Status (CCS) – ROS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Ped,        1,   1);  // Port Enabled/Disabled (PED) – RW1CS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Oca,        3,   1);  // Over-current Active (OCA) – RO
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Pr,         4,   1);  // Port Reset (PR) – RW1S
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Pls,        5,   4);  // Port Link State (PLS) – RWS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Pp,         9,   1);  // Port Power (PP) – RWS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Speed,      10,  4);  // Port Speed (Port Speed) – ROS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Pic,        14,  2);  // Port Indicator Control (PIC) – RWS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Lws,        16,  1);  // Port Link State Write Strobe (LWS) – RW
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Csc,        17,  1);  // Connect Status Change (CSC) – RW1CS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Pec,        18,  1);  // Port Enabled/Disabled Change (PEC) – RW1CS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Wrc,        19,  1);  // Warm Port Reset Change (WRC) – RW1CS/RsvdZ
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Occ,        20,  1);  // Over-current Change (OCC) – RW1CS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Prc,        21,  1);  // Port Reset Change (PRC) – RW1CS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Plc,        22,  1);  // Port Link State Change (PLC) – RW1CS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Cec,        23,  1);  // Port Config Error Change (CEC) – RW1CS/RsvdZ
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Cas,        24,  1);  // Cold Attach Status (CAS) – RO
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Wce,        25,  1);  // Wake on Connect Enable (WCE) – RWS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Wde,        26,  1);  // Wake on Disconnect Enable (WDE) – RWS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Woe,        27,  1);  // Wake on Over-current Enable (WOE) – RWS
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Dr,         30,  1);  // Device Removableq (DR) - RO
NN_USB_DEFINE_FIELD32(OpRegPortSc,      Wpr,        31,  1);  // Warm Port Reset (WPR) – RW1S/RsvdZ
static const uint32_t OpRegPortScCommandMask = 0x80FF01FF;
//
static const size_t OpRegPortScSize = 0x10; // Size per block

NN_USB_DEFINE_REG_OFFSET(OpRegPortPmsc, 0x404);
NN_USB_DEFINE_FIELD32(OpRegPortPmsc,    L1s,        0,   3);  // L1 Status
NN_USB_DEFINE_FIELD32(OpRegPortPmsc,    Rwe,        3,   1);  // Remote Wake Enable
NN_USB_DEFINE_FIELD32(OpRegPortPmsc,    Besl,       4,   4);  // Best Effort Service Latency
NN_USB_DEFINE_FIELD32(OpRegPortPmsc,    L1ds,       8,   8);  // L1 Device Slot
NN_USB_DEFINE_FIELD32(OpRegPortPmsc,    Hle,        16,  1);  // Hardware LPM Enable
NN_USB_DEFINE_FIELD32(OpRegPortPmsc,    TestMode,   28,  4);  // Port Test Control

static const size_t OpRegPortPmscSize = 0x10; // Size per block

/**
 * @brief xHCI Host Controller Runtime Registers
 */
NN_USB_DEFINE_REG_OFFSET(RtMfindex, 0x00);
NN_USB_DEFINE_FIELD32(RtMfindex,          Index,    0,   14);  // Microframe Index
//
NN_USB_DEFINE_REG_OFFSET(RtIntReg0Iman, 0x20);
NN_USB_DEFINE_FIELD32(RtIntReg0Iman,      Ip,       0,   1);   // Interrupt Pending
NN_USB_DEFINE_FIELD32(RtIntReg0Iman,      Ie,       1,   1);   // Interrupt Enable
//
NN_USB_DEFINE_REG_OFFSET(RtIntReg0Imod, 0x24);
NN_USB_DEFINE_FIELD32(RtIntReg0Imod,      Imodi,    0,   16);  // Interrupt Moderation Interval
NN_USB_DEFINE_FIELD32(RtIntReg0Imod,      Imodc,   16,   16);  // Interrupt Moderation Counter
//
NN_USB_DEFINE_REG_OFFSET(RtIntReg0Erstsz, 0x28);
NN_USB_DEFINE_FIELD32(RtIntReg0Erstsz,    Field,    0,   16);  // number of Event Ring Segment Table entries
//
NN_USB_DEFINE_REG_OFFSET(RtIntReg0ErstbaLo, 0x30);
NN_USB_DEFINE_REG_OFFSET(RtIntReg0ErstbaHi, 0x34);
NN_USB_DEFINE_REG_OFFSET(RtIntReg0ErdpLo, 0x38);
NN_USB_DEFINE_FIELD32(RtIntReg0ErdpLo,    Ehb,      3,   1);  // Event Handler Busy
NN_USB_DEFINE_REG_OFFSET(RtIntReg0ErdpHi, 0x3C);
static const size_t RtIntRegBlockSize = 0x20; // Size per block

/**
 * @brief xHCI Host Controller Doorbell Registers
 */
NN_USB_DEFINE_FIELD32(DbReg,          Target,       0,     8);  // target of the doorbell reference
static const uint8_t DbRegTargetEndpoint0    = 1;
static const uint8_t DbRegTargetEndpoint1Out = 2;
static const uint8_t DbRegTargetEndpoint1In  = 3;
static const uint8_t DbRegTargetEndpoint2Out = 4;
static const uint8_t DbRegTargetEndpoint2In  = 5;
NN_USB_DEFINE_FIELD32(DbReg,          StreamId,    16,    16);  // Identify Stream of endpoint that doorbell reference is targeting


/**
 * @brief Transfer TRBs
 */
// qw0
NN_USB_DEFINE_FIELD64(XferTrbQw0, DirIn,           7,  1);   // control transfer - direction field
NN_USB_DEFINE_FIELD64(XferTrbQw0, WLength,         48, 16);  // control transfer - length field
// dw2
NN_USB_DEFINE_FIELD32(XferTrbDw2, TransferLength,  0,   17); // size of the data buffer referenced by the Data Buffer Pointer
NN_USB_DEFINE_FIELD32(XferTrbDw2, TdSize,          17,  5);  // provides an indicator of the number of packets remaining in the TD
NN_USB_DEFINE_FIELD32(XferTrbDw2, IntTarget,       22,  10); // defines the index of the Interrupter that will receive events generated by this TRB
// dw3
NN_USB_DEFINE_FIELD32(XferTrbDw3, C,                0,  1);  // mark the Enqueue Pointer of the Transfer ring
NN_USB_DEFINE_FIELD32(XferTrbDw3, Ent,              1,  1);  // evaluate the next TRB before saving the endpoint state
NN_USB_DEFINE_FIELD32(XferTrbDw3, Isp,              2,  1);  // Interrupt-on Short Packet
NN_USB_DEFINE_FIELD32(XferTrbDw3, Ns,               3,  1);  // xHC permitted to set No Snoop bit in Requeste Attributes of PCIe transactions
NN_USB_DEFINE_FIELD32(XferTrbDw3, Ch,               4,  1);  // associate this TRB with the next TRB on the Ring
NN_USB_DEFINE_FIELD32(XferTrbDw3, Ioc,              5,  1);  // upon completion, xHC places Xfer event TRB on event ring and asserts interrupt
NN_USB_DEFINE_FIELD32(XferTrbDw3, Idt,              6,  1);  // Immediate Data
NN_USB_DEFINE_FIELD32(XferTrbDw3, IsocTbc,          7,  2);  // Isoch Transfer - Burst Count
NN_USB_DEFINE_FIELD32(XferTrbDw3, Bei,              9,  1);  // Block Event Interrupt
NN_USB_DEFINE_FIELD32(XferTrbDw3, TrbType,          10, 6);  // Transfer TRB type ID
static const int8_t XferTrbTypeNormal      = 1;
static const int8_t XferTrbTypeSetupStage  = 2;
static const int8_t XferTrbTypeDataStage   = 3;
static const int8_t XferTrbTypeStatusStage = 4;
static const int8_t XferTrbTypeIsoch       = 5;
static const int8_t XferTrbTypeLink        = 6;
static const int8_t XferTrbTypeEventData   = 7;
static const int8_t XferTrbTypeNoOp        = 8;
NN_USB_DEFINE_FIELD32(XferTrbDw3, CtrlSetupStage,   16,  2);  // control transfer - direction, valid only for setup stage
static const int8_t XferTrbCtrlSetupStageNoData  = 0;
static const int8_t XferTrbCtrlSetupStageOutData = 2;
static const int8_t XferTrbCtrlSetupStageInData  = 3;
NN_USB_DEFINE_FIELD32(XferTrbDw3, CtrlDirIn,        16,  1);  // control transfer - direction, valid only for data and status stage
NN_USB_DEFINE_FIELD32(XferTrbDw3, IsocTlbpc,        16,  4);  // isoch transfer - Transfer Last Burst Packet Count
NN_USB_DEFINE_FIELD32(XferTrbDw3, IsocFrameId,      20, 11);  // isoch transfer - Frame ID, identifies the target 1ms frame
NN_USB_DEFINE_FIELD32(XferTrbDw3, IsocSia,          31,  1);  // isoch transfer - scheduled as soon as possible

/**
 * @brief Setup Stage TRB
 */
NN_USB_DEFINE_FIELD32(SetupStageTrbDw2, TransferLength,   0,  17);
NN_USB_DEFINE_FIELD32(SetupStageTrbDw2, IntTarget,       22,  10);

NN_USB_DEFINE_FIELD32(SetupStageTrbDw3, C,                0,  1);
NN_USB_DEFINE_FIELD32(SetupStageTrbDw3, Ioc,              5,  1);
NN_USB_DEFINE_FIELD32(SetupStageTrbDw3, Idt,              6,  1);
NN_USB_DEFINE_FIELD32(SetupStageTrbDw3, TrbType,         10,  6);
NN_USB_DEFINE_FIELD32(SetupStageTrbDw3, Trt,             16,  2);

/**
 * @brief Data Stage TRB
 */
NN_USB_DEFINE_FIELD32(DataStageTrbDw2, TransferLength,   0,  17);
NN_USB_DEFINE_FIELD32(DataStageTrbDw2, TdSize,          17,   5);
NN_USB_DEFINE_FIELD32(DataStageTrbDw2, IntTarget,       22,  10);

NN_USB_DEFINE_FIELD32(DataStageTrbDw3, C,                0,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Ent,              1,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Isp,              2,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Ns,               3,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Ch,               4,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Ioc,              5,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Idt,              6,  1);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, TrbType,         10,  6);
NN_USB_DEFINE_FIELD32(DataStageTrbDw3, Dir,             16,  1);

/**
 * @brief Status Stage TRB
 */
NN_USB_DEFINE_FIELD32(StatusStageTrbDw2, IntTarget,       22,  10);

NN_USB_DEFINE_FIELD32(StatusStageTrbDw3, C,                0,  1);
NN_USB_DEFINE_FIELD32(StatusStageTrbDw3, Ent,              1,  1);
NN_USB_DEFINE_FIELD32(StatusStageTrbDw3, Ch,               4,  1);
NN_USB_DEFINE_FIELD32(StatusStageTrbDw3, Ioc,              5,  1);
NN_USB_DEFINE_FIELD32(StatusStageTrbDw3, TrbType,         10,  6);
NN_USB_DEFINE_FIELD32(StatusStageTrbDw3, Dir,             16,  1);

/**
 * @brief Isoch TRB
 */
NN_USB_DEFINE_FIELD32(IsochTrbDw3, C,                0,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Ent,              1,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Isp,              2,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Ns,               3,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Ch,               4,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Ioc,              5,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Idt,              6,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Tbc,              7,  2);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Bei,              9,  1);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, TrbType,         10,  6);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Tlbpc,           16,  4);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, FrameId,         20, 11);
NN_USB_DEFINE_FIELD32(IsochTrbDw3, Sia,             31,  1);

/**
 * @brief Event TRBs
 */
// dw0
NN_USB_DEFINE_FIELD32(EventTrbDw0, PortId,         24,    8); // port status - Root port number that generated this event
NN_USB_DEFINE_FIELD32(EventTrbDw0, DbReason,        0,    5); // doorbell - value written to DB Target field
// qw0
NN_USB_DEFINE_FIELD64(EventTrbQw0, DevNotifyDataHi, 0,   32); // device - Upper Notification Data
NN_USB_DEFINE_FIELD64(EventTrbQw0, DevNotifyType,   36,   4); // device - Notification type
NN_USB_DEFINE_FIELD64(EventTrbQw0, DevNotifyDataLo, 40,  24); // device - Lower Notification Data
// dw2
NN_USB_DEFINE_FIELD32(EventTrbDw2, TransferLength,  0,   24); // transfer - reflect the residual number of bytes not transferred
NN_USB_DEFINE_FIELD32(EventTrbDw2, CompletionCode,  24,   8); // completion status that can be identified by a TRB
// dw3
NN_USB_DEFINE_FIELD32(EventTrbDw3, C,                0,   1); // mark the Dequeue Pointer of an Event Ring
NN_USB_DEFINE_FIELD32(EventTrbDw3, Ed,               1,   1); // TRB Pointer field contains a 64-bit value provided by the Event Data
NN_USB_DEFINE_FIELD32(EventTrbDw3, TrbType,         10,   6); // Event TRB type ID
static const int8_t EventTrbTypeTransfer           = 32;
static const int8_t EventTrbTypeCommandCompletion  = 33;
static const int8_t EventTrbTypePortStatusChange   = 34;
static const int8_t EventTrbTypeBandwidthRequest   = 35;
static const int8_t EventTrbTypeDoorbell           = 36;
static const int8_t EventTrbTypeHostController     = 37;
static const int8_t EventTrbTypeDeviceNotification = 38;
static const int8_t EventTrbTypeMfindexWrap        = 39;
NN_USB_DEFINE_FIELD32(EventTrbDw3, EndpointId,      16,   5); // ID of the Endpoint that generated the event
NN_USB_DEFINE_FIELD32(EventTrbDw3, VfId,            16,   8); // command - ID of the Virtual Function that generated the event
NN_USB_DEFINE_FIELD32(EventTrbDw3, SlotId,          24,   8); // ID of the Device Slot that generated the event

// Completion codes relevant for Event TRBs
static const uint8_t CompletionCodeInvalid                        = 0;
static const uint8_t CompletionCodeSuccess                        = 1;
static const uint8_t CompletionCodeDataBufferError                = 2;
static const uint8_t CompletionCodeBabbleError                    = 3;
static const uint8_t CompletionCodeTransactError                  = 4;
static const uint8_t CompletionCodeTrbError                       = 5;
static const uint8_t CompletionCodeStallError                     = 6;
static const uint8_t CompletionCodeResourceError                  = 7;
static const uint8_t CompletionCodeBandwidthError                 = 8;
static const uint8_t CompletionCodeNoSlotAvailableError           = 9;
static const uint8_t CompletionCodeStreamTypeError                = 10;
static const uint8_t CompletionCodeSlotNotEnabledError            = 11;
static const uint8_t CompletionCodeEndpointNotEnabledError        = 12;
static const uint8_t CompletionCodeShortPacket                    = 13;
static const uint8_t CompletionCodeRingUnderrun                   = 14;
static const uint8_t CompletionCodeRingOverrun                    = 15;
static const uint8_t CompletionCodeVfEventRingFullError           = 16;
static const uint8_t CompletionCodeParameterError                 = 17;
static const uint8_t CompletionCodeBandwidthOverrunError          = 18;
static const uint8_t CompletionCodeContextStateError              = 19;
static const uint8_t CompletionCodeNoPingResponseError            = 20;
static const uint8_t CompletionCodeEventRingFullError             = 21;
static const uint8_t CompletionCodeIncompatibleDeviceError        = 22;
static const uint8_t CompletionCodeMissedServiceError             = 23;
static const uint8_t CompletionCodeCommandRingStopped             = 24;
static const uint8_t CompletionCodeCommandAborted                 = 25;
static const uint8_t CompletionCodeStopped                        = 26;
static const uint8_t CompletionCodeStoppedLengthInvalid           = 27;
static const uint8_t CompletionCodeStoppedShortPacket             = 28;
static const uint8_t CompletionCodeMaxExitLatencyTooLargeError    = 29;
// completion code 30 is reserved
static const uint8_t CompletionCodeIsochBufferOverrun             = 31;
static const uint8_t CompletionCodeEventLostError                 = 32;
static const uint8_t CompletionCodeUndefinedError                 = 33;
static const uint8_t CompletionCodeInvalidStreamIdError           = 34;
static const uint8_t CompletionCodeSecondaryBandwidthError        = 35;
static const uint8_t CompletionCodeSplitTransactionError          = 36;

/**
 * @brief Command TRBs
 */
// dw0
NN_USB_DEFINE_FIELD32(CommandTrbDw0, Dcs,           0,    1); // Set TR Dequeue Pointer - identifies value of xHC CCS flag
NN_USB_DEFINE_FIELD32(CommandTrbDw0, Sct,           1,    3); // Set TR Dequeue Pointer - Stream Context Type
NN_USB_DEFINE_FIELD32(CommandTrbDw0, NewTrDeqPtrLo, 4,   28); // Set TR Dequeue Pointer - New TR Dequeue Pointer
NN_USB_DEFINE_FIELD32(CommandTrbDw0, Type,          0,    5); // force header - Packet Type, per 8.3.1.2 of USB3 specification
// dw2
NN_USB_DEFINE_FIELD32(CommandTrbDw2, StreamId,     16,    16); // Set TR Dequeue Pointer - Stream Context new TR Dequeue Pointer
NN_USB_DEFINE_FIELD32(CommandTrbDw2, VfIntTarget,  22,    10); // Force Event - VF Interrupter Target
// dw3
NN_USB_DEFINE_FIELD32(CommandTrbDw3, C,             0,     1); // mark the Enqueue Pointer of a Command Ring
NN_USB_DEFINE_FIELD32(CommandTrbDw3, Bsr,           9,     1); // Address Device - Block Set Address Request
NN_USB_DEFINE_FIELD32(CommandTrbDw3, Dc,            9,     1); // Configure Endpoint - deconfigure the Device Slot
NN_USB_DEFINE_FIELD32(CommandTrbDw3, Tsp,           9,     1); // Transfer State Preserve
NN_USB_DEFINE_FIELD32(CommandTrbDw3, TrbType,      10,     6); // Event TRB type ID
static const int8_t CommandTrbTypeLink                     = 6;
static const int8_t CommandTrbTypeEnableSlot               = 9;
static const int8_t CommandTrbTypeDisableSlot              = 10;
static const int8_t CommandTrbTypeAddressDevice            = 11;
static const int8_t CommandTrbTypeConfigureEndpoint        = 12;
static const int8_t CommandTrbTypeEvaluateContext          = 13;
static const int8_t CommandTrbTypeResetEndpoint            = 14;
static const int8_t CommandTrbTypeStopEndpoint             = 15;
static const int8_t CommandTrbTypeSetTrDequeuePointer      = 16;
static const int8_t CommandTrbTypeResetDevice              = 17;
static const int8_t CommandTrbTypeForceEvent               = 18;
static const int8_t CommandTrbTypeNegotiateBandwidth       = 19;
static const int8_t CommandTrbTypeSetLatencyToleranceValue = 20;
static const int8_t CommandTrbTypeGetPortBandwidth         = 21;
static const int8_t CommandTrbTypeForceHeader              = 22;
static const int8_t CommandTrbTypeNoOp                     = 23;
NN_USB_DEFINE_FIELD32(CommandTrbDw3, SlotType,      16,    5); // Enable Slot - identifies the type of Slot enabled by this command
NN_USB_DEFINE_FIELD32(CommandTrbDw3, EndpointId,    16,    5); // Stop Endpoint - DCI of the endpoint to be stopped
NN_USB_DEFINE_FIELD32(CommandTrbDw3, VfId,          16,    8); // Force Event Command - ID of the Virtual Function who’s Event Ring will receive Event
NN_USB_DEFINE_FIELD32(CommandTrbDw3, Belt,          16,   12); // Negotiate Bandwidth - Best Effort Latency Tolerance Value
NN_USB_DEFINE_FIELD32(CommandTrbDw3, DevSpeed,      16,    4); // Get Port Bandwidth - bus speed of interest
NN_USB_DEFINE_FIELD32(CommandTrbDw3, Sp,            23,    1); // Stop Endpoint - Stop Endpoint Command issued to suspended endpoint
NN_USB_DEFINE_FIELD32(CommandTrbDw3, SlotId,        24,    8); // ID of the Device Slot to which command is issued
NN_USB_DEFINE_FIELD32(CommandTrbDw3, HubSlotId,     24,    8); // Get Port Bandwidth - identifies the hub ports
NN_USB_DEFINE_FIELD32(CommandTrbDw3, RootHubPort,   24,    8); // Force Header - identifies root hub port to which packet shall be issued

/**
 * @brief Link TRB
 */
// dw2
NN_USB_DEFINE_FIELD32(LinkTrbDw2, IntTarget,       22,   10); // ndex of the Interrupter that will receive Transfer Events generated by this TRB
// dw3
NN_USB_DEFINE_FIELD32(LinkTrbDw3, C,                0,   1);  // mark the Enqueue Pointer location of a Transfer or Command Ring
NN_USB_DEFINE_FIELD32(LinkTrbDw3, Tc,               1,   1);  // xHC shall toggle its interpretation of the Cycle bit
NN_USB_DEFINE_FIELD32(LinkTrbDw3, Ch,               4,   1);  // associate this TRB with the next TRB on the Ring
NN_USB_DEFINE_FIELD32(LinkTrbDw3, Ioc,              5,   1);  // upon completion, xHC places Xfer event TRB on event ring and asserts interrupt
NN_USB_DEFINE_FIELD32(LinkTrbDw3, TrbType,         10,   6);  // Link TRB type ID

/**
 * @brief Event Data TRB
 */
// dw2
NN_USB_DEFINE_FIELD32(EventDataTrbDw2, IntTarget,    22,   10); // Index of the Interrupter that will receive Transfer Events generated by this TRB
NN_USB_DEFINE_FIELD32(EventDataTrbDw2, Ent,           0,   1);  // evaluate the next TRB before saving the endpoint state
NN_USB_DEFINE_FIELD32(EventDataTrbDw2, Ch,            4,   1);  // associate this TRB with the next TRB on the Ring
NN_USB_DEFINE_FIELD32(EventDataTrbDw2, Ioc,           5,   1);  // upon completion, xHC places Xfer event TRB on event ring and asserts interrupt
NN_USB_DEFINE_FIELD32(EventDataTrbDw2, Bei,           9,   1);  // Transfer Event generated by IOC shall not assert an interrupt
NN_USB_DEFINE_FIELD32(EventDataTrbDw2, TrbType,      10,   6);  // Event Data TRB type ID


/**
 * @brief Device Context
 *        Contains slot context followed by endpoint contexts.
 *        Host controller hardware writes to this context.
 */
static const size_t DeviceCtxSize = 0x1000;
NN_USB_DEFINE_REG_OFFSET(DeviceCtxSlot,          0x00);
NN_USB_DEFINE_REG_OFFSET(DeviceCtxEndpoint0,     0x40);
NN_USB_DEFINE_REG_OFFSET(DeviceCtxEndpoint1Out,  0x80);
NN_USB_DEFINE_REG_OFFSET(DeviceCtxEndpoint1In,   0xC0);

/**
 * @brief Slot Context
 */
static const size_t SlotCtxSize = 0x40;
// dw0
NN_USB_DEFINE_FIELD32(SlotCtxDw0, RouteString,        0,   20); // Used by hubs to route packets to the correct downstream port
NN_USB_DEFINE_FIELD32(SlotCtxDw0, Speed,             20,    4); // Indicates the speed of the device
NN_USB_DEFINE_FIELD32(SlotCtxDw0, Mtt,               25,    1); // Multi-TT set to '1' by software if this is a High-speed hub
NN_USB_DEFINE_FIELD32(SlotCtxDw0, Hub,               26,    1); // Set to '1' by software if this device is a USB hub, or '0' if it is a USB function
NN_USB_DEFINE_FIELD32(SlotCtxDw0, ContextEntries,    27,    5); // Identifies the index of the last valid Endpoint Context, valid range of 1-31
// dw1
NN_USB_DEFINE_FIELD32(SlotCtxDw1, MaxExitLatency,     0,   16); // worst case time to wake links in the path, given PM settings
NN_USB_DEFINE_FIELD32(SlotCtxDw1, RootHubPortNumber, 16,    8); // Identifies Root Hub Port Number to access the USB device
NN_USB_DEFINE_FIELD32(SlotCtxDw1, NumberofPorts,     24,    8); // If hub indicates num downstream facing ports, else zero
// dw2
NN_USB_DEFINE_FIELD32(SlotCtxDw2, TtHubSlotId,        0,    8); // If low/full device via HS hub, Slot ID of parent High-speed hub
NN_USB_DEFINE_FIELD32(SlotCtxDw2, TtPortNumber,       8,    8); // If low/full device via HS hub, number of the downstream port of parent hub
NN_USB_DEFINE_FIELD32(SlotCtxDw2, TtThinkTime,       16,    2); // If HS hub, time of TT of the hub required to proceed to next full/low speed txn
NN_USB_DEFINE_FIELD32(SlotCtxDw2, InterrupterTarget, 22,   10); // Index of Interrupter that receive Bandwidth Request Events and Device Notification Events
// dw3
NN_USB_DEFINE_FIELD32(SlotCtxDw3, UsbDeviceAddress,   0,    8); // Address assigned to device by the xHC, set upon successful completion of Set Address Command
NN_USB_DEFINE_FIELD32(SlotCtxDw3, SlotState,         27,    5); // updated by the xHC when a Device Slot transitions from one state to another
static const uint8_t SlotCtxDw3SlotStateEnable     = 0;
static const uint8_t SlotCtxDw3SlotStateDefault    = 1;
static const uint8_t SlotCtxDw3SlotStateAddressed  = 2;
static const uint8_t SlotCtxDw3SlotStateConfigured = 3;
// dw4-dw7 are reserved

/**
 * @brief Endpoint Context
 */
static const size_t EndpointCtxSize = 0x40;
// dw0
NN_USB_DEFINE_FIELD32(EndpointCtxDw0, State,        0,       3); // Current operational state of the endpoint
static const uint8_t EndpointCtxDw0StateDisabled = 0;
static const uint8_t EndpointCtxDw0StateRunning  = 1;
static const uint8_t EndpointCtxDw0StateHalted   = 2;
static const uint8_t EndpointCtxDw0StateStopped  = 3;
static const uint8_t EndpointCtxDw0StateError    = 4;
NN_USB_DEFINE_FIELD32(EndpointCtxDw0,  Mult,                8,       2); // Maximum bursts within Interval endpoint supports, valid range 0-2
NN_USB_DEFINE_FIELD32(EndpointCtxDw0,  MaxPStreams,        10,       5); // Maximum number of Primary Stream IDs this endpoint supports
NN_USB_DEFINE_FIELD32(EndpointCtxDw0,  Lsa,                15,       1); // Secondary Stream Arrays, 1=disable
NN_USB_DEFINE_FIELD32(EndpointCtxDw0,  Interval,           16,       8); // Period between consecutive requests, 125 μs * 2^Interval
NN_USB_DEFINE_FIELD32(EndpointCtxDw0,  MaxEsitPayloadHi,   24,       8); // If LEC = '1', indicates high order 8 bits of Max ESIT Payload value
// dw1
NN_USB_DEFINE_FIELD32(EndpointCtxDw1,  Cerr,                1,       2); // 2-bit down count of consecutive Bus Errors allowed while executing a TD
NN_USB_DEFINE_FIELD32(EndpointCtxDw1,  EpType,              3,       3); // Endpoint validity and type
static const uint8_t EndpointCtxDw1EpTypeNotValid  = 0;
static const uint8_t EndpointCtxDw1EpTypeIsochOut  = 1;
static const uint8_t EndpointCtxDw1EpTypeBulkOut   = 2;
static const uint8_t EndpointCtxDw1EpTypeIntOut    = 3;
static const uint8_t EndpointCtxDw1EpTypeControl   = 4;
static const uint8_t EndpointCtxDw1EpTypeIsochIn   = 5;
static const uint8_t EndpointCtxDw1EpTypeBulkIn    = 6;
static const uint8_t EndpointCtxDw1EpTypeIntIn     = 7;
NN_USB_DEFINE_FIELD32(EndpointCtxDw1,  Hid,                 7,       1); // 1 will disable the Host Initiated Stream selection feature
NN_USB_DEFINE_FIELD32(EndpointCtxDw1,  MaxBurstSize,        8,       8); // maximum xtns scheduled per opportunity
NN_USB_DEFINE_FIELD32(EndpointCtxDw1,  MaxPacketSize,      16,      16); // maximum packet size endpoint is capable of sending or receiving
// qw1 (dw2)
NN_USB_DEFINE_FIELD64(EndpointCtxQw1,  Dcs,                 0,       1); // value of Consumer Cycle State (CCS) flag for TRB referenced by TR Dequeue Pointer
NN_USB_DEFINE_FIELD64(EndpointCtxQw1,  TrDequeuePointer,    4,      60); //
// dw4
NN_USB_DEFINE_FIELD64(EndpointCtxDw4,  AverageTrbLength,    0,      16); // average Length of the TRBs executed by this endpoint
NN_USB_DEFINE_FIELD64(EndpointCtxDw4,  MaxEsitPayloadLo,   16,      16); // low order 16 bits of the Max ESIT Payload
// dw5-dw7 are reserved

/**
 * @brief Input Context
 */
static const size_t InputCtxSize = 0x40 * 33;
NN_USB_DEFINE_REG_OFFSET(InputCtxControl,      0x00);
NN_USB_DEFINE_REG_OFFSET(InputCtxSlot,         0x40);
NN_USB_DEFINE_REG_OFFSET(InputCtxEndpoint0,    0x80);
NN_USB_DEFINE_REG_OFFSET(InputCtxEndpoint1Out, 0xC0);
NN_USB_DEFINE_REG_OFFSET(InputCtxEndpoint1In,  0x100);

/*
 * @brief Output Context
 */
static const size_t OutputCtxSize = 0x40 * 32;
NN_USB_DEFINE_REG_OFFSET(OutputCtxSlot,         0x00);
NN_USB_DEFINE_REG_OFFSET(OutputCtxEndpoint0,    0x40);
NN_USB_DEFINE_REG_OFFSET(OutputCtxEndpoint1Out, 0x80);
NN_USB_DEFINE_REG_OFFSET(OutputCtxEndpoint1In,  0xC0);


// Input Control Context
static const size_t InputCtxControlSize = 0x40;
// dw0
NN_USB_DEFINE_FIELD32(InputControlCtxDw0, DropEndpoint1Out,     2,       1);
NN_USB_DEFINE_FIELD32(InputControlCtxDw0, DropEndpoint1In,      3,       1);
// dw1
NN_USB_DEFINE_FIELD32(InputControlCtxDw1, AddSlot,              0,       1);
NN_USB_DEFINE_FIELD32(InputControlCtxDw1, AddEndpoint0,         1,       1);
NN_USB_DEFINE_FIELD32(InputControlCtxDw1, AddEndpoint1Out,      2,       1);
NN_USB_DEFINE_FIELD32(InputControlCtxDw1, AddEndpoint1In,       3,       1);
// dw7
NN_USB_DEFINE_FIELD32(InputControlCtxDw7, ConfigurationValue,   0,       8);
NN_USB_DEFINE_FIELD32(InputControlCtxDw7, InterfaceNumber,      8,       8);
NN_USB_DEFINE_FIELD32(InputControlCtxDw7, AlternateSetting,    16,       8);


//////////////////////////////////////////////////////////////////////////
//  XHCI Standard Types
//////////////////////////////////////////////////////////////////////////
/**
 * @brief Transfer Request Block (TRB)
 *        The contents and format of fields vary greatly depending upon type
 */
union DmaTrb
{
    volatile uint64_t qw[2];
    volatile uint32_t dw[4];
    volatile uint8_t   b[16];
};
static const DmaTrb ZeroedDmaTrb;

/**
 * @brief Event Ring Segment Table (ERST)
 */
struct DmaErst
{
    uint64_t ringSegmentBaseAddress;  // Base address of TRB ring
    uint32_t ringSegmentSize;         // Number of TRBs
    uint32_t reserved;
};

/**
 * @brief Slot Context
 */
union DmaSlotCtx
{
    volatile uint64_t qw[4];
    volatile uint32_t dw[8];
    volatile uint8_t   b[32];
};

/**
 * @brief Endpoint Context
 */
union DmaEndpointContext
{
    volatile uint64_t qw[4];
    volatile uint32_t dw[8];
    volatile uint8_t   b[32];
};

/**
 * @brief Input Control Context
 */
union DmaInputControlContext
{
    volatile uint64_t qw[4];
    volatile uint32_t dw[8];
    volatile uint8_t   b[32];
};


//////////////////////////////////////////////////////////////////////////
// Private Driver Types
//////////////////////////////////////////////////////////////////////////

// General purpose structure for describing a DMA buffer internal to this driver
struct DmaBuffer
{
    const char*                   pDebugName;
    size_t                        size;
    nn::dd::DeviceVirtualAddress  ioVa;
    union
    {
        uintptr_t   uintptr;
        void*       pVoid;
        uint32_t*   pU64;
        uint32_t*   pU32;
        uint8_t*    pU8;
        DmaTrb*     pTrb;
    }u;
};

// See section 4.5.3 of XHCI spec
enum DeviceContextState
{
    DeviceContextState_Disabled,
    DeviceContextState_Enabled,
    DeviceContextState_Default,
    DeviceContextState_Addressed,
    DeviceContextState_Configured,
};

enum DeviceContextEvent
{
    DeviceContextEvent_Invalid,
    DeviceContextEvent_SlotEnableCompletion,
    DeviceContextEvent_SlotDisableCompletion,
    DeviceContextEvent_SetDefaultAddressCompletion,
    DeviceContextEvent_SetAddressCompletion,
    DeviceContextEvent_UpdateCompletion,
    DeviceContextEvent_UpdateEp0Completion,
    DeviceContextEvent_ResetEndpointHaltCompletion,
    DeviceContextEvent_ResetEndpointDequeueCompletion,
    DeviceContextEvent_OpenEndpointCompletion,
    DeviceContextEvent_CloseEndpointCompletion,
};

struct XhciEndpoint;

struct DeviceContext : public HostControllerDriverDeviceContext
{
    DeviceContextState            state;
    DmaBuffer                     outputBuffer;  // This is the main device context, per section 6.2.1 of XHCI spec
    DmaBuffer                     inputBuffer;   // This is the input context, per section 6.2.5 of XHCI spec
    XhciEndpoint*                 pControlEndpoint;
    XhciEndpoint*                 pInEndpoints[15];
    XhciEndpoint*                 pOutEndpoints[15];
};

struct MetaTrb
{
    UsbRequestBlock* pUrb;
    int32_t          sequence;
    uint32_t         bufSize;         // buffer size represented by this whole trb
    uint32_t         remainingSize;   // size of buffers beyond this TRB
};

// Transfer ring handling
struct TransferRing
{
    DmaBuffer                     dmaTrbList;
    MetaTrb*                      pMetaTrbList;
    uint8_t                       ccs;
    int32_t                       freeIndex, pendIndex;
    int32_t                       pendCount, totalCount;
};

struct XhciEndpoint
{
    DeviceContext*                pDeviceContext;
    HostControllerDriverEndpoint* pHcEp;
    DmaEndpointContext*           pDmaInputContext;
    DmaEndpointContext*           pDmaOutputContext;
    TransferRing                  transferRing;
    bool                          isEnabled;
    uint8_t                       dci;
};

// Command ring handling
struct Command;
typedef void (XhciDriver::*DoCommandCallback)(Command *pCmd);
struct Command
{
    DoCommandCallback callback;
    void*             data;
    DmaTrb*           pCommandTrb;
    DmaTrb*           pEventTrb;
    Result            status;
};

//////////////////////////////////////////////////////////////////////////
// Private Data
//////////////////////////////////////////////////////////////////////////
    detail::RegisterBlock&        CapRegister;
    detail::RegisterBlock         OpRegister;
    detail::RegisterBlock         DbRegister;
    detail::RegisterBlock         RtRegister;
    size_t                        m_PageSize;
    int32_t                       m_MaxDeviceSlotCount;
    uint16_t                      m_ControllerVersion;

    // Device context
    nn::dd::DeviceVirtualAddress* m_pDmaDeviceContextArray;

    // Scratch pad
    uint32_t                      m_ScratchPadBufferCount;
    nn::dd::DeviceVirtualAddress* m_pDmaScratchPadBufferArray;

    // Event Ring Segment Table
    int32_t                       m_EventRingSegmentTableEntryCount;
    DmaErst*                      m_pDmaEventRingSegmentTable;
    DmaTrb*                       m_pDmaEvents;
    nn::dd::DeviceVirtualAddress  m_DmaEventsIoVa;
    uint8_t                       m_EventCcs;
    int32_t                       m_EventIndex;

    // Command Ring
    Command                       m_CommandRing[MaxCommandTrbs];
    int32_t                       m_CommandFreeIndex;
    int32_t                       m_CommandPendIndex;
    uint8_t                       m_CommandPcs;
    int32_t                       m_CommandPendCount;
    DmaTrb*                       m_pDmaCommandRing;
    nn::dd::DeviceVirtualAddress  m_DmaCommandRingIoVa;

    bool                          m_IsContext64b;
    DeviceContext*                m_pDevices[LimitMaxDeviceSlotCount];

    bool                          m_IsControllerStall;

//////////////////////////////////////////////////////////////////////////
// Base class methods
//////////////////////////////////////////////////////////////////////////
    Result CreateDeviceContextAsync(HubPortNumber *portHierarchy, UsbDeviceSpeed speed,
                                    uint8_t ttDevice, HubPortNumber ttPort,
                                    CreateDeviceContextCallback callback, void* context);
    Result DestroyDeviceContextAsync(HostControllerDriverDeviceContext* pDeviceContext,
                                     DestroyDeviceContextCallback callback, void* context);
    Result AddressDeviceAsync(HostControllerDriverDeviceContext* pDeviceContext,
                              AddressDeviceCallback callback, void* context);
    void   UpdateDeviceContext(HostControllerDriverDeviceContext* pDeviceContext);
    Result UpdateEp0Async(HostControllerDriverDeviceContext* pDeviceContext, uint8_t maxPacketSize0,
                          UpdateEp0Callback callback, void *context);
    Result OpenEpAsync(HostControllerDriverEndpoint *pHcEp);
    Result ResetEpAsync(HostControllerDriverEndpoint *pHcEp);
    Result CloseEpAsync(HostControllerDriverEndpoint *pHcEp);
    Result SubmitUrbAsync(UsbRequestBlock *pUrb);
    Result GetRootHubPortStatus(HubPortNumber portNumber, UsbHubPortStatus *pPs);
    Result ClearRootHubPortStatus(HubPortNumber portNumber, uint16_t wPortStatus);
    Result SetRootHubPortAttribute(HubPortNumber portNumber, PortAttribute attribute, bool isTrue);
    Result ResetRootHub();
    UsbDeviceSpeed GetPortSpeed(HubPortNumber portNumber);
    void   PollEventRing();
    void   HandleTransferCompletionEvent(DmaTrb* pEventTrb);
    void   HandleCommandCompletionEvent(DmaTrb* pEventTrb);
    void   FlushCommandRing();
    void   Isr();
    void   OnControllerStall();
    void   SetTestMode(uint32_t port, TestMode mode);

//////////////////////////////////////////////////////////////////////////
// Internal methods
//////////////////////////////////////////////////////////////////////////
    Result InitializeDmaBuffer(DmaBuffer* pBuf, size_t size, const char* debugName);
    Result FinalizeDmaBuffer(DmaBuffer* pBuf);

    Result InitializeDriverMemory();
    Result FinalizeDriverMemory();

    Result InitializeEndpoint0(DeviceContext* pXdc);
    Result FinalizeEndpoint0(DeviceContext* pXdc);
    Result FinalizeEndpoint(DeviceContext* pXdc, XhciEndpoint* pXep);

    Result StartController();
    Result HaltController();
    Result ResetController();

    Result DestroyDeviceContext(HostControllerDriverDeviceContext* pDeviceContext);

    // xHCI Command Interface
    Result DoEnableSlot(DeviceContext *pXdc);
    Result DoDisableSlot(int slotId);
    Result DoAddressDevice(int slotId, nn::dd::DeviceVirtualAddress inputContext, int bsr);
    Result DoConfigureEndpoint(int slotId, nn::dd::DeviceVirtualAddress inputContext, int dc, XhciEndpoint *pXep);
    Result DoEvaluateContext(int slotId, nn::dd::DeviceVirtualAddress inputContext);
    Result DoResetEndpoint(int slotId, int dci, int tsp);
    Result DoStopEndpoint(int slotId, int dci, int sp);
    Result DoSetTrDequeuePointer(int slotId, int dci, nn::dd::DeviceVirtualAddress dequeuePointer, int dcs, int streamId, int sct);
    Result DoCommandAsync(DmaTrb& commandTrb, DoCommandCallback callback, void *data);
    void   OnEnableSlotCompletion(Command *pCmd);
    void   OnDisableSlotCompletion(Command *pCmd);
    void   OnAddressDeviceCompletion(Command *pCmd);
    void   OnConfigureEndpointCompletion(Command *pCmd);
    void   OnEvaluateContextCompletion(Command *pCmd);
    void   OnResetEndpointCompletion(Command *pCmd);
    void   OnStopEndpointCompletion(Command *pCmd);
    void   OnSetTrDequeuePointerCompletion(Command *pCmd);

    DmaEndpointContext* GetEndpointInputContext(DeviceContext* pDeviceContext,
                                                UsbEndpointDirection direction,
                                                uint8_t epNumber);
    DmaEndpointContext* GetEndpointOutputContext(DeviceContext* pDeviceContext,
                                                 UsbEndpointDirection direction,
                                                 uint8_t epNumber);
    DeviceContext* GetXhciDeviceContext(int32_t slotId);

    XhciEndpoint* GetXhciEndpoint(DeviceContext* pXdc, UsbEndpointDirection epDir,
                                  UsbEndpointType epType, EndpointNumber epNumber, bool isOperationalRequired);
    XhciEndpoint* GetXhciEndpoint(DeviceContext* pXdc, uint8_t deviceContextIndex, bool isOperationalRequired);
    Result SetXhciEndpoint(DeviceContext* pXdc, UsbEndpointDirection epDir,
                           UsbEndpointType epType, EndpointNumber epNumber,
                           XhciEndpoint* pEp);

    Result InitializeTransferRing(TransferRing* pTr, int32_t trbCount, uint8_t cycleState);
    void   ResetTransferRing(TransferRing* pTr, uint8_t cycleState);
    Result FinalizeTransferRing(TransferRing* pTr);

    Result OpenTd(TransferRing *pTr);
    int    AcquireTrb(TransferRing *pTr);
    Result CloseTd(TransferRing *pTr);

    Result HandleControlUrbSubmit(UsbRequestBlock *pUrb);
    Result HandleNormalUrbSubmit(UsbRequestBlock *pUrb);
    Result HandleIsochUrbSubmit(UsbRequestBlock *pUrb);

    // helpers
    const char* GetTrbName(int trbType);
    const char* GetCompletionString(int completionCode);
    Result      CheckTransferTrbCompletionCode(uint8_t completionCode);
    Result      CheckCommandTrbCompletionCode(uint8_t completionCode);

    uint32_t GetEndpointInterval(HostControllerDriverEndpoint *pHcEp);
    uint32_t GetEndpointMult(HostControllerDriverEndpoint *pHcEp);
    uint8_t  GetEndpointCtxDw1EpType(HostEndpoint *pHep);
    int32_t  GetSlotContextEntryCount(DeviceContext* pXdc);
    uint32_t MakeInputContextControlMask(HostEndpoint *pHep);

    uint32_t GetCurrentFrameId();
    uint32_t CalculateTrbCount(uint32_t dataSize);
    uint32_t CalculateTdSize(UsbRequestBlock *pUrb, uint32_t xferNum, uint32_t n);
    uint32_t GetMaxPacketSize(XhciEndpoint *pXep);
    uint32_t GetMaxBurstSize(XhciEndpoint *pXep);
    uint32_t CalculateTdpc(XhciEndpoint *pXep, uint32_t tdXferSize);
    uint32_t GetAverageTrbLength(XhciEndpoint *pXep);
    uint32_t GetMaxEsitPayload(XhciEndpoint *pXep);
    uint32_t CalculateTbc(XhciEndpoint *pXep, uint32_t tdXferSize);
    uint32_t CalculateTlbpc(XhciEndpoint *pXep, uint32_t tdXferSize);
};


} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
