﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {

//--------------------------------------------------------------------------
//  DeviceManager::ManagedDevice implementation
//--------------------------------------------------------------------------
Result DeviceManager::ManagedDevice::Initialize(DeviceManager *pDm, bool isHub, bool isRootHub)
{
    m_pDm       = pDm;
    m_IsHub     = isHub;
    m_IsRootHub = isRootHub;

    // Update rolling UID in manager
    m_pDm->m_RollingDeviceUid = ((m_pDm->m_RollingDeviceUid + 1) == 0) ?
        1 : (m_pDm->m_RollingDeviceUid + 1);

    // Assign
    m_Uid = m_pDm->m_RollingDeviceUid;

    // Add to appropriate device list
    m_IsRootHub ? m_pDm->m_RootHubList.push_back(*this) :
                  m_pDm->m_ExternalDeviceList.push_back(*this);

    m_pDm->m_TotalDeviceCount++;

    NN_USB_LOG_INFO("DeviceManager: TotalDeviceCount increased to %d.\n", m_pDm->m_TotalDeviceCount);

    return ResultSuccess();
}

Result DeviceManager::ManagedDevice::Finalize()
{
    Result result = ResultSuccess();

    if (!m_IsHub && (m_pIfArray != nullptr))
    {
        // destroy associated managed interfaces
        for (uint32_t ifIndex = 0; ifIndex < HsLimitMaxInterfacesPerConfigurationCount; ifIndex++)
        {
            ManagedInterface *pMi;
            DeviceInterface *pDif = m_pIfArray + ifIndex;
            if ((pMi = pDif->pManagedInterface) != nullptr)
            {
                NN_USB_BREAK_UPON_ERROR(pMi->Finalize());
                delete pMi;
                pDif->pManagedInterface = nullptr;
            }
        }
    }

    // Remove from appropriate device list
    m_IsRootHub ? m_pDm->m_RootHubList.erase(m_pDm->m_RootHubList.iterator_to(*this)) :
                  m_pDm->m_ExternalDeviceList.erase(m_pDm->m_ExternalDeviceList.iterator_to(*this));

    m_pDm->m_TotalDeviceCount--;

    NN_USB_LOG_INFO("DeviceManager: TotalDeviceCount decreased to %d.\n", m_pDm->m_TotalDeviceCount);

    return result;
}

Result DeviceManager::ManagedDevice::Offer(DeviceInterface *ifArray)
{
    Result result = ResultSuccess();

    if ((ifArray == nullptr) || m_IsHub)
    {
        // Nothing to do
        return result;
    }

    m_pIfArray = ifArray;
    // Create managed interfaces
    for (int32_t ifIndex = 0; ifIndex < HsLimitMaxInterfacesPerConfigurationCount; ifIndex++)
    {
        DeviceInterface *pDif = m_pIfArray + ifIndex;
        if (pDif->pActAltSetting != nullptr)
        {
            ManagedInterface *pMi;
            NN_USB_BREAK_UPON_MEM_ALLOC_FAIL(pMi = new ManagedInterface(m_pHs, m_pDm, this, pDif));
            if ((result = pMi->Initialize()).IsSuccess())
            {
                pDif->pManagedInterface = pMi;
            }
            else
            {
                delete pMi;
                NN_USB_BREAK_UPON_ERROR(result);
            }
        }
    }

    return result;
}

Result DeviceManager::ManagedDevice::Revoke()
{
    Result result = ResultSuccess();

    if ((m_pIfArray == nullptr) || m_IsHub)
    {
        // Nothing to do
        return result;
    }

    // Create managed interfaces
    for (int32_t ifIndex = 0; ifIndex < HsLimitMaxInterfacesPerConfigurationCount; ifIndex++)
    {
        ManagedInterface *pMi;
        DeviceInterface *pDif = m_pIfArray + ifIndex;
        if ((pMi = pDif->pManagedInterface) != nullptr)
        {
            pMi->Fsm::QueueFsmEvent(ManagedInterface::Event_Revoke);
        }
    }

    return result;
}

void DeviceManager::ManagedDevice::HostEndpointCreateConfirm(HostEndpoint *pHep, Result status)
{
    ManagedInterface *pMif = reinterpret_cast<ManagedInterface *>(pHep->adminContext);
    pMif->QueueFsmEvent(ManagedInterface::Event_EndpointCreateConfirm, status, pHep);
}

void DeviceManager::ManagedDevice::HostEndpointDestroyConfirm(HostEndpoint *pHep)
{
    ManagedInterface *pMif = reinterpret_cast<ManagedInterface *>(pHep->adminContext);
    pMif->QueueFsmEvent(ManagedInterface::Event_EndpointDestroyConfirm, pHep);
}

DeviceUid DeviceManager::ManagedDevice::GetUid()
{
    return m_Uid;
}

bool DeviceManager::ManagedDevice::IsHub()
{
    return m_IsHub;
}

bool DeviceManager::ManagedDevice::IsRootHub()
{
    return m_IsRootHub;
}

//--------------------------------------------------------------------------
//  DeviceManager::Interface implementation
//--------------------------------------------------------------------------
const char *DeviceManager::ManagedInterface::StateNames[DeviceManager::ManagedInterface::State_Maximum] =
{
    "Null", "Offered", "Acquired", "Released", "Revoked"
};

const char *DeviceManager::ManagedInterface::EventNames[DeviceManager::ManagedInterface::Event_Maximum] =
{
    "Try", "Continue", "Acquire", "Releasing", "Revoke", "AdminRequest",
    "EndpointCreateConfirm", "EndpointDestroyConfirm", "SetAltIfConfirm", "Error"
};
void* DeviceManager::ManagedInterface::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(
        size,
        DeviceManager::ManagedInterface::ObjectMemAlignmentSize,
        "DeviceManager::ManagedInterface"
    );
}
void DeviceManager::ManagedInterface::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "DeviceManager::ManagedInterface");
}

Result DeviceManager::ManagedInterface::Initialize()
{
    Result result = ResultSuccess();

    do
    {
        UsbDeviceDescriptor deviceDescriptor;
        Device *pDevice = m_pManagedDevice->m_pDevice;
        char ifObjName[HsLimitMaxDebugNameSize];
        if (m_pIf->pActAltSetting == nullptr)
        {
            result = ResultInternalStateError();
            break;
        }

        nn::util::SNPrintf(ifObjName, sizeof(ifObjName),
                           "Device%d ManagedInterface%d",
                           pDevice->GetUid(),
                           m_pIf->pActAltSetting->bInterfaceNumber);

        pDevice->GetDeviceDescriptor(&deviceDescriptor);
        NN_USB_BREAK_UPON_ERROR(m_pDm->m_InterfaceHandleManager.Assign(this, &m_InterfaceHandle));
        NN_USB_BREAK_UPON_ERROR(Fsm::Initialize(ifObjName,
                                                EventNames, Event_Maximum,
                                                StateNames, State_Maximum,
                                                State_Offered));
        m_pHs->AdvertiseDeviceInterfaceToClients(&deviceDescriptor, m_pIf->pActAltSetting);
    } while (false);


    return result;
}

Result DeviceManager::ManagedInterface::Finalize()
{
    Result result = ResultSuccess();

    do
    {
        // Sanity check
        NN_USB_BREAK_UNLESS(State_Revoked == Fsm::GetState(), ResultInternalStateError());

        // Purge any pending requests
        while (!m_AdminRequestList.empty())
        {
            CompleteCurrentAdminRequest(ResultInterfaceInvalid());
        }

        NN_USB_BREAK_UPON_ERROR(Fsm::Finalize());
    } while (false);

    return result;
}

DeviceManager::ManagedDevice* DeviceManager::ManagedInterface::GetManagedDevice()
{
    return m_pManagedDevice;
}

Device* DeviceManager::ManagedInterface::GetDevice()
{
    Device *pDevice = nullptr;
    if (m_pManagedDevice != nullptr)
    {
        pDevice = m_pManagedDevice->m_pDevice;
    }
    return pDevice;
}

DeviceInterface* DeviceManager::ManagedInterface::GetDeviceInterface()
{
    return m_pIf;
}

InterfaceHandle DeviceManager::ManagedInterface::GetInterfaceHandle()
{
    return m_InterfaceHandle;
}

void DeviceManager::ManagedInterface::SubmitIfAdminRequest(DeferredRequestBase *pRequest)
{
    Fsm::QueueFsmEvent(Event_AdminRequest, pRequest);

}

void DeviceManager::ManagedInterface::SetAltIfConfirm(Result status)
{
    Fsm::QueueFsmEvent(Event_SetAltIfConfirm);
}

bool DeviceManager::ManagedInterface::InitiateManagedEndpointDestruction()
{
    bool isDestroyConfirmPending = false;
    UsbEndpointMask destroyed = 0;
    Device *pDevice = m_pManagedDevice->m_pDevice;
    while (destroyed ^ m_OpenEndpointMask)
    {
        UsbEndpointMask foundMask;
        UsbEndpointDirection epDir;
        EndpointNumber epNumber;
        if ((foundMask = Util::GetEndpointFromMask(destroyed ^ m_OpenEndpointMask, &epDir, &epNumber)) != 0)
        {
            Result result;
            HostEndpoint *pHep;
            if(!pDevice->GetManagedHostEndpoint(epNumber, epDir, &pHep).IsSuccess()) break;
            NN_USB_BREAK_UPON_ERROR(pDevice->DestroyManagedHostEndpoint(pHep, this));
            isDestroyConfirmPending = true;
            destroyed |= foundMask;
        }
    }
    return isDestroyConfirmPending;
}

void DeviceManager::ManagedInterface::DispatchNextAdminRequest()
{
    Result result = ResultSuccess();
    if (m_AdminRequestList.empty()) return;
    Device *pDevice = GetDevice();
    DeviceInterface *pDif = GetDeviceInterface();
    int32_t ifIndex = pDif->pActAltSetting->bInterfaceNumber;
    DeferredRequestBase *pBaseRequest = &m_AdminRequestList.front();
    if (pBaseRequest->GetCategory() == 'I')
    {
        switch (pBaseRequest->GetCommand())
        {
        case ClientIfSession::Command_SetInterface:
            {
                ClientIfSession::SetInterfaceDeferredRequest *pRequest =
                    static_cast<ClientIfSession::SetInterfaceDeferredRequest *>(pBaseRequest);

                // We don't allow this operation when endpoints are open
                if (m_OpenEndpointMask != 0) NN_USB_BREAK_UPON_ERROR(ResultResourceBusy());
                NN_USB_BREAK_UPON_ERROR(pDevice->SetInterface(ifIndex, pRequest->m_Data.bAlternateSetting));

                // Successfully initiated, now defer until completion
                result = ResultDeferred();
                break;
            }
        case ClientIfSession::Command_GetInterface:
            {
                ClientIfSession::GetInterfaceDeferredRequest *pRequest =
                    static_cast<ClientIfSession::GetInterfaceDeferredRequest *>(pBaseRequest);
                result = pDevice->GetInterfaceProfile(ifIndex, pRequest->m_Data.pData);
                pRequest->m_Data.pData->handle = m_InterfaceHandle;
                break;
            }
        case ClientIfSession::Command_GetAltUsbIf:
            {
                ClientIfSession::GetInterfaceDeferredRequest *pRequest =
                    static_cast<ClientIfSession::GetInterfaceDeferredRequest *>(pBaseRequest);
                result = pDevice->GetAlternateInterfaceProfile(ifIndex,  pRequest->m_Data.bAlternateSetting,
                                                     pRequest->m_Data.pData);
                pRequest->m_Data.pData->handle = m_InterfaceHandle;
                break;
            }
        case ClientIfSession::Command_OpenUsbEp:
            {
                ClientIfSession::OpenUsbEpDeferredRequest *pRequest =
                    static_cast<ClientIfSession::OpenUsbEpDeferredRequest *>(pBaseRequest);
                ClientIfSession::OpenUsbEpArgDataType *pArgs = &pRequest->m_Data;
                UsbEndpointMask epMask = Util::GetEndpointMask(pArgs->epDirection, pArgs->epNumber);
                if (m_OpenEndpointMask & epMask) NN_USB_BREAK_UPON_ERROR(ResultResourceBusy());
                NN_USB_BREAK_UPON_ERROR(pDevice->CreateManagedHostEndpoint(ifIndex, pArgs->epType,
                                                                           pArgs->epNumber, pArgs->epDirection,
                                                                           pArgs->maxUrbCount, pArgs->maxXferSize, false,
                                                                           this));
                // Successfully initiated, now defer until completion
                result = ResultDeferred();
                break;
            }
        default:
            NN_USB_ABORT("Invalid command %d", pBaseRequest->GetCommand());
            break;
        }

    }
    else if (pBaseRequest->GetCategory() == 'E')
    {
        switch (pBaseRequest->GetCommand())
        {
        case ClientEpSession::Command_Close:
            {
                HostEndpoint *pHep;
                ClientEpSession::CloseDeferredRequest *pRequest =
                    static_cast<ClientEpSession::CloseDeferredRequest *>(pBaseRequest);
                ClientEpSession::CloseArgDataType *pArgs = &pRequest->m_Data;
                if((result = pDevice->GetManagedHostEndpoint(pArgs->epNumber,
                                                             pArgs->epDirection, &pHep)).IsFailure()) break;
                NN_USB_BREAK_UPON_ERROR(pDevice->DestroyManagedHostEndpoint(pHep, this));
                // Successfully initiated, now defer until completion
                result = ResultDeferred();
                break;
            }
        default:
            NN_USB_ABORT("Invalid command %d", pBaseRequest->GetCommand());
            break;
        }
    }
    else
    {
        NN_USB_ABORT("Invalid request of %c", pBaseRequest->GetCategory());
    }
    if (!ResultDeferred::Includes(result))
    {
        CompleteCurrentAdminRequest(result);
    }
}

void DeviceManager::ManagedInterface::HandleAcquiredEndpointCreateConfirm(HostEndpoint *pHep, Result status)
{
    NN_USB_ABORT_UNLESS(!m_AdminRequestList.empty());
    //DeviceInterface *pDif = GetDeviceInterface();
    DeferredRequestBase *pBaseRequest = &m_AdminRequestList.front();
    UsbEndpointMask singleCreatedEndpoint = Util::GetEndpointMask(pHep);
    if ((pBaseRequest->GetCategory() == 'I') && (pBaseRequest->GetCommand() == ClientIfSession::Command_OpenUsbEp))
    {
        ClientIfSession::OpenUsbEpDeferredRequest *pRequest =
            static_cast<ClientIfSession::OpenUsbEpDeferredRequest *>(pBaseRequest);
        if(status.IsSuccess())
        {
            m_OpenEndpointMask |= singleCreatedEndpoint;
            if(pRequest->m_Data.pReturnedEpDescriptor!= nullptr)
            {
                memcpy(pRequest->m_Data.pReturnedEpDescriptor, pHep->pDescriptor, UsbDescriptorSize_Endpoint);
            }
        }
        NN_USB_LOG_TRACE("%s - Endpoint %d has been opened.\n", Fsm::GetName(), pRequest->m_Data.epNumber);
        CompleteCurrentAdminRequest(status);
    }
    else
    {
        NN_USB_ABORT("Invalid request of %c", pBaseRequest->GetCategory());
    }
}

void DeviceManager::ManagedInterface::HandleAcquiredEndpointDestroyConfirm(UsbEndpointMask singleDestroyedEndpoint)
{
    Result result = ResultSuccess();
    NN_USB_ABORT_UNLESS(!m_AdminRequestList.empty());
    DeferredRequestBase *pBaseRequest = &m_AdminRequestList.front();
    if (pBaseRequest->GetCategory() == 'E')
    {
        switch (pBaseRequest->GetCommand())
        {
        case ClientEpSession::Command_Close:
            {
                ClientEpSession::CloseDeferredRequest *pRequest =
                    static_cast<ClientEpSession::CloseDeferredRequest *>(pBaseRequest);
                NN_UNUSED(pRequest);
                NN_USB_LOG_TRACE("%s - Endpoint %d has been closed.\n", Fsm::GetName(), pRequest->m_Data.epNumber);
                CompleteCurrentAdminRequest(result);
                break;
            }
        default:
            {
                NN_USB_ABORT("Invalid command of %d", pBaseRequest->GetCommand());
                break;
            }
        }
    }
    else
    {
        NN_USB_ABORT("Invalid request of %c", pBaseRequest->GetCategory());
    }
}

void DeviceManager::ManagedInterface::CompleteCurrentAdminRequest(Result result)
{
    DeferredRequestBase *pRequest = &m_AdminRequestList.front();
    m_AdminRequestList.pop_front();
    pRequest->CompleteRequest(result);
}

Result DeviceManager::ManagedInterface::FsmHandler(int32_t state, LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (state)
    {
    case State_Null:
        break;
    case State_Offered:
        result = OfferedStateHandler(pEvent);
        break;
    case State_Acquired:
        result = AcquiredStateHandler(pEvent);
        break;
    case State_Releasing:
        result = ReleasingStateHandler(pEvent);
        break;
    case State_Revoked:
        result = RevokedStateHandler(pEvent);
        break;
    default:
        NN_USB_ABORT("DeviceManager::Interface FSM state %d invalid", state);
        break;
    }
    return result;
}

Result DeviceManager::ManagedInterface::OfferedStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_pDm->m_OfferedIfList.push_back(*this);
            break;
        }
    case Event_AcquireRequest:
        {
            ClientRootSession::AcquireUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientRootSession::AcquireUsbIfDeferredRequest *>(pEvent->args[0].pData);

            result = m_pDm->GetProfiles(m_InterfaceHandle,
                                        pRequest->m_Data.pDeviceProfile,
                                        pRequest->m_Data.pInterfaceProfile);
            if (result.IsSuccess())
            {
                m_pClient = pRequest->m_Data.pClient;
                Fsm::SetState(State_Acquired);
            }
            pRequest->CompleteRequest(result);
            break;
        }
    case Event_AdminRequest:
        {
            DeferredRequestBase *pRequest = reinterpret_cast<DeferredRequestBase *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultInterfaceInvalidState());
            break;
        }
    case Event_Revoke:
        {
            Fsm::SetState(State_Revoked);
            break;
        }
    case Event_ReleaseRequest:
        {
            ClientIfSession::ReleaseUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientIfSession::ReleaseUsbIfDeferredRequest *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultInterfaceInvalidState());
            break;
        }
    case Fsm::Event_Exit:
        {
            m_pDm->m_OfferedIfList.erase(m_pDm->m_OfferedIfList.iterator_to(*this));
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    return result;
}

Result DeviceManager::ManagedInterface::AcquiredStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_pDm->m_AcquiredIfList.push_back(*this);
            break;
        }
    case Event_AdminRequest:
        {
            DeferredRequestBase *pRequest = reinterpret_cast<DeferredRequestBase *>(pEvent->args[0].pData);
            bool isIdle = m_AdminRequestList.empty();
            m_AdminRequestList.push_back(*pRequest);
            if (isIdle)
            {
                DispatchNextAdminRequest();
            }
            break;
        }
    case Event_AcquireRequest:
        {
            ClientRootSession::AcquireUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientRootSession::AcquireUsbIfDeferredRequest *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultResourceBusy());
            break;
        }
    case Event_EndpointCreateConfirm:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            HandleAcquiredEndpointCreateConfirm(pHep, pEvent->status);
            DispatchNextAdminRequest();
            break;
        }
    case Event_EndpointDestroyConfirm:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            m_OpenEndpointMask &= ~Util::GetEndpointMask(pHep);
            HandleAcquiredEndpointDestroyConfirm(Util::GetEndpointMask(pHep));
            DispatchNextAdminRequest();
            break;
        }
    case Event_SetAltIfConfirm:
        {
            Device *pDevice = m_pManagedDevice->m_pDevice;
            DeviceInterface *pDif = GetDeviceInterface();
            ClientIfSession::SetInterfaceDeferredRequest *pRequest =
                static_cast<ClientIfSession::SetInterfaceDeferredRequest *>(&m_AdminRequestList.front());
            result = pDevice->GetInterfaceProfile(pDif->pActAltSetting->bInterfaceNumber,
                                              pRequest->m_Data.pData);
            NN_USB_LOG_TRACE("%s - Alternate setting %d %s applied.\n",
                             Fsm::GetName(), pRequest->m_Data.bAlternateSetting,
                             (pEvent->status.IsSuccess()) ? "successfully" : "FAILED to be");
            CompleteCurrentAdminRequest(pEvent->status);
            DispatchNextAdminRequest();
            break;
        }
    case Event_ReleaseRequest:
        {
            ClientIfSession::ReleaseUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientIfSession::ReleaseUsbIfDeferredRequest *>(pEvent->args[0].pData);
            m_pPendingReleaseRequest = pRequest;
            Fsm::SetState(State_Releasing);
            break;
        }
    case Event_Revoke:
        {
            Fsm::SetState(State_Revoked);
            break;
        }
    case Fsm::Event_Exit:
        {
            m_pClient->SignalIfStateChange(GetInterfaceHandle());
            m_pDm->m_AcquiredIfList.erase(m_pDm->m_AcquiredIfList.iterator_to(*this));
            m_pClient = nullptr;
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    return result;
}

Result DeviceManager::ManagedInterface::ReleasingStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            // Expecting confirmations?
            if (InitiateManagedEndpointDestruction())
            {
                break;
            }
            Fsm::QueueFsmEvent(Event_Continue);
            break;
        }
    case Event_EndpointDestroyConfirm:
        {
            HostEndpoint *pHep = reinterpret_cast<HostEndpoint *>(pEvent->args[0].pData);
            m_OpenEndpointMask &= ~Util::GetEndpointMask(pHep);

            // More to come?
            if (m_OpenEndpointMask != 0)
            {
                break;
            }
            Fsm::QueueFsmEvent(Event_Continue);
            break;
        }
    case Event_Continue:
        {
            // Purge any other pending requests
            while (!m_AdminRequestList.empty())
            {
                CompleteCurrentAdminRequest(ResultInterfaceInvalid());
            }

            // Clean interface now available again
            Fsm::SetState(State_Offered);
            break;
        }
    case Event_AcquireRequest:
        {
            ClientRootSession::AcquireUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientRootSession::AcquireUsbIfDeferredRequest *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultResourceBusy());
            break;
        }
    case Event_AdminRequest:
        {
            DeferredRequestBase *pRequest = reinterpret_cast<DeferredRequestBase *>(pEvent->args[0].pData);
            m_AdminRequestList.push_back(*pRequest);
            break;
        }
    case Event_Revoke:
        {
            Fsm::SetState(State_Revoked);
            break;
        }
    case Event_ReleaseRequest:
        {
            ClientIfSession::ReleaseUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientIfSession::ReleaseUsbIfDeferredRequest *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultInterfaceInvalidState());
            break;
        }
    case Fsm::Event_Exit:
        {
            // Complete the release request which put us here
            if (m_pPendingReleaseRequest != nullptr)
            {
                m_pPendingReleaseRequest->CompleteRequest(result);
                m_pPendingReleaseRequest = nullptr;
            }
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    return result;
}

Result DeviceManager::ManagedInterface::RevokedStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            NN_USB_BREAK_UPON_ERROR(m_pDm->m_InterfaceHandleManager.Release(m_InterfaceHandle));
            break;
        }
    case Event_EndpointDestroyConfirm:
        {
            break;
        }
    case Event_AcquireRequest:
        {
            ClientRootSession::AcquireUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientRootSession::AcquireUsbIfDeferredRequest *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultInterfaceInvalidState());
            break;
        }
    case Event_AdminRequest:
        {
            DeferredRequestBase *pRequest = reinterpret_cast<DeferredRequestBase *>(pEvent->args[0].pData);
            m_AdminRequestList.push_back(*pRequest);
            break;
        }
    case Event_ReleaseRequest:
        {
            ClientIfSession::ReleaseUsbIfDeferredRequest *pRequest =
                reinterpret_cast<ClientIfSession::ReleaseUsbIfDeferredRequest *>(pEvent->args[0].pData);
            pRequest->CompleteRequest(ResultInterfaceInvalidState());
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }

    return result;
}

//--------------------------------------------------------------------------
//  DeviceManager implementation
//--------------------------------------------------------------------------
Result DeviceManager::Initialize()
{
    Result result = ResultSuccess();

    return result;
}

Result DeviceManager::Finalize()
{
    Result result = ResultSuccess();



    return result;
}

bool DeviceManager::IsTotalDeviceLimitExceeded()
{
    return (m_TotalDeviceCount >= HsLimitMaxDevicesCount);
}

Result DeviceManager::GetProfiles(InterfaceHandle   ifHandle,
                                  DeviceProfile    *pDeviceProfile,
                                  InterfaceProfile *pInterfaceProfile)
{
    Result result = ResultInterfaceInvalid();
    ManagedInterface *pMi;
    if ((pMi = m_InterfaceHandleManager.Get(ifHandle)) != nullptr)
    {
        Device *pDevice = pMi->GetDevice();
        DeviceInterface *pDif = pMi->GetDeviceInterface();
        if ((pDevice != nullptr) && (pDif != nullptr) && (pDif->pActAltSetting != nullptr))
        {
            pDevice->GetDeviceProfile(pDeviceProfile);
            result = pDevice->GetInterfaceProfile(pDif->pActAltSetting->bInterfaceNumber, pInterfaceProfile);
            pInterfaceProfile->handle = ifHandle;
        }
    }
    return result;
}

Result DeviceManager::QueryAllInterfaces(ClientRootSession::QueryUsbIfDeferredRequest *pRequest)
{
    Result result = ResultSuccess();
    int32_t ifCount = 0;
    ClientRootSession::QueryUsbIfArgDataType *pArgs = &pRequest->m_Data;

    for (InterfaceListType::iterator it = m_OfferedIfList.begin();
         (it != m_OfferedIfList.end()) && (ifCount < pArgs->maxOutputCount); it++)
    {
        Device *pDevice = it->GetDevice();
        DeviceInterface *pDif = it->GetDeviceInterface();
        if ((pDevice != nullptr) && (pDif != nullptr))
        {
            UsbDeviceDescriptor deviceDescriptor;
            UsbConfigDescriptor configDescriptor;
            pDevice->GetDeviceDescriptor(&deviceDescriptor);
            pDevice->GetConfigDescriptor(&configDescriptor);
            if (Util::EvaluateDeviceId(pArgs->pDevFilter, pDif->pActAltSetting, &deviceDescriptor))
            {
                InterfaceQueryOutput *pQo = pArgs->pQueryOut + ifCount;
                int32_t ifIndex = pDif->pActAltSetting->bInterfaceNumber;
                pDevice->GetInterfaceProfile(ifIndex, &pQo->ifProfile);
                pDevice->GetObjectName(pQo->deviceProfile.deviceDebugName,
                                       sizeof(pQo->deviceProfile.deviceDebugName));
                pQo->ifProfile.handle          = -1;  // invalid handle so the interface cannot be acquired
                pQo->deviceProfile.cfgDesc     = configDescriptor;
                pQo->deviceProfile.deviceDesc  = deviceDescriptor;
                pQo->deviceProfile.deviceSpeed = pDevice->GetSpeed();
                pQo->deviceProfile.deviceUid   = pDevice->GetUid();
                pQo->deviceProfile.enumerationTimestamp = pDevice->GetEnumerationTimeStamp();
                ifCount++;
            }
        }
    }

    for (InterfaceListType::iterator it = m_AcquiredIfList.begin();
         (it != m_AcquiredIfList.end()) && (ifCount < pArgs->maxOutputCount); it++)
    {
        Device *pDevice = it->GetDevice();
        DeviceInterface *pDif = it->GetDeviceInterface();
        if ((pDevice != nullptr) && (pDif != nullptr))
        {
            UsbDeviceDescriptor deviceDescriptor;
            UsbConfigDescriptor configDescriptor;
            pDevice->GetDeviceDescriptor(&deviceDescriptor);
            pDevice->GetConfigDescriptor(&configDescriptor);
            if (Util::EvaluateDeviceId(pArgs->pDevFilter, pDif->pActAltSetting, &deviceDescriptor))
            {
                InterfaceQueryOutput *pQo = pArgs->pQueryOut + ifCount;
                int32_t ifIndex = pDif->pActAltSetting->bInterfaceNumber;
                pDevice->GetInterfaceProfile(ifIndex, &pQo->ifProfile);
                pDevice->GetObjectName(pQo->deviceProfile.deviceDebugName,
                                       sizeof(pQo->deviceProfile.deviceDebugName));
                pQo->ifProfile.handle          = -1;  // invalid handle so the interface cannot be acquired
                pQo->deviceProfile.cfgDesc     = configDescriptor;
                pQo->deviceProfile.deviceDesc  = deviceDescriptor;
                pQo->deviceProfile.deviceSpeed = pDevice->GetSpeed();
                pQo->deviceProfile.deviceUid   = pDevice->GetUid();
                pQo->deviceProfile.enumerationTimestamp = pDevice->GetEnumerationTimeStamp();
                ifCount++;
            }
        }
    }

    *pArgs->pQueryOutCount = ifCount;
    return result;
}

Result DeviceManager::QueryAvailableInterfaces(ClientRootSession::QueryUsbIfDeferredRequest *pRequest)
{
    Result result = ResultSuccess();
    int32_t ifCount = 0;
    ClientRootSession::QueryUsbIfArgDataType *pArgs = &pRequest->m_Data;
    for (InterfaceListType::iterator it = m_OfferedIfList.begin();
         (it != m_OfferedIfList.end()) && (ifCount < pArgs->maxOutputCount); it++)
    {
        Device *pDevice = it->GetDevice();
        DeviceInterface *pDif = it->GetDeviceInterface();
        if ((pDevice != nullptr) && (pDif != nullptr))
        {
            UsbDeviceDescriptor deviceDescriptor;
            UsbConfigDescriptor configDescriptor;
            pDevice->GetDeviceDescriptor(&deviceDescriptor);
            pDevice->GetConfigDescriptor(&configDescriptor);
            if (Util::EvaluateDeviceId(pArgs->pDevFilter, pDif->pActAltSetting, &deviceDescriptor))
            {
                InterfaceQueryOutput *pQo = pArgs->pQueryOut + ifCount;
                int32_t ifIndex = pDif->pActAltSetting->bInterfaceNumber;
                pDevice->GetInterfaceProfile(ifIndex, &pQo->ifProfile);
                pDevice->GetObjectName(pQo->deviceProfile.deviceDebugName,
                                       sizeof(pQo->deviceProfile.deviceDebugName));
                pQo->ifProfile.handle          = it->GetInterfaceHandle();
                pQo->deviceProfile.cfgDesc     = configDescriptor;
                pQo->deviceProfile.deviceDesc  = deviceDescriptor;
                pQo->deviceProfile.deviceSpeed = pDevice->GetSpeed();
                pQo->deviceProfile.deviceUid   = pDevice->GetUid();
                pQo->deviceProfile.enumerationTimestamp = pDevice->GetEnumerationTimeStamp();
                ifCount++;
            }
        }
    }
    *pArgs->pQueryOutCount = ifCount;
    return result;
}

Result DeviceManager::QueryAcquiredInterfaces(ClientRootSession::QueryUsbIfDeferredRequest *pRequest)
{
    Result result = ResultSuccess();
    int32_t ifCount = 0;
    ClientRootSession::QueryUsbIfArgDataType *pArgs = &pRequest->m_Data;
    for (InterfaceListType::iterator it = m_AcquiredIfList.begin();
         (it != m_AcquiredIfList.end()) && (ifCount < pArgs->maxOutputCount); it++)
    {
        Device *pDevice = it->GetDevice();
        DeviceInterface *pDif = it->GetDeviceInterface();
        if ((pDevice != nullptr) && (pDif != nullptr))
        {
            UsbDeviceDescriptor deviceDescriptor;
            UsbConfigDescriptor configDescriptor;
            pDevice->GetDeviceDescriptor(&deviceDescriptor);
            pDevice->GetConfigDescriptor(&configDescriptor);
            if (it->m_pClient == pRequest->m_Data.pClient)
            {
                InterfaceQueryOutput *pQo = pArgs->pQueryOut + ifCount;
                int32_t ifIndex = pDif->pActAltSetting->bInterfaceNumber;
                pDevice->GetInterfaceProfile(ifIndex, &pQo->ifProfile);
                pDevice->GetObjectName(pQo->deviceProfile.deviceDebugName,
                                       sizeof(pQo->deviceProfile.deviceDebugName));
                pQo->ifProfile.handle          = it->GetInterfaceHandle();
                pQo->deviceProfile.cfgDesc     = configDescriptor;
                pQo->deviceProfile.deviceDesc  = deviceDescriptor;
                pQo->deviceProfile.deviceSpeed = pDevice->GetSpeed();
                pQo->deviceProfile.deviceUid   = pDevice->GetUid();
                pQo->deviceProfile.enumerationTimestamp = pDevice->GetEnumerationTimeStamp();
                ifCount++;
            }
        }
    }
    *pArgs->pQueryOutCount = ifCount;
    return result;
}

Result DeviceManager::NewUsbIfAvailableEvent(ClientRootSession::NewUsbIfAvailableEventDeferredRequest *pRequest)
{
    Result result = ResultSuccess();
    for (InterfaceListType::iterator it = m_OfferedIfList.begin(); it != m_OfferedIfList.end(); it++)
    {
        Device *pDevice = it->GetDevice();
        DeviceInterface *pDif = it->GetDeviceInterface();
        if ((pDevice != nullptr) && (pDif != nullptr))
        {
            UsbDeviceDescriptor deviceDescriptor;
            pDevice->GetDeviceDescriptor(&deviceDescriptor);
            pRequest->m_Data.isMatched = Util::EvaluateDeviceId(&pRequest->m_Data.deviceFilter, pDif->pActAltSetting,
                                                                &deviceDescriptor);
            if(pRequest->m_Data.isMatched) break;
        }
    }
    return result;
}

Result DeviceManager::AcquireUsbIf(ClientRootSession::AcquireUsbIfDeferredRequest *pRequest)
{
    Result result = ResultDeferred();
    ManagedInterface *pMi;
    if ((pMi = m_InterfaceHandleManager.Get(pRequest->m_Data.ifHandle)) != nullptr)
    {
        pMi->QueueFsmEvent(ManagedInterface::Event_AcquireRequest, pRequest);
    }
    else
    {
        result = ResultInterfaceInvalid();
    }
    return result;
}

Result DeviceManager::ReleaseUsbIf(ClientIfSession::ReleaseUsbIfDeferredRequest *pRequest)
{
    Result result = ResultDeferred();
    ManagedInterface *pMi;
    if ((pMi = m_InterfaceHandleManager.Get(pRequest->m_Data.ifHandle)) != nullptr)
    {
        pMi->QueueFsmEvent(ManagedInterface::Event_ReleaseRequest, pRequest);
    }
    else
    {
        result = ResultInterfaceInvalid();
    }
    return result;
}

Result DeviceManager::SubmitControlTransfer(ClientIfSession::CtrlXferDeferredRequest *pRequest)
{
    Result result = ResultInterfaceInvalid();
    ManagedInterface *pMi;
    if ((pMi = m_InterfaceHandleManager.Get(pRequest->m_Data.ifHandle)) != nullptr)
    {
        Device *pDevice = pMi->GetDevice();
        ControlTransferManager *pCtrlMgr;
        if ((result = pDevice->GetControlTransferManager(&pCtrlMgr)).IsSuccess())
        {
            result = pCtrlMgr->SubmitAsync(pRequest);
            if (result.IsSuccess())
            {
                result = ResultDeferred();
            }
        }
    }
    return result;
}

Result DeviceManager::SubmitDeferredIfAdminRequest(InterfaceHandle ifHandle,
                                                   DeferredRequestBase *pRequest)
{
    Result result = ResultDeferred();
    ManagedInterface *pMi;
    if ((pMi = m_InterfaceHandleManager.Get(ifHandle)) != nullptr)
    {
        pMi->SubmitIfAdminRequest(pRequest);
    }
    else
    {
        result = ResultInterfaceInvalid();
    }
    return result;
}

Result DeviceManager::GetCurrentFrameId(InterfaceHandle ifHandle, FrameNumber *frame)
{
    ManagedInterface *pMi = m_InterfaceHandleManager.Get(ifHandle);

    if (pMi == nullptr)
    {
        return ResultInterfaceInvalid();
    }

    Device *pDevice = pMi->GetDevice();
    HostControllerDriver *pHc = pDevice->GetHostControllerDriver();

    *frame = pHc->GetCurrentFrameId();

    return ResultSuccess();
}

Result DeviceManager::SubmitUrb(ClientEpSession::UrbDeferredRequest *pRequest)
{
    Result                        result;
    ManagedInterface             *pMi;
    HostEndpoint                 *pHep;
    Device                       *pDevice;
    UsbRequestBlock              *pUrb;
    HostControllerDriverEndpoint *pHcEp;

    auto& data = pRequest->m_Data;

    pMi = m_InterfaceHandleManager.Get(data.ifHandle);
    if (pMi == nullptr)
    {
        result = ResultInterfaceInvalid();
        goto bail1;
    }

    pDevice = pMi->GetDevice();
    result = pDevice->GetManagedHostEndpoint(data.epNumber,
                                             data.epDirection,
                                             &pHep);
    if (result.IsFailure())
    {
        goto bail1;
    }

    pHcEp = pHep->pHcEp;
    result = pHcEp->CreateUrb(pRequest, &pUrb);
    if (result.IsFailure())
    {
        goto bail1;
    }

    result = pUrb->Submit();
    if (result.IsFailure())
    {
        pHcEp->DestroyUrb(pUrb);
        goto bail1;
    }

    return ResultSuccess();

bail1:
    NN_USB_LOG_WARN(
        "SubmitUrb: ifHandle=0x%x Ep-%d-%s: failed with %d:%d\n",
        pRequest->m_Data.ifHandle,
        pRequest->m_Data.epNumber,
        pRequest->m_Data.epDirection == UsbEndpointDirection_ToDevice ? "OUT" : "IN",
        result.GetModule(), result.GetDescription()
    );

    return result;
}

Result DeviceManager::ResetDevice(ClientIfSession::ResetDeviceDeferredRequest* pRequest)
{
    Result result = ResultSuccess();
    ManagedInterface *pMi;
    if ((pMi = m_InterfaceHandleManager.Get(pRequest->m_Data.ifHandle)) != nullptr)
    {
        Device *pDevice = pMi->GetDevice();
        result = pDevice->Reset();
    }
    else
    {
        result = ResultInterfaceInvalid();
    }
    return result;
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
