﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file    usb_DsUrbRing.h
 * @brief   URB (USB Request Block) Ring
 */

#pragma once

#include "usb_DsCommon.h"

namespace nn {
namespace usb {
namespace ds {

class DsController;

class DsUrbRing
{
public:
    DsUrbRing(DsController *pController, uint8_t address) NN_NOEXCEPT
        : m_Head(0)
        , m_Tail(0)
        , m_Mutex(false)
        , m_CompletionEvent(nn::os::EventClearMode_ManualClear, true)
        , m_pController(pController)
        , m_Address(address)
    {
          Reset();
    }

    ~DsUrbRing() NN_NOEXCEPT
    {
        // do nothing
    }
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;

    Result Reset() NN_NOEXCEPT;

    Result Enqueue(uint32_t *pOutUrbId,
                   nn::dd::ProcessHandle processHandle,
                   uint64_t address, uint32_t bytes) NN_NOEXCEPT;

    Result Dequeue(Result result, uint32_t bytesRemaining, bool lockMutex) NN_NOEXCEPT;
    Result Flush() NN_NOEXCEPT;
    Result GetReport(UrbReport *pReport) NN_NOEXCEPT;
    nn::os::SystemEventType* GetCompletionEvent() NN_NOEXCEPT;
    void BlockForCompletion() NN_NOEXCEPT;

    void SetController(DsController * pController) NN_NOEXCEPT;

private:
    uint32_t              m_Head;
    uint32_t              m_Tail;

    nn::os::Mutex         m_Mutex;
    nn::os::SystemEvent   m_CompletionEvent;

    DsController         *m_pController;
    uint8_t               m_Address;

    struct Urb {
        uint32_t      id;

        dd::ProcessHandle processHandle;
        uint64_t      procVa;
        uint64_t      ioVa;
        uint32_t      bytes;

        UrbStatus     status;
        uint32_t      transferredSize;


    } m_Urb[DsLimitRingSize];

private:
    Result Prime() NN_NOEXCEPT;

    uint32_t Advance(uint32_t index)
    {
        return (index + 1) % NN_USB_ARRAY_SIZE(m_Urb);
    }

    bool IsEmpty() NN_NOEXCEPT
    {
        // caller maintains the lock
        return m_Head == m_Tail;
    }

    bool IsFull() NN_NOEXCEPT
    {
        // caller maintains the lock
        return Advance(m_Head) == m_Tail;
    }

};

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn

