﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_DsServiceManager.h"
#include "usb_DsServiceImpl.h"
#include "usb_DsServiceName.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace usb {
namespace ds {

Result DsServiceManager::Initialize(detail::UsbPlatform *pPlatform) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    m_pPlatform = pPlatform;

    // Register the service port
    NN_USB_ABORT_UPON_ERROR(
        InitializePort(0, DsLimitMaxSessionCount, ServiceName)
    );

    // Enable the service port.
    // The port doesn't accept any connection until we call LoopAuto().
    Start();

    // Create and start the service thread pool
    // TODO: maybe dynamically create/destroy service threads?
    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_ThreadPool); i++)
    {
        NN_USB_RETURN_UPON_ERROR(
            nn::os::CreateThread(&m_ThreadPool[i],
                                 ThreadEntryStatic,
                                 this,
                                 m_ThreadStack[i],
                                 sizeof(m_ThreadStack[i]),
                                 NN_SYSTEM_THREAD_PRIORITY(usb,DsIpcServer))
        );
        nn::os::SetThreadNamePointer(&m_ThreadPool[i], NN_SYSTEM_THREAD_NAME(usb, DsIpcServer));
        nn::os::StartThread(&m_ThreadPool[i]);
    }

    return result;
}

Result DsServiceManager::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    RequestStop();

    // revoke the thread pool
    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_ThreadPool); i++)
    {
        nn::os::WaitThread(&m_ThreadPool[i]);
        nn::os::DestroyThread(&m_ThreadPool[i]);
    }

    // no way to destroy the usb:ds service name, too bad

    return result;
}

Result DsServiceManager::OnNeedsToAccept(int portIndex, PortForAllInOne *pPort) NN_NOEXCEPT
{
    NN_UNUSED(portIndex);

    return AcceptImpl(
        pPort,
        detail::Factory::CreateSharedEmplaced<IDsService, DsServiceImpl>(
            detail::UsbMemoryGetAllocator(),
            m_pPlatform
        )
    );
}

void DsServiceManager::ThreadEntryStatic(void *arg) NN_NOEXCEPT
{
    DsServiceManager *pThis = reinterpret_cast<DsServiceManager *>(arg);

    pThis->LoopAuto();
}

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
