﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE 1

#include "usb_DsDescriptor.h"
#include "usb_DsProtocol.h"

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsDescriptor"

using namespace nn::usb;

namespace nn {
namespace usb {
namespace ds {

//////////////////////////////////////////////////////////////////////////////
// Public Methods
//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::Initialize(DsProtocol *pProtocol) NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_pProtocol = pProtocol;
    m_Length    = 0;

    m_StringTable.Initialize();
    m_InterfaceData.Initialize();

    // default configuration descriptor
    m_ConfigurationDescriptor.bLength             = UsbDescriptorSize_Config;
    m_ConfigurationDescriptor.bDescriptorType     = UsbDescriptorType_Config;
    m_ConfigurationDescriptor.wTotalLength        = UsbDescriptorSize_Config;
    m_ConfigurationDescriptor.bNumInterfaces      = 0;
    m_ConfigurationDescriptor.bConfigurationValue = 1;
    m_ConfigurationDescriptor.iConfiguration      = 0;
    m_ConfigurationDescriptor.bmAttributes        = UsbConfigDescriptorAttributeMask_One | UsbConfigDescriptorAttributeMask_SelfPower;
    m_ConfigurationDescriptor.bMaxPower           = 0; // This willbe populated at the time of GET DESCRIPTOR

    memset(m_InterfaceIsOccupied, 0, sizeof(m_InterfaceIsOccupied));
    memset(m_EndpointDescriptor,  0, sizeof(m_EndpointDescriptor));
    memset(m_EpBitmap, 0, sizeof(m_EpBitmap));

    // Default EP0 descriptor assignment.
    m_EndpointDescriptor[0] =
    {
        UsbDescriptorSize_Endpoint,                 // bLength
        UsbDescriptorType_Endpoint,                 // bDescriptorType
        0,                                          // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeControl,   // bmAttributes
        64,                                         // wMaxPacketSize
        0                                           // bInterval
    };

    m_EndpointDescriptor[16] =
    {
        UsbDescriptorSize_Endpoint,                 // bLength
        UsbDescriptorType_Endpoint,                 // bDescriptorType
        0x80,                                       // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeControl,   // bmAttributes
        64,                                         // wMaxPacketSize
        0                                           // bInterval
    };

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_pProtocol = nullptr;
    m_InterfaceData.Finalize();
    m_StringTable.Finalize();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::ClearDeviceData() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_StringTable.ClearData();
    m_InterfaceData.ClearData();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::AddUsbStringDescriptor(UsbStringDescriptor *pUsbStringDescriptor, uint8_t *pIndex) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_StringTable.AddUsbStringDescriptor(pUsbStringDescriptor, pIndex);
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_StringTable.DeleteUsbStringDescriptor(index);
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::SetUsbDeviceDescriptor(UsbDeviceDescriptor *pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    UsbDeviceDescriptor *pUsbDeviceDescriptorDest = NULL;

    switch (usbDeviceSpeed)
    {
    case UsbDeviceSpeed_Low:
    case UsbDeviceSpeed_Full:

        pUsbDeviceDescriptorDest = &m_DeviceDescriptorLowFullSpeed;

        break;

    case UsbDeviceSpeed_High:

        pUsbDeviceDescriptorDest = &m_DeviceDescriptorHighSpeed;

        break;

    case UsbDeviceSpeed_Super:

        pUsbDeviceDescriptorDest = &m_DeviceDescriptorSuperSpeed;

        break;

    default:

        NN_USB_LOG_INFO("%s(@%d) unsupported UsbDeviceSpeed %d\n", __FUNCTION__,__LINE__, usbDeviceSpeed);

        break;
    }

    if (pUsbDeviceDescriptorDest)
    {
        memcpy(pUsbDeviceDescriptorDest, pUsbDeviceDescriptor, sizeof(UsbDeviceDescriptor));
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::SetBinaryObjectStore(uint8_t *pData, int size) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // size already checked
    memcpy(&m_BinaryObjectStore, pData, size);
    m_BinaryObjectStoreSize = size;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::AppendConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pData, int bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_InterfaceData.AppendConfigurationData(bInterfaceNumber, usbDeviceSpeed, pData, bytes);
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::GetConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pDest, int *pSize) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_InterfaceData.GetConfigurationData(bInterfaceNumber, usbDeviceSpeed, pDest, pSize);
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::SetReportData(uint8_t bInterfaceNumber, uint8_t *pData, int bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_InterfaceData.SetReportData(bInterfaceNumber, pData, bytes);
}


Result DsDescriptor::AddEndpoint(uint8_t epAddress, uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    int epIndex = detail::GetEndpointIndex(epAddress);

    // make sure the endpoint number is still available
    if (m_EndpointDescriptor[epIndex].bLength != 0)
    {
        return ResultResourceBusy();
    }

    // bLength is used here to flag whether the endpoint is used
    m_EndpointDescriptor[epIndex].bLength = sizeof(UsbEndpointDescriptor);

    // remember the interface-endpoint relationship
    m_EpBitmap[ifNum]  |= 1 << epIndex;
    m_EpParent[epIndex] = ifNum;

    return ResultSuccess();
}

uint8_t DsDescriptor::GetParentInterface(uint8_t epAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;
    int epIndex = detail::GetEndpointIndex(epAddress);

    return m_EpParent[epIndex];
}

uint8_t DsDescriptor::GetParentInterfaceByIndex(int epIndex) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_EpParent[epIndex];
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::AddInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result = ResultSuccess();

    if (m_InterfaceIsOccupied[bInterfaceNumber])
    {
        result = ResultResourceBusy();
    }
    else
    {
        m_InterfaceIsOccupied[bInterfaceNumber] = true;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsDescriptor::DelInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Result result = ResultSuccess();

    if (m_InterfaceIsOccupied[bInterfaceNumber])
    {
        m_InterfaceIsOccupied[bInterfaceNumber] = false;
    }
    else
    {
        return ResultOperationDenied();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
int DsDescriptor::GetStringDescriptor(void *pDestination, uint8_t index, int length) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    int bytes = -1;

    UsbStringDescriptor *pUsbStringDesciptor = m_StringTable.GetUsbStringDescriptor(index);

    if (pUsbStringDesciptor)
    {
        bytes = NN_USB_MIN(pUsbStringDesciptor->bLength, length);
        memcpy(pDestination, pUsbStringDesciptor, bytes);
    }

    return bytes;
}


//////////////////////////////////////////////////////////////////////////////
int DsDescriptor::GetDeviceDescriptor(void *pDestination, int length, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    int bytes = 0;
    UsbDeviceDescriptor *pUsbDeviceDescriptor = NULL;

    switch (usbDeviceSpeed)
    {
    case UsbDeviceSpeed_Low:
    case UsbDeviceSpeed_Full:

        pUsbDeviceDescriptor = &m_DeviceDescriptorLowFullSpeed;

        break;

    case UsbDeviceSpeed_High:

        pUsbDeviceDescriptor = &m_DeviceDescriptorHighSpeed;

        break;

    case UsbDeviceSpeed_Super:

        pUsbDeviceDescriptor = &m_DeviceDescriptorSuperSpeed;

        break;

    default:

        NN_USB_LOG_INFO("%s(@%d) unsupported UsbDeviceSpeed %d\n", __FUNCTION__,__LINE__, usbDeviceSpeed);

        break;
    }

    // we never modify the device descriptor, so no need to use a lock
    if (pUsbDeviceDescriptor)
    {
        bytes = NN_USB_MIN(pUsbDeviceDescriptor->bLength, length);

        memcpy(pDestination, pUsbDeviceDescriptor, bytes);
    }

    return bytes;
}


//////////////////////////////////////////////////////////////////////////////
int DsDescriptor::GetConfigurationDescriptor(void *pDestination, uint8_t index, int length) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    int bytes = NN_USB_MIN(length, m_Length);

    // we support only 1 configuration for now
    if (index != 0)
    {
        return -1;
    }

    memcpy(pDestination, &m_Buffer, bytes);

    return bytes;
}


//////////////////////////////////////////////////////////////////////////////
int DsDescriptor::GetBinaryObjectStore(void *pDestination, int length) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    int bytes = NN_USB_MIN(m_BinaryObjectStoreSize, length);

    memcpy(pDestination, &m_BinaryObjectStore, bytes);

    return bytes;
}


//////////////////////////////////////////////////////////////////////////////
int DsDescriptor::GetReportData(uint8_t bInterfaceNumber, uint8_t *pDest, int length) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    uint8_t data[MaxInterfaceDataSize];
    int bytes = 0;

    if (m_InterfaceData.GetReportData(bInterfaceNumber, data, &bytes).IsSuccess())
    {
        if (length >= bytes)
        {
            memcpy(pDest, data, bytes);
        }
    }

    return bytes;
}


Result DsDescriptor::DelEndpoint(uint8_t epAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    int epIndex = detail::GetEndpointIndex(epAddress);
    uint8_t ifNum = m_EpParent[epIndex];
    UsbEndpointDescriptor *pDescriptor = &m_EndpointDescriptor[epIndex];

    if (pDescriptor->bLength == 0)
    {
        return ResultOperationDenied();
    }

    memset(pDescriptor, 0, UsbDescriptorSize_Endpoint);

    m_EpBitmap[ifNum] &= ~(1 << epIndex);

    // mark the parent to be an invalid interface number
    m_EpParent[epIndex] = DsLimitMaxInterfacesPerConfigurationCount;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
void DsDescriptor::UpdateConfigurationDescriptor(UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    uint8_t ifCount = 0;
    UsbConfigDescriptor *pUsbConfigurationDescriptor = reinterpret_cast<UsbConfigDescriptor*>(m_Buffer);

    if (usbDeviceSpeed == UsbDeviceSpeed_Super)
    {
        m_ConfigurationDescriptor.bMaxPower       = 900 / 8; // USB 3.1 Spec 9.6.3 8mA units for gen X speed
    }
    else
    {
        m_ConfigurationDescriptor.bMaxPower       = 500 / 2; // 2mA units
    }

    // configuration descriptor
    memcpy(pUsbConfigurationDescriptor, &m_ConfigurationDescriptor, sizeof(UsbConfigDescriptor));
    m_Length = sizeof(UsbConfigDescriptor);

    // interface data
    for (uint8_t i = 0; i < DsLimitMaxInterfacesPerConfigurationCount; i++)
    {
        if (m_pProtocol->IsInterfaceEnabled(i))
        {
            ifCount++;

            int bytes;
            Result result = m_InterfaceData.GetConfigurationData(i, usbDeviceSpeed, &m_Buffer[m_Length], &bytes);

            if (result.IsSuccess())
            {
                m_Length += bytes;
            }
        }
    }

    // patch the configuration descriptor
    pUsbConfigurationDescriptor->bNumInterfaces = ifCount;
    pUsbConfigurationDescriptor->wTotalLength   = m_Length;
}

int DsDescriptor::GetEndpointByIndex(void *pDestination, int index) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    UsbEndpointDescriptor *pUsbEndpointDescriptor = &m_EndpointDescriptor[index];

    if (pUsbEndpointDescriptor->bLength)
    {
        memcpy(pDestination, pUsbEndpointDescriptor, pUsbEndpointDescriptor->bLength);
    }

    return pUsbEndpointDescriptor->bLength;
}

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
