﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <mutex>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/usb/usb_Result.h>
#include "usb_IoVaManager.h"

/**
 * @file
 * @brief sMMU DMA map manager
 *
 * @details
 *
 *
 */
namespace nn { namespace usb { namespace detail {

class SmmuMapManager
{
public:
    SmmuMapManager() : m_Mutex(true) {}
    ~SmmuMapManager() {}
    Result Initialize(nn::dd::DeviceName dmaDeviceName, nn::dd::DeviceVirtualAddress baseIoVa, size_t ioVaHeapSize);
    Result Finalize();

    // Mapping
    Result Map(nn::dd::DeviceVirtualAddress *pOutIoVa,
               nn::dd::ProcessHandle processHandle, uint64_t processAddress, size_t size,
               nn::dd::MemoryPermission cpuPermission, nn::dd::MemoryPermission devicePermission);
    Result Map(nn::dd::DeviceVirtualAddress *pOutIoVa,
               nn::dd::ProcessHandle processHandle, uint64_t processAddress, size_t size, size_t alignment,
               nn::dd::MemoryPermission cpuPermission, nn::dd::MemoryPermission devicePermission);
    Result Map(nn::dd::DeviceVirtualAddress *pOutIoVa,
               void* pBuffer,  size_t size,
               nn::dd::MemoryPermission cpuPermission, nn::dd::MemoryPermission devicePermission);
    Result Map(nn::dd::DeviceVirtualAddress *pOutIoVa, void* pBuffer, size_t size);
    Result Map(nn::dd::DeviceVirtualAddress *pOutIoVa, void* pBuffer, size_t size, size_t alignment);
    Result Map(void* pBuffer, size_t size);

    // Unmapping
    Result Unmap(nn::dd::ProcessHandle procHandle, uint64_t procVa, uint32_t size);
    Result Unmap(void *pBuffer, uint32_t size);
    Result Unmap(void *pBuffer);
    Result UnmapByIoVa(nn::dd::DeviceVirtualAddress ioVa, uint32_t size);
    Result UnmapByIoVa(nn::dd::DeviceVirtualAddress ioVa);

    // Get
    nn::dd::DeviceVirtualAddress GetIoVa(nn::dd::ProcessHandle procHandle, uint64_t procVa);
    nn::dd::DeviceVirtualAddress GetIoVa(void* pBuffer);

    // Dynamic allocation of DMA-capable memory
    void*  AllocMemory (nn::dd::DeviceVirtualAddress *pOutIoVa, size_t size, const char *objectName);
    void*  CallocMemory(nn::dd::DeviceVirtualAddress *pOutIoVa, size_t size, const char *objectName);
    Result FreeMemory(void *pBuffer, const char *objectName);
    Result FreeMemory(void *pBuffer, size_t size, const char *objectName);

private:
    nn::os::Mutex                   m_Mutex;
    nn::dd::DeviceName              m_DeviceName;
    nn::dd::DeviceAddressSpaceType  m_DeviceAddressSpace;
    IoVaManager<nn::dd::DeviceAddressSpaceMemoryRegionAlignment, 1> m_IoVaManager;
};

class SmmuSpace
{
public:
    uint64_t                       procVa;
    uint32_t                       size;
    nn::dd::DeviceVirtualAddress   ioVa;

public:
    SmmuSpace()
        : procVa(0)
        , size(0)
        , ioVa(0)
    {
    }

    bool Includes(uint64_t addr)
    {
        return (addr >= procVa) && (addr < procVa + size);
    }
};


} // end of namespace detail
} // end of namespace usb
} // end of namespace nn

