﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/updater/updater.h>

#include "updater_Common.h"
#include "updater_UpdateTypeResolver.h"

namespace nn { namespace updater {

namespace
{
const char* MountName = "bip"; // boot image package

const char* BctPath = "bip:/nx/bct";
const char* Package1Path = "bip:/nx/package1";
const char* Package2Path = "bip:/nx/package2";

const char* BctPathA = "bip:/a/bct";
const char* Package1PathA = "bip:/a/package1";
const char* Package2PathA = "bip:/a/package2";

const char* ResolveFile(const char* filenames[], int filenameCount) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(filenameCount > 0);
    for (int i = 0; i < filenameCount; ++i)
    {
        fs::DirectoryEntryType type;
        NN_RESULT_TRY(fs::GetEntryType(&type, filenames[i]))
            NN_RESULT_CATCH_ALL
        {
            continue;
        }
        NN_RESULT_END_TRY;

        if (type != fs::DirectoryEntryType_File)
        {
            continue;
        }

        return filenames[i];
    }
    // 見つからなかった場合は最後を返す
    return filenames[filenameCount - 1];
}
}

const char* GetMountName() NN_NOEXCEPT
{
    return MountName;
}

const char* ResolveBctPath(BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    switch(bootImageUpdateType)
    {
    case BootImageUpdateType::Original:
        {
            const char* candidates[] = {BctPath};
            return ResolveFile(candidates, static_cast<int>(NN_ARRAY_SIZE(candidates)));
        }
    case BootImageUpdateType::OdinMariko:
        {
            const char* candidates[] = {BctPathA, BctPath};
            return ResolveFile(candidates, static_cast<int>(NN_ARRAY_SIZE(candidates)));
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

const char* ResolvePackage1Path(BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    switch(bootImageUpdateType)
    {
    case BootImageUpdateType::Original:
        {
            const char* candidates[] = {Package1Path};
            return ResolveFile(candidates, static_cast<int>(NN_ARRAY_SIZE(candidates)));
        }
    case BootImageUpdateType::OdinMariko:
        {
            const char* candidates[] = {Package1PathA, Package1Path};
            return ResolveFile(candidates, static_cast<int>(NN_ARRAY_SIZE(candidates)));
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

const char* ResolvePackage2Path(BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    switch(bootImageUpdateType)
    {
    case BootImageUpdateType::Original:
        {
            const char* candidates[] = {Package2Path};
            return ResolveFile(candidates, static_cast<int>(NN_ARRAY_SIZE(candidates)));
        }
    case BootImageUpdateType::OdinMariko:
        {
            const char* candidates[] = {Package2PathA, Package2Path};
            return ResolveFile(candidates, static_cast<int>(NN_ARRAY_SIZE(candidates)));
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

bool NeedsUpdateSecureInfo(BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT
{
    switch(bootImageUpdateType)
    {
    case BootImageUpdateType::Original:
        return true;
    case BootImageUpdateType::OdinMariko:
        return false;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

}}
