﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_TransferMemory.h>

#include <nn/uart/driver/uart.h>
#include <nn/uart/driver/uart_PortApiDev.h> // OpenSessionForDev

#include <nn/uart/uart_IManager.sfdl.h>
#include <nn/uart/uart_IPortSession.sfdl.h>

#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/uart/server/uart_ManagerImpl.h>

namespace nn { namespace uart { namespace server {

// TransferMemory の Utility クラス
// Map 済みならデストラクタで自動的に Unmap する。
class UartTransferMemory : public nn::os::TransferMemory
{
public:
    explicit UartTransferMemory() NN_NOEXCEPT
        :  m_MappedAddress(nullptr)
    {}

    ~UartTransferMemory() NN_NOEXCEPT
    {
        if(m_MappedAddress != nullptr)
        {
            Unmap();
        }
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   既存の Transfer memory ハンドルを関連付け、同時に Map もします。
     *
     * @param[in]  ownerPermission Transfer memory を生成する際に満たすべきアクセス権
     * @param[in]  size            Transfer memory のサイズ
     * @param[in]  handle          Transfer memory のハンドル
     * @param[in]  managed         Transfer memory のハンドル管理フラグ
     *
     * @retresult
     *   @handleresult{nn::os::ResultInvalidHandle}
     *   @handleresult{nn::os::ResultInvalidTransferMemoryState}
     *   @handleresult{nn::os::ResultInvalidTransferMemorySize}
     *   @handleresult{nn::os::ResultOutOfAddressSpace}
     * @endretresult
     *
     * @details
     *  指定されたハンドルを使って Transfer memory オブジェクトを初期化します。@n
     *  詳細は @ref nn::os::AttachTransferMemory 、@ref nn::os::MapTransferMemory を参照して下さい。
     */
    nn::Result AttachAndMap(nn::os::MemoryPermission ownerPermission, size_t size, nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        void* mappedAddress;

        Attach(size, handle, managed);
        NN_RESULT_DO(Map(&mappedAddress, ownerPermission));
        m_MappedAddress = mappedAddress;

        return nn::ResultSuccess();
    }

    void* GetMappedAddress() const NN_NOEXCEPT
    {
        return m_MappedAddress;
    }

private:
    void* m_MappedAddress;
};


// IPortSession 実装クラス
class PortSessionImpl
{
public:
    explicit PortSessionImpl(ManagerImpl* pParent) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {}

    // Session のデストラクタで Session のクローズを行う
    ~PortSessionImpl()
    {
        nn::uart::driver::ClosePort(&m_InternalPortSession);

        // UartTransferMemory のデストラクタで Unmap も行われる
    }

    nn::Result OpenPort(nn::sf::Out<bool> isSuccess, std::int32_t name, std::int32_t baudRate, std::int32_t flowControlMode,
                        nn::sf::NativeHandle&& sendHandle, std::uint64_t sendLength,
                        nn::sf::NativeHandle&& receiveHandle, std::uint64_t receiveLength,
                        bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT;
    nn::Result OpenPortForDev(nn::sf::Out<bool> isSuccess, std::int32_t portIndex,std::int32_t baudRate, std::int32_t flowControlMode,
                        nn::sf::NativeHandle&& sendHandle, std::uint64_t sendLength,
                        nn::sf::NativeHandle&& receiveHandle, std::uint64_t receiveLength,
                        bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT;
    nn::Result GetWritableLength(nn::sf::Out<std::int64_t> size) NN_NOEXCEPT;
    nn::Result Send(nn::sf::Out<std::int64_t> doneBytes, const nn::sf::InBuffer& inData) NN_NOEXCEPT;
    nn::Result GetReadableLength(nn::sf::Out<std::int64_t> size) NN_NOEXCEPT;
    nn::Result Receive(nn::sf::Out<std::int64_t> doneBytes, const nn::sf::OutBuffer& outData) NN_NOEXCEPT;
    nn::Result BindPortEvent(nn::sf::Out<bool> result, nn::sf::Out<nn::sf::NativeHandle> handle, std::int32_t eventType, std::int64_t threshold) NN_NOEXCEPT;
    nn::Result UnbindPortEvent(nn::sf::Out<bool> result, std::int32_t eventType) NN_NOEXCEPT;

private:

    nn::Result OpenPortImpl(nn::sf::Out<bool> isSuccess, bool isForDev, std::int32_t number, std::int32_t baudRate, std::int32_t flowControlMode,
                            nn::sf::NativeHandle&& sendHandle, std::uint64_t sendLength,
                            nn::sf::NativeHandle&& receiveHandle, std::uint64_t receiveLength,
                            bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT;

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl> m_Parent;

    // UART ドライバライブラリが持つ PortSession 構造体
    nn::uart::driver::PortSession m_InternalPortSession;

    // Session に紐付けられる SystemEvent 4 種類
    nn::os::SystemEvent m_SendBufferEmptyEvent;
    nn::os::SystemEvent m_SendBufferReadyEvent;
    nn::os::SystemEvent m_ReceiveBufferReadyEvent;
    nn::os::SystemEvent m_ReceiveEndEvent;

    // UartTransferMemoryObject
    UartTransferMemory m_SendTransferMemory;
    UartTransferMemory m_ReceiveTransferMemory;
};



// ここから各 UART API の内部実装
nn::Result PortSessionImpl::OpenPort(nn::sf::Out<bool> isSuccess, std::int32_t name, std::int32_t baudRate, std::int32_t flowControlMode,
                                 nn::sf::NativeHandle&& sendHandle, std::uint64_t sendLength,
                                 nn::sf::NativeHandle&& receiveHandle, std::uint64_t receiveLength,
                                 bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT
{
    return OpenPortImpl(isSuccess, false, name, baudRate, flowControlMode, std::move(sendHandle), sendLength, std::move(receiveHandle), receiveLength,
                        isInvertTx, isInvertRx, isInvertRts, isInvertCts);
}

nn::Result PortSessionImpl::OpenPortForDev(nn::sf::Out<bool> isSuccess, std::int32_t portIndex,std::int32_t baudRate, std::int32_t flowControlMode,
                        nn::sf::NativeHandle&& sendHandle, std::uint64_t sendLength,
                        nn::sf::NativeHandle&& receiveHandle, std::uint64_t receiveLength,
                        bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT
{
    return OpenPortImpl(isSuccess, true, portIndex, baudRate, flowControlMode, std::move(sendHandle), sendLength, std::move(receiveHandle), receiveLength,
                        isInvertTx, isInvertRx, isInvertRts, isInvertCts);
}


// OpenPort と OpenPortForDev で共通で使われる処理
nn::Result PortSessionImpl::OpenPortImpl(nn::sf::Out<bool> isSuccess, bool isForDev, std::int32_t number, std::int32_t baudRate, std::int32_t flowControlMode,
                                       nn::sf::NativeHandle&& sendHandle, std::uint64_t sendLength,
                                       nn::sf::NativeHandle&& receiveHandle, std::uint64_t receiveLength,
                                       bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT
{
    // PortConfigType
    nn::uart::PortConfigType portConfigType;

    NN_SDK_ASSERT(sendLength < 0x100000000ull);
    NN_SDK_ASSERT(receiveLength < 0x100000000ull);

    // TransferMemory の Attach と Map
    if(sendLength > 0)
    {
        NN_RESULT_DO(m_SendTransferMemory.AttachAndMap(nn::os::MemoryPermission_ReadWrite,
            static_cast<size_t>(sendLength), sendHandle.GetOsHandle(), sendHandle.IsManaged()));
        sendHandle.Detach();
    }

    if(receiveLength > 0)
    {
        NN_RESULT_DO(m_ReceiveTransferMemory.AttachAndMap(nn::os::MemoryPermission_ReadWrite,
            static_cast<size_t>(receiveLength), receiveHandle.GetOsHandle(), receiveHandle.IsManaged()));
        receiveHandle.Detach();
    }

    // OpenPort() に渡すためにバラバラにしていた設定周りを再度 PortConfig に設定する
    nn::uart::driver::InitializePortConfig(&portConfigType, static_cast<nn::uart::BaudRate>(baudRate),
                                           static_cast<char*>(m_SendTransferMemory.GetMappedAddress()), static_cast<size_t>(sendLength),
                                           static_cast<char*>(m_ReceiveTransferMemory.GetMappedAddress()), static_cast<size_t>(receiveLength));

    nn::uart::driver::SetPortConfigFlowControlMode(&portConfigType, static_cast<nn::uart::FlowControlMode>(flowControlMode));
    nn::uart::driver::SetPortConfigInvertPins(&portConfigType, isInvertTx, isInvertRx, isInvertRts, isInvertCts);

    bool result;

    if(isForDev)
    {
        result = nn::uart::driver::OpenPortForDev(&m_InternalPortSession, static_cast<int>(number), portConfigType);
    }
    else
    {
        result = nn::uart::driver::OpenPort(&m_InternalPortSession, static_cast<nn::uart::PortName>(number), portConfigType);
    }

    *isSuccess = result;

    return nn::ResultSuccess();
}

nn::Result PortSessionImpl::GetWritableLength(nn::sf::Out<std::int64_t> size) NN_NOEXCEPT
{
    *size = static_cast<int64_t>(nn::uart::driver::GetWritableLength(&m_InternalPortSession));
    return nn::ResultSuccess();
}

nn::Result PortSessionImpl::Send(nn::sf::Out<std::int64_t> doneBytes, const nn::sf::InBuffer& inData) NN_NOEXCEPT
{
    size_t bytes;

    nn::uart::driver::Send(&bytes, &m_InternalPortSession, inData.GetPointerUnsafe(), inData.GetSize());

    *doneBytes = static_cast<int64_t>(bytes);
    return nn::ResultSuccess();
}

nn::Result PortSessionImpl::GetReadableLength(nn::sf::Out<std::int64_t> size) NN_NOEXCEPT
{
    *size = static_cast<int64_t>(nn::uart::driver::GetReadableLength(&m_InternalPortSession));
    return nn::ResultSuccess();
}

nn::Result PortSessionImpl::Receive(nn::sf::Out<std::int64_t> doneBytes, const nn::sf::OutBuffer& outData) NN_NOEXCEPT
{
    size_t bytes;

    auto result = nn::uart::driver::Receive(&bytes, outData.GetPointerUnsafe(), outData.GetSize(), &m_InternalPortSession);

    *doneBytes = static_cast<int64_t>(bytes);
    return result;
}

nn::Result PortSessionImpl::BindPortEvent(nn::sf::Out<bool> result, nn::sf::Out<nn::sf::NativeHandle> handle, std::int32_t eventType, std::int64_t threshold) NN_NOEXCEPT
{
    nn::os::SystemEventType* internalEventType;

    // PortEventType ごとに内部のイベントを選択する
    switch(static_cast<nn::uart::PortEventType>(eventType))
    {
    case nn::uart::PortEventType_SendBufferEmpty:
        internalEventType = m_SendBufferEmptyEvent.GetBase();
        break;

    case nn::uart::PortEventType_SendBufferReady:
        internalEventType = m_SendBufferReadyEvent.GetBase();
        break;

    case nn::uart::PortEventType_ReceiveBufferReady:
        internalEventType = m_ReceiveBufferReadyEvent.GetBase();
        break;

    case nn::uart::PortEventType_ReceiveEnd:
        internalEventType = m_ReceiveEndEvent.GetBase();
        break;

    default:NN_UNEXPECTED_DEFAULT;

    }

    // session のメンバ変数である m_SystemEvent を BindInterrupt で UART ドライバライブラリ内部のポートに紐付け、
    // その SystemEvent を外に渡す
    bool isSuccess = nn::uart::driver::BindPortEvent(internalEventType,&m_InternalPortSession,
                                                     static_cast<nn::uart::PortEventType>(eventType),
                                                     static_cast<size_t>(threshold));
    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(internalEventType);

    // 内部の IsManaged を false にしているので、寿命管理はされない。
    *handle = nn::sf::NativeHandle( internalHandle, false );

    *result = isSuccess;

    return nn::ResultSuccess();
}

nn::Result PortSessionImpl::UnbindPortEvent(nn::sf::Out<bool> result, std::int32_t eventType) NN_NOEXCEPT
{
    nn::os::SystemEventType* internalEventType;

    // クライアントから渡された eventTyepe で Unbind する Event を選択する
    switch(static_cast<nn::uart::PortEventType>(eventType))
    {
    case nn::uart::PortEventType_SendBufferEmpty:
        internalEventType = m_SendBufferEmptyEvent.GetBase();
        break;

    case nn::uart::PortEventType_SendBufferReady:
        internalEventType = m_SendBufferReadyEvent.GetBase();
        break;

    case nn::uart::PortEventType_ReceiveBufferReady:
        internalEventType = m_ReceiveBufferReadyEvent.GetBase();
        break;

    case nn::uart::PortEventType_ReceiveEnd:
        internalEventType = m_ReceiveEndEvent.GetBase();
        break;

    default:NN_UNEXPECTED_DEFAULT;

    }

    *result = nn::uart::driver::UnbindPortEvent(internalEventType, &m_InternalPortSession);

    // TORIAEZU : UART は ドライバ内部で event を Destroy しないため、別途ここで Destroy しないと
    // Bind -> Unbind を複数回繰り返せないため、Destroy しておく
    nn::os::DestroySystemEvent(internalEventType);


    return nn::ResultSuccess();
}


// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::HasPort(nn::sf::Out<bool> hasPort, std::int32_t name) NN_NOEXCEPT
{
    *hasPort = nn::uart::driver::HasPort(static_cast<nn::uart::PortName>(name));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::HasPortForDev(nn::sf::Out<bool> hasPort, std::int32_t portIndex) NN_NOEXCEPT
{
    *hasPort = nn::uart::driver::HasPortForDev(static_cast<int>(portIndex));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::IsSupportedBaudRate(nn::sf::Out<bool> isSupported, std::int32_t name, std::int32_t baudRate) NN_NOEXCEPT
{
    *isSupported = nn::uart::driver::IsSupportedBaudRate(static_cast<nn::uart::PortName>(name),
                                                         static_cast<nn::uart::BaudRate>(baudRate));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::IsSupportedBaudRateForDev(nn::sf::Out<bool> isSupported, std::int32_t portIndex, std::int32_t baudRate) NN_NOEXCEPT
{
    *isSupported = nn::uart::driver::IsSupportedBaudRateForDev(static_cast<int>(portIndex),
                                                               static_cast<nn::uart::BaudRate>(baudRate));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::IsSupportedFlowControlMode(nn::sf::Out<bool> isSupported, std::int32_t name, std::int32_t flowControlMode) NN_NOEXCEPT
{
    *isSupported = nn::uart::driver::IsSupportedFlowControlMode(static_cast<nn::uart::PortName>(name),
                                                                static_cast<nn::uart::FlowControlMode>(flowControlMode));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::IsSupportedFlowControlModeForDev(nn::sf::Out<bool> isSupported, std::int32_t portIndex, std::int32_t flowControlMode) NN_NOEXCEPT
{
    *isSupported = nn::uart::driver::IsSupportedFlowControlModeForDev(static_cast<int>(portIndex),
                                                                static_cast<nn::uart::FlowControlMode>(flowControlMode));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::IsSupportedPortEvent(nn::sf::Out<bool> isSupported, std::int32_t name, std::int32_t portEvent) NN_NOEXCEPT
{
    *isSupported = nn::uart::driver::IsSupportedPortEvent(static_cast<nn::uart::PortName>(name),
                                                          static_cast<nn::uart::PortEventType>(portEvent));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::IsSupportedPortEventForDev(nn::sf::Out<bool> isSupported, std::int32_t portIndex, std::int32_t portEvent) NN_NOEXCEPT
{
    *isSupported = nn::uart::driver::IsSupportedPortEventForDev(static_cast<int>(portIndex),
                                                                static_cast<nn::uart::PortEventType>(portEvent));
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::CreatePortSession(nn::sf::Out<nn::sf::SharedPointer<nn::uart::IPortSession>> outSession) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<IPortSession, PortSessionImpl>(&m_Allocator, this);

    NN_ABORT_UNLESS_NOT_NULL(p);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}}
