﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバの API インタフェース部分。
 */

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/os/os_SystemEvent.h>

#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/uart_Result.h>
#include <nn/uart/driver/uart_Lib.h>
#include <nn/uart/driver/uart_PortApi.h>
#include <nn/uart/driver/uart_Suspend.h>
#include "detail/uart_Driver.h"
#include "detail/uart_PortSessionImpl.h"
#include "detail/uart_TargetSpec.h"
#include "detail/uart_Command.h"

namespace {

#define NN_DETAIL_UART_SIZEOF_MEMBER(type, member)  sizeof((reinterpret_cast<type *>(0))->member)
NN_STATIC_ASSERT(offsetof(nn::uart::driver::PortSession, _impl) == 0);
NN_STATIC_ASSERT(offsetof(nn::uart::driver::detail::PortSessionImplPadded, impl) == 0);
NN_STATIC_ASSERT(sizeof(nn::uart::driver::detail::PortSessionImplPadded) ==
                 NN_DETAIL_UART_SIZEOF_MEMBER(nn::uart::driver::PortSession, _impl));
NN_STATIC_ASSERT(NN_ALIGNOF(nn::uart::driver::detail::PortSessionImpl) == NN_ALIGNOF(nn::uart::driver::PortSession));
NN_STATIC_ASSERT(nn::uart::driver::PortSessionSize - sizeof(nn::uart::driver::detail::PortSessionImpl) > 0);

void AssertLibraryInitialized()
{
    NN_SDK_REQUIRES(nn::uart::driver::detail::Driver::GetInstance().IsInitialized(),
        "UART Error: Library is not initialized\n");
}

void Check(const nn::uart::driver::detail::PortSessionImpl& sessionImpl) NN_NOEXCEPT
{
    // セッションが Open 状態であることを確認
    NN_SDK_REQUIRES(sessionImpl.IsOpen(), "UART Error: Port session is not open.\n");

    // ポートセッションがセッションホルダーに登録されていることを確認
    // OpenPort() でオープンせず、セッション構造体内のオープンフラグだけ別の要因で
    // 立ってしまっているような場合をチェック
    NN_SDK_REQUIRES(nn::uart::driver::detail::Driver::GetInstance().VerifyOpenPortSession(sessionImpl),
                    "UART Error: Port session looks open but not registered to the library.\n");
}

nn::uart::driver::detail::PortSessionImpl& GetOpenSessionImpl(nn::uart::driver::PortSession& session) NN_NOEXCEPT
{
    auto&& sessionImpl = nn::uart::driver::detail::ToPortSessionImpl(session);
    Check(sessionImpl);
    return sessionImpl;
}

const nn::uart::driver::detail::PortSessionImpl& GetOpenSessionImpl(const nn::uart::driver::PortSession& session) NN_NOEXCEPT
{
    auto&& sessionImpl = nn::uart::driver::detail::ToPortSessionImpl(session);
    Check(sessionImpl);
    return sessionImpl;
}

}

namespace nn {
namespace uart {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    nn::uart::driver::detail::Driver::GetInstance().Initialize();
}

bool IsInitialized() NN_NOEXCEPT
{
    return nn::uart::driver::detail::Driver::GetInstance().IsInitialized();
}

void Finalize() NN_NOEXCEPT
{
    nn::uart::driver::detail::Driver::GetInstance().Finalize();
}

bool HasPort(PortName name) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::Driver::GetInstance().HasPort(
            nn::uart::driver::detail::ConvertPortNameToPortIndex(name));
}

bool HasPortForDev(int portIndex) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::Driver::GetInstance().HasPort(portIndex);
}

bool IsSupportedBaudRate(PortName name, BaudRate baudRate) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::IsSupportedBaudRateImpl(
            nn::uart::driver::detail::ConvertPortNameToPortIndex(name),
            baudRate);
}

bool IsSupportedBaudRateForDev(int portIndex, BaudRate baudRate) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::IsSupportedBaudRateImpl(
            portIndex,
            baudRate);
}

bool IsSupportedFlowControlMode(PortName name, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::IsSupportedFlowControlModeImpl(
            nn::uart::driver::detail::ConvertPortNameToPortIndex(name),
            flowControlMode);
}

bool IsSupportedFlowControlModeForDev(int portIndex, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::IsSupportedFlowControlModeImpl(
            portIndex,
            flowControlMode);
}

bool IsSupportedPortEvent(PortName name, PortEventType portEvent) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::IsSupportedPortEventImpl(
            nn::uart::driver::detail::ConvertPortNameToPortIndex(name),
            portEvent);
}

bool IsSupportedPortEventForDev(int portIndex, PortEventType portEvent) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return nn::uart::driver::detail::IsSupportedPortEventImpl(
            portIndex,
            portEvent);
}

void InitializePortConfig(PortConfigType* pOutPortConfig, BaudRate baudRate,
        char* sendBuffer, size_t sendBufferLength,
        char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPortConfig);
    NN_SDK_REQUIRES(sendBuffer || receiveBuffer, "Either sendBuffer or receiveBuffer must not be null.\n");

    pOutPortConfig->_baudRate            = baudRate;
    pOutPortConfig->_flowControlMode     = FlowControlMode_None; // Default
    pOutPortConfig->_sendBuffer          = sendBuffer;
    pOutPortConfig->_sendBufferLength    = sendBuffer ? sendBufferLength : 0;
    pOutPortConfig->_receiveBuffer       = receiveBuffer;
    pOutPortConfig->_receiveBufferLength = receiveBuffer ? receiveBufferLength : 0;
    pOutPortConfig->_isInvertTx          = false; // Default
    pOutPortConfig->_isInvertRx          = false; // Default
    pOutPortConfig->_isInvertRts         = false; // Default
    pOutPortConfig->_isInvertCts         = false; // Default
}

void SetPortConfigFlowControlMode(PortConfigType* pPortConfig, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPortConfig);
    pPortConfig->_flowControlMode = flowControlMode;
}

void SetPortConfigInvertPins(PortConfigType* pPortConfig, bool isInvertTx, bool isInvertRx, bool isInvertRts, bool isInvertCts) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPortConfig);
    pPortConfig->_isInvertTx = isInvertTx;
    pPortConfig->_isInvertRx = isInvertRx;
    pPortConfig->_isInvertRts = isInvertRts;
    pPortConfig->_isInvertCts = isInvertCts;
}

bool OpenPort(PortSession* pOutSession, PortName name, const PortConfigType& portConfig) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_REQUIRES(nn::uart::driver::detail::Driver::GetInstance().HasPort(
            nn::uart::driver::detail::ConvertPortNameToPortIndex(name)));

    return nn::uart::driver::detail::Driver::GetInstance().OpenPortSession(
            pOutSession,
            nn::uart::driver::detail::ConvertPortNameToPortIndex(name),
            portConfig);
}

bool OpenPortForDev(PortSession* pOutSession, int portIndex, const PortConfigType& portConfig) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_REQUIRES(nn::uart::driver::detail::Driver::GetInstance().HasPort(portIndex));

    return nn::uart::driver::detail::Driver::GetInstance().OpenPortSession(
            pOutSession,
            portIndex,
            portConfig);
}

size_t GetWritableLength(const PortSession* pSession) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    return GetOpenSessionImpl(*pSession).GetWritableLength();
}

void Send(size_t* pOutDoneBytes, PortSession* pSession, const void* data, size_t dataBytes) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    NN_SDK_REQUIRES_NOT_NULL(data);

    auto& sessionImpl = GetOpenSessionImpl(*pSession);

    nn::uart::driver::detail::CommandSend command;
    command.Setup(static_cast<const char*>(data), dataBytes);
    sessionImpl.Send(command);

    if (pOutDoneBytes)
    {
        *pOutDoneBytes = command.GetParam().out.doneBytes;
    }
}

size_t GetReadableLength(const PortSession* pSession) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    return GetOpenSessionImpl(*pSession).GetReadableLength();
}

nn::Result Receive(size_t* pOutDoneBytes, void* outData, size_t dataBytes, PortSession* pSession) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    NN_SDK_REQUIRES_NOT_NULL(outData);

    auto& sessionImpl = GetOpenSessionImpl(*pSession);

    nn::uart::driver::detail::CommandReceive command;
    command.Setup(static_cast<char*>(outData), dataBytes);
    sessionImpl.Receive(command);

    if (pOutDoneBytes)
    {
        *pOutDoneBytes = command.GetParam().out.doneBytes;
    }
    return command.GetParam().out.result;
}

bool BindPortEvent(nn::os::SystemEventType* pEvent, PortSession* pSession, PortEventType eventType, size_t threshold) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    NN_SDK_REQUIRES_NOT_NULL(pSession);

    // 有効な eventType かどうかは BindEvent 内でチェックする
    auto& sessionImpl = nn::uart::driver::detail::ToPortSessionImpl(*pSession);
    return sessionImpl.BindEvent(pEvent, eventType, threshold);
}

bool UnbindPortEvent(nn::os::SystemEventType* pEvent, PortSession* pSession) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    NN_SDK_REQUIRES_NOT_NULL(pSession);

    auto& sessionImpl = nn::uart::driver::detail::ToPortSessionImpl(*pSession);
    return sessionImpl.UnbindEvent(pEvent);
}

void ClosePort(PortSession* pSession) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pSession);

    // 多重クローズを許容するため、オープン状態かどうかはチェックしない
    // auto& sessionImpl = GetOpenSessionImpl(*pSession);
    auto& sessionImpl = nn::uart::driver::detail::ToPortSessionImpl(*pSession);
    nn::uart::driver::detail::Driver::GetInstance().ClosePortSession(sessionImpl);
}

void SuspendAllPorts() NN_NOEXCEPT
{
    nn::uart::driver::detail::Driver::GetInstance().SuspendOpenPorts();
}

void ResumeAllPorts() NN_NOEXCEPT
{
    nn::uart::driver::detail::Driver::GetInstance().ResumeOpenPorts();
}

} // driver
} // uart
} // nn
