﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/uart/detail/uart_Log.h>

#include "uart_Driver.h"
#include "uart_PortSessionImpl.h"
#include "uart_Interrupt.h"
#include "uart_TargetSpec.h"

#ifdef NN_DETAIL_UART_ENABLE_DMA
#include <nne/dma/dma.h>
#endif

namespace nn {
namespace uart {
namespace driver {
namespace detail {

void Driver::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    ++m_LibInitCount;
    if (m_LibInitCount > 1)
    {
        return;
    }
    NN_SDK_ASSERT_EQUAL(m_LibInitCount, 1);

    m_IsSuspended = false;

    // 割り込みハンドラ用のスレッドを用意し、割り込みイベントを登録
    auto& interruptNotifier = InterruptNotifier::GetInstance();
    interruptNotifier.Initialize();
    interruptNotifier.StartThread();

    for (PortSessionHolder& sessionHolder : m_PortSessionHolder)
    {
        sessionHolder.Reset();
    }
}

void Driver::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    NN_SDK_REQUIRES(m_LibInitCount > 0);
    --m_LibInitCount;
    if (m_LibInitCount > 0)
    {
        return;
    }
    NN_SDK_ASSERT_EQUAL(m_LibInitCount, 0);

    // 以下、Initialize と逆順に終了していく

    for (PortSessionHolder& sessionHolder : m_PortSessionHolder)
    {
        if (sessionHolder.IsPortSessionRegistered())
        {
            auto pSessionImpl = sessionHolder.GetPortSessionImpl();
            NN_DETAIL_UART_WARN("UART Warning: Port session %p was still open upon shutdown.\n", pSessionImpl);

            // 閉じられてないポートがいながらシャットダウンされようとしている
            // リリースビルドではシャットダウンを止めたくないので、
            // 強制的に全セッションを無効化して先へ進む
            // 中で interruptNotifier への SetHandler が呼ばれる可能性がある
            ClosePortSessionBody(*pSessionImpl);
        }
    }

    // ClosePortSessionBody() で interruptNotifier への SetHandler が呼ばれる可能性があるので、そのあと終わらせる
    auto& interruptNotifier = InterruptNotifier::GetInstance();
    interruptNotifier.StopThread();
    interruptNotifier.Finalize();
}

void Driver::SuspendOpenPorts() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    NN_SDK_ASSERT(IsInitialized());

    if (!m_IsSuspended)
    {
        // NN_DETAIL_UART_INFO("Suspending\n");
        m_IsSuspended = true;

        // 割り込み処理スレッドを止める
        auto& interruptNotifier = InterruptNotifier::GetInstance();
        interruptNotifier.StopThreadForSuspending();

        for (PortSessionHolder& sessionHolder : m_PortSessionHolder)
        {
            if (sessionHolder.IsPortSessionRegistered())
            {
                auto pSessionImpl = sessionHolder.GetPortSessionImpl();
                pSessionImpl->Suspend();
                m_IsOpenPortExistBeforeSleeping = true;
            }
        }

        // 1つでも Open しているポートがあった場合、強制的に DMA の Finalize を呼ぶ。
        if (m_IsOpenPortExistBeforeSleeping)
        {
#ifdef NN_DETAIL_UART_ENABLE_DMA
            nne::dma::tegra::Finalize();
#endif
        }

        // 最後にクロックとパワーを止める
        for (PortSessionHolder& sessionHolder : m_PortSessionHolder)
        {
            if (sessionHolder.IsPortSessionRegistered())
            {
                auto pSessionImpl = sessionHolder.GetPortSessionImpl();
                pSessionImpl->DisablePowerAndClock();
            }
        }
    }
}

void Driver::ResumeOpenPorts() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    NN_SDK_ASSERT(IsInitialized());

    if (m_IsSuspended)
    {
        // 1つでも Open しているポートがあった場合、強制的に DMA の Initialize を呼ぶ。
        if (m_IsOpenPortExistBeforeSleeping)
        {
#ifdef NN_DETAIL_UART_ENABLE_DMA
            nne::dma::tegra::Initialize();
            m_IsOpenPortExistBeforeSleeping = false;
#endif
        }

        // NN_DETAIL_UART_INFO("Resuming\n");
        for (PortSessionHolder& sessionHolder : m_PortSessionHolder)
        {
            if (sessionHolder.IsPortSessionRegistered())
            {
                auto pSessionImpl = sessionHolder.GetPortSessionImpl();
                pSessionImpl->Resume();
            }
        }

        // 割り込み処理スレッドを走らせる
        auto& interruptNotifier = InterruptNotifier::GetInstance();
        interruptNotifier.StartThreadForResuming();

        m_IsSuspended = false;
    }
}

bool Driver::OpenPortSession(PortSession* pOutSession, int portIndex, const PortConfigType& portConfig) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    NN_SDK_ASSERT(pOutSession);
    NN_SDK_ASSERT(IsInitialized());
    NN_SDK_ASSERT(!IsSuspended());

    PortSessionImpl& session = ToPortSessionImpl(*pOutSession);

    // Open 時、未初期化の PortSession を渡してくるのでこれを期待してはいけない
    // NN_SDK_REQUIRES(!session.IsOpen(), "UART Error: Port session is already open.\n");

    // 渡されたバッファへのポインタが、すでにセッションホルダーに登録されていないか確認
    NN_SDK_REQUIRES(!IsPortSessionRegistered(&session),
                    "UART Error: Port session looks closed but already registered to the library.\n");

    // 渡された領域に対して placement new で PortSessionImpl として初期化する
    new (&session) PortSessionImpl(portIndex, portConfig);

    // 違うセッションオブジェクトがすでにターゲットのポートに登録されていた場合に false
    auto& sessionHolder = GetPortSessionHolder(session.GetPortIndex());
    if (!sessionHolder.RegisterPortSession(&session))
    {
        // 初期化はキャンセルする
        session.~PortSessionImpl();
        return false;
    }

    session.Open();

    return true;
}

void Driver::ClosePortSession(PortSessionImpl& session) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    NN_SDK_ASSERT(IsInitialized());
    NN_SDK_ASSERT(!IsSuspended());

    if (!session.IsOpen() || !VerifyOpenPortSession(session))
    {
        NN_DETAIL_UART_WARN("UART Warning: ClosePort(): Session %p is already closed.\n", &session);
        return;
    }

    // Finalize と処理共有するため別関数へ
    ClosePortSessionBody(session);
}

void Driver::ClosePortSessionBody(PortSessionImpl& session) NN_NOEXCEPT
{
    // 以下、OpenPortSession() と逆順に処理する

    session.Close();

    // セッション登録を解除
    auto& sessionHolder = GetPortSessionHolder(session.GetPortIndex());
    sessionHolder.UnregisterPortSession(&session);

    // Open で placement new しているので、明示的にデストラクタを呼ぶ
    session.~PortSessionImpl();
}

bool Driver::IsPortSessionRegistered(const PortSessionImpl* pSession) const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(m_DriverMutex);

    NN_SDK_ASSERT(pSession);
    NN_SDK_ASSERT(IsInitialized());

    // 指定したセッションオブジェクトがすでにいずれかのセッションホルダーに登録されているか
    for (const PortSessionHolder& sessionHolder : m_PortSessionHolder)
    {
        if (sessionHolder.VerifyPortSession(pSession))
        {
            return true;
        }
    }
    return false;
}
bool Driver::VerifyOpenPortSession(const PortSessionImpl& session) const NN_NOEXCEPT
{
    auto& sessionHolder = GetPortSessionHolder(session.GetPortIndex());
    return sessionHolder.VerifyPortSession(&session);
}

} // detail
} // driver
} // uart
} // nn
