﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   キャラクタベースの汎用循環バッファ。入力・出力時とも登録されたバッファへのmemcpyを伴う。
 *          コードは cafe からの再利用。
 */

#pragma once

#include <cstring>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

namespace nn {
namespace uart {
namespace driver {
namespace detail {

/**
 * @brief   Character-based circular buffer.
 *          The buffer can hold specified buffer size - 1 bytes.
 */
class CircularBuffer
{
public:
    CircularBuffer() NN_NOEXCEPT :
        m_IsInitialized(false),
        m_IsEmpty(true),
        m_WritePos(0),
        m_ReadPos(0),
        m_BufferLength(0),
        m_Buffer(nullptr) {}

    /**
     * @brief   Initialize circular buffer with specified buffer pointer
     * @pre     buffer must not be null..
     */
    void Initialize(char* buffer, size_t bufferLength) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsInitialized);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_REQUIRES(bufferLength > 1);
        m_Buffer = buffer;
        m_BufferLength = bufferLength;
        m_ReadPos = 0;
        m_WritePos = 0;
        m_IsInitialized = true;
    }

    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);
        m_IsInitialized = false;
        m_Buffer = nullptr;
        m_BufferLength = 0;
        m_ReadPos = 0;
        m_WritePos = 0;
    }

    /**
     * @brief       Sets up the circular buffer so there are no bytes available for reading
     * @param[in]   clearMem    Cleanse whole buffer with 0.
     */
    void Empty(bool clearMem) NN_NOEXCEPT
    {
        m_WritePos = 0;
        m_ReadPos = 0;
        m_IsEmpty = true;
        if (clearMem)
        {
            std::memset(m_Buffer, 0x00, m_BufferLength);
        }
    }

    /**
     * @brief Write data into circular buffer.  If there is no space left in the buffer,
     *        some old data will be thrown away
     * @param[out]  pWasOverwritten Set true if old data was overwritten
     * @param[in]   data            Pointer to start of data that should be copied into buffer
     * @param[in]   count           Number of bytes to copy from 'data' into buffer
     * @return      Number of bytes written, which will always be 'count'
     */
    size_t Overwrite(bool *pWasOverwritten, const char* data, size_t count) NN_NOEXCEPT;

    /**
     * @brief Write data into circular buffer.
     * @param[in]   data    Pointer to start of data that should be copied into buffer
     * @param[in]   count   Number of bytes to copy from 'data' into buffer
     * @return      Number of bytes actually copied into buffer.  If there are less
     *              than 'count' bytes available, this number will be returned.  If
     *              there are no bytes available, 0 will be returned.
     */
    size_t Write(const char* data, size_t count) NN_NOEXCEPT;

    /**
     * @brief Read data out of a circular buffer, but don't advance the internal pointer.
     * @param[in]   position    This is an integer that keeps track of the current read pointer.
     *                          The value will be read out of it before the copy and the new offset will
     *                          be written back into it after the copy.
     * @param[out]  data        Buffer to copy data into
     * @param[in]   count       Maximum number of bytes that can be copied into data
     * @return      Number of bytes read, up to count.  If fewer than count bytes are
     *              available for reading, this will return how many bytes were actually
     *              read.  If there are no bytes available for reading, this will return 0
     */
    size_t ReadAt(size_t *position, char* data, size_t count) NN_NOEXCEPT;

    /**
     * @brief Read data out of a circular buffer.  This will advance the read pointer
     *        (m_ReadPos) the buffer so subsequent reads will read the next bytes of data.
     * @param[out]  data    Buffer to copy data into
     * @param[in]   count   Maximum number of bytes that can be copied into data
     * @return      Number of bytes read, up to count.  If fewer than count bytes are
     *              available for reading, this will return how many bytes were actually
     *              read.  If there are no bytes available for reading, this will return 0
     */
    size_t Read(char* data, size_t count) NN_NOEXCEPT
    {
        size_t readCount = ReadAt(&m_ReadPos, data, count);
        if (readCount > 0 && m_ReadPos == m_WritePos)
        {
            m_IsEmpty = true;
        }
        return readCount;
    }

    bool IsWritableSpaceExist(size_t size)
    {
        const size_t writableLength = GetWritableLength();

        return (size <= writableLength);
    }

    /**
     * @brief Returns if the circular buffer is empty
     */
    NN_FORCEINLINE
    bool IsEmpty() const NN_NOEXCEPT
    {
        return m_IsEmpty;
    }

    /**
     * @brief Returns if the circular buffer is full
     */
    NN_FORCEINLINE
    bool IsFull() const NN_NOEXCEPT
    {
        return (!m_IsEmpty && m_ReadPos == m_WritePos);
    }

    /**
     * @brief Returns the maximum number of bytes that can be read out.
     */
    NN_FORCEINLINE
    size_t GetReadableLength() const NN_NOEXCEPT
    {
        if (IsFull())
        {
            return m_BufferLength;
        }
        return wrapPosition(m_WritePos + m_BufferLength - m_ReadPos);
    }

    /**
     * @brief Returns the maximum number of bytes that can be written into the buffer.
     *        When buffer is empty, this returns (m_BufferLength).
     */
    NN_FORCEINLINE
    size_t GetWritableLength() const NN_NOEXCEPT
    {
        if (IsEmpty())
        {
            return m_BufferLength;
        }
        return wrapPosition(m_ReadPos + m_BufferLength - m_WritePos);
    }

    NN_FORCEINLINE
    size_t GetBufferLength() const NN_NOEXCEPT
    {
        return m_BufferLength;
    }

private:
    NN_FORCEINLINE
    size_t wrapPosition(size_t position) const NN_NOEXCEPT
    {
        return position % m_BufferLength;
    }

private:
    bool    m_IsInitialized;
    bool    m_IsEmpty;        // Indicates whether the buffer is empty or full when m_ReadPos == m_WritePos
    size_t  m_WritePos;
    size_t  m_ReadPos;
    size_t  m_BufferLength;
    char*   m_Buffer;
};


} // detail
} // driver
} // uart
} // nn
